package com.ejianc.foundation.operatelog.service.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.admin.indices.alias.Alias;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.indices.*;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.foundation.operatelog.service.IOperateLogService;
import com.ejianc.foundation.operatelog.vo.OperateLogVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;

@Service
public class OperateLogServiceImpl implements IOperateLogService {
	private Logger logger = LoggerFactory.getLogger(this.getClass());
	private final static Integer QUERY_TIMEOUT = 60;
	private final static String ASC = "asc";
	private final static String INDEX_NAME = "ejclogspace";
	
	@Autowired(required=false)
	private RestHighLevelClient client;

	@Override
	public IPage<OperateLogVO> queryPageList(QueryParam queryParam) {
		IPage<OperateLogVO> page = new Page<>();

		SearchRequest searchRequest = new SearchRequest(INDEX_NAME);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			if(param.getValue() == null || StringUtils.isBlank(param.getValue() + "")) {
				continue;
			}
			if(QueryParam.EQ.equals(param.getType())) {
				boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString()));
			} else if(QueryParam.LIKE.equals(param.getType())) {
				boolQuery.must(QueryBuilders.matchQuery(entry.getKey(), param.getValue().toString()));
			} else if(QueryParam.IN.equals(param.getType())) {
				boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString().split(",")));
			} else if(QueryParam.BETWEEN.equals(param.getType())) {
				String[] dataArr = param.getValue().toString().split(",");
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true));
			} else if(QueryParam.GT.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue().toString()));
			} else if(QueryParam.GE.equals(param.getType())){
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue().toString()));
			} else if(QueryParam.LT.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue().toString()));
			} else if(QueryParam.LE.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue().toString()));
			}
		}
		sourceBuilder.query(boolQuery);
		
		//排序
		Map<String, String> orderMap = queryParam.getOrderMap();
		if(orderMap != null && !orderMap.isEmpty()) {
			GetIndexRequest request = new GetIndexRequest(INDEX_NAME);
			try {
				GetIndexResponse response = client.indices().get(request, RequestOptions.DEFAULT);
				Map<String, Object> mappings = response.getMappings().get(INDEX_NAME).getSourceAsMap();
				Map<String, Object> properties = (Map<String, Object>) mappings.get("properties");
				for(Map.Entry<String, String> entry : orderMap.entrySet()) {
					Map<String, Object> fieldMapping = (Map<String, Object>) properties.get(entry.getKey());
					if("text".equals(fieldMapping.get("type").toString())){
						sourceBuilder.sort(new FieldSortBuilder(entry.getKey()+".keyword").order(ASC.equals(entry.getValue().toLowerCase())?SortOrder.ASC:SortOrder.DESC));
					}else {
						sourceBuilder.sort(new FieldSortBuilder(entry.getKey()).order(ASC.equals(entry.getValue().toLowerCase())?SortOrder.ASC:SortOrder.DESC));
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		
		sourceBuilder.from(queryParam.getPageIndex() <= 0 ? 0 : (queryParam.getPageIndex()-1)*queryParam.getPageSize());
        sourceBuilder.size(queryParam.getPageSize());
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        try {
        	page = queryPage(searchRequest);
        } catch (IOException e) {
        	try { //重试一次
				page = queryPage(searchRequest);
			} catch (IOException e1) {
				e1.printStackTrace();
        		throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e1.getMessage());
			}
        }
        return page;
	}


	public Boolean checkAndCreateESIndex(){
		logger.info("判断索引{}是否存在.........",INDEX_NAME);
		try {
			GetIndexRequest request = new GetIndexRequest(INDEX_NAME);
			boolean exist = client.indices().exists(request, RequestOptions.DEFAULT);
			if(!exist){
				logger.info("索引{}不存在.........新建",INDEX_NAME);
				CreateIndexRequest createIndexRequest = new CreateIndexRequest(INDEX_NAME);
				createIndexRequest.settings(Settings.builder()
						.put("index.number_of_shards", 1)
						.put("index.number_of_replicas", 1)
				);
				XContentBuilder mapping = jsonBuilder().startObject().startObject("properties");
				mapping.startObject("id").field("type", "text").endObject();
				mapping.startObject("content")
						.field("type", "text")
							.startObject("fields")
								.startObject("keyword")
									.field("type", "keyword")
									.field("ignore_above", 1200)
								.endObject()
							.endObject()
						.endObject();
				mapping.startObject("paramContent")
				.field("type", "text")
					.startObject("fields")
						.startObject("keyword")
							.field("type", "keyword")
							.field("ignore_above", 1200)
						.endObject()
					.endObject()
				.endObject();
				mapping.startObject("operateTime")
							.field("type", "date")
							.field("format","yyyy-MM-dd HH:mm:ss")
						.endObject();
				mapping.startObject("operateUser")
						.field("type", "text")
							.startObject("fields")
								.startObject("keyword")
									.field("type", "keyword")
									.field("ignore_above", 1200)
								.endObject()
							.endObject()
						.endObject();
				mapping.startObject("operateUserName")
						.field("type", "text")
							.startObject("fields")
								.startObject("keyword")
									.field("type", "keyword")
									.field("ignore_above", 1200)
								.endObject()
							.endObject()
						.endObject();
				mapping.endObject().endObject();
				createIndexRequest.mapping(mapping);
				/** 设置索引别名*/
				createIndexRequest.alias(new Alias("API调用日志"));
				CreateIndexResponse response = client.indices().create(createIndexRequest, RequestOptions.DEFAULT);
				boolean acknowledged = response.isAcknowledged();
				if (!acknowledged) {
					response = client.indices().create(createIndexRequest, RequestOptions.DEFAULT);
					acknowledged = response.isAcknowledged();
					if (!acknowledged) {
						logger.info("创建 API调用日志 ES库失败！");
						return false;
					}
					logger.info("索引{}不存在.........新建成功",INDEX_NAME);
				}
			}else {
				logger.info("索引{}存在.........",INDEX_NAME);
			}
			return true;
		} catch (IOException e) {
			e.printStackTrace();
		}
		return false;
	}

	@Override
	public List<OperateLogVO> exportData(List<String> param) {

		SearchRequest searchRequest = new SearchRequest(INDEX_NAME);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();

		//查询参数
		BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
		boolQuery.must(QueryBuilders.rangeQuery("operateTime").from(param.get(0)).to(param.get(1)).includeLower(true).includeUpper(true));
		sourceBuilder.query(boolQuery);

		//排序
		GetIndexRequest request = new GetIndexRequest(INDEX_NAME);
		try {
			GetIndexResponse response = client.indices().get(request, RequestOptions.DEFAULT);
			Map<String, Object> mappings = response.getMappings().get(INDEX_NAME).getSourceAsMap();
			Map<String, Object> properties = (Map<String, Object>) mappings.get("properties");
			Map<String, Object> fieldMapping = (Map<String, Object>) properties.get("operateTime");
			if("text".equals(fieldMapping.get("type").toString())){
				sourceBuilder.sort(new FieldSortBuilder("operateTime.keyword").order(SortOrder.DESC));
			}else {
				sourceBuilder.sort(new FieldSortBuilder("operateTime").order(SortOrder.DESC));
			}
		} catch (IOException e) {
			e.printStackTrace();
		}

		sourceBuilder.trackTotalHits(true);
		sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
		searchRequest.source(sourceBuilder);
		List<OperateLogVO> list = new ArrayList<>();
		try {
			list = queryList(searchRequest);
		} catch (IOException e) {
			try { //重试一次
				list = queryList(searchRequest);
			} catch (IOException e1) {
				e1.printStackTrace();
				throw new BusinessException("导出失败，MSG："+e1.getMessage());
			}
		}

		return list;
	}

	//查询es
	private IPage<OperateLogVO> queryPage(SearchRequest searchRequest) throws IOException {
		IPage<OperateLogVO> page = new Page<>();
		List<OperateLogVO> list = new ArrayList<>();
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHits hits = response.getHits();
        for (SearchHit hit : hits) {
            String sourceAsString = hit.getSourceAsString();
            list.add(JSON.parseObject(sourceAsString, OperateLogVO.class));
        }
        page.setRecords(list);
        page.setTotal(hits.getTotalHits().value);
        return page;
	}
	private List<OperateLogVO> queryList(SearchRequest searchRequest) throws IOException {
		List<OperateLogVO> list = new ArrayList<>();
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHits hits = response.getHits();
        for (SearchHit hit : hits) {
            String sourceAsString = hit.getSourceAsString();
            list.add(JSON.parseObject(sourceAsString, OperateLogVO.class));
        }
        return list;
	}

}
