package com.ejianc.business.income.controller.api;

import com.ejianc.business.income.service.IProductionPlanService;
import com.ejianc.business.income.service.IProductionService;
import com.ejianc.business.income.utils.ComputeUtil;
import com.ejianc.business.income.vo.BlocKanBanVo;
import com.ejianc.business.income.vo.ProductionPlanVO;
import com.ejianc.business.income.vo.ProductionVo;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @ClassName: BlocKanBanApi
 * @Description:
 * @Author: 曹鹏辉
 * @Date: 2021/9/3 11:00
 */
@RestController
@RequestMapping("/api/blocKanBan/")
public class IncomeBlocKanBanApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IProductionPlanService planService;

    @Autowired
    private IProductionService productionService;


    /**
     * 集团看板产值top5
     * @param orgId
     * @param productionFlag 0-产值金额(默认)，1-产值完成率
     * @param yearFlag 0-今年(默认)，1-全年
     * @return
     */
    @RequestMapping(value = "queryProduction", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<BlocKanBanVo>> queryProduction(@RequestParam(value = "orgId", required = false) Long orgId
            , @RequestParam(value = "productionFlag") Integer productionFlag, @RequestParam(value = "yearFlag") Integer yearFlag){

        if (orgId == null) {
            Long tenantid = InvocationInfoProxy.getTenantid();
            CommonResponse<OrgVO> orgByTenantId = orgApi.findOrgByTenantId(tenantid);
            orgId = orgByTenantId.getData().getId();
        }

        // 本下组织
        List<Long> orgIds = orgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList());

        // 获取集团下所有组织信息
        List<OrgVO> data = getOrgVoList();
        // 获得分子公司map
        Map<String, OrgVO> innerCodeMap = data.stream().filter(vo -> vo.getOrgType() == 2).collect(Collectors.toMap(OrgVO::getInnerCode, Function.identity()));
        // 全部组织map
        Map<Long, OrgVO> idMap = data.stream().collect(Collectors.toMap(OrgVO::getId, Function.identity()));

        // 获取产值计划总金额
        Map<String, BigDecimal> planTaxMnyMap = queryTotalPlanTaxMny(orgIds, idMap, innerCodeMap, yearFlag);
        // 获取产值总金额
        Map<String, BigDecimal> finishTaxMnyMap = queryTotalFinishTaxMny(orgIds, idMap, innerCodeMap, yearFlag);

        Set<String> orgNameSet = new HashSet<>(planTaxMnyMap.keySet());
        orgNameSet.addAll(finishTaxMnyMap.keySet());

        List<BlocKanBanVo> voList = new ArrayList<>();

        for (String orgName : orgNameSet) {
            BlocKanBanVo vo = new BlocKanBanVo();
            vo.setOrgName(orgName);
            vo.setTotalPlanTaxMny(planTaxMnyMap.getOrDefault(orgName, BigDecimal.ZERO));
            vo.setTotalFinishTaxMny(finishTaxMnyMap.getOrDefault(orgName, BigDecimal.ZERO));
            vo.setProductionCompleteRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalFinishTaxMny(), vo.getTotalPlanTaxMny()), new BigDecimal("100")));

            if (vo.getProductionCompleteRate().compareTo(new BigDecimal("0E-8")) == 0) {
                vo.setProductionCompleteRate(BigDecimal.ZERO);
            }

            voList.add(vo);
        }

        if (productionFlag == 0) {
            voList = voList.stream().filter(vo -> vo.getTotalFinishTaxMny() != null).sorted(Comparator.comparing(BlocKanBanVo::getTotalFinishTaxMny).reversed().thenComparing(BlocKanBanVo::getTotalPlanTaxMny,Comparator.reverseOrder())).limit(5).collect(Collectors.toList());
        }else {
            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getProductionCompleteRate).reversed().thenComparing(BlocKanBanVo::getTotalFinishTaxMny, Comparator.reverseOrder()).thenComparing(BlocKanBanVo::getTotalPlanTaxMny,Comparator.reverseOrder())).limit(5).collect(Collectors.toList());
        }

        return CommonResponse.success("查询产值top5成功", voList);

    }

    /**
     * 获取集团下所有组织信息
     * @return
     */
    public List<OrgVO> getOrgVoList() {
        List<Integer> typeList = new ArrayList<>();
        typeList.add(1);
        typeList.add(2);
        typeList.add(3);
        typeList.add(4);
        typeList.add(5);
        CommonResponse<List<OrgVO>> orgByType = orgApi.findOrgByType(InvocationInfoProxy.getTenantid(), typeList, null);

        return orgByType.getData();
    }

    /**
     * 获取分子公司下 产值计划总金额
     * @param orgIds
     * @param idMap
     * @param innerCodeMap
     * @param yearFlag
     * @return
     */
    public Map<String, BigDecimal> queryTotalPlanTaxMny(List<Long> orgIds, Map<Long, OrgVO> idMap
            , Map<String, OrgVO> innerCodeMap, Integer yearFlag) {

        List<ProductionPlanVO> productionPlanVOS = planService.queryTotalPlanMny(orgIds, yearFlag);

        Map<String, List<ProductionPlanVO>> map = new HashMap<>();
        Map<String, BigDecimal> resultMap = new HashMap<>();

        for (ProductionPlanVO vo : productionPlanVOS) {
            OrgVO orgVO = idMap.get(vo.getOrgId());
            String innerCode = orgVO.getInnerCode();

            String orgName = null;
            for (Map.Entry<String, OrgVO> entry : innerCodeMap.entrySet()) {
                if (innerCode.contains(entry.getKey())) {
                    orgName = entry.getValue().getName();

                    if (map.containsKey(orgName)) {
                        map.get(orgName).add(vo);
                    }else {
                        map.put(orgName, new ArrayList<>());
                        map.get(orgName).add(vo);
                    }
                }
            }
        }

        for (Map.Entry<String, List<ProductionPlanVO>> entry : map.entrySet()) {
            BigDecimal totalPlanTaxMny = entry.getValue().stream().map(ProductionPlanVO::getPlanTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            resultMap.put(entry.getKey(), totalPlanTaxMny);
        }

        return resultMap;
    }

    /**
     * 获取分子公司下 产值总金额
     * @param orgIds
     * @param idMap
     * @param innerCodeMap
     * @param yearFlag
     * @return
     */
    public Map<String, BigDecimal> queryTotalFinishTaxMny(List<Long> orgIds, Map<Long, OrgVO> idMap
            , Map<String, OrgVO> innerCodeMap, Integer yearFlag) {

        List<ProductionVo> productionVos = productionService.queryTotalFinishTaxMny(orgIds, yearFlag);

        Map<String, List<ProductionVo>> map = new HashMap<>();
        Map<String, BigDecimal> resultMap = new HashMap<>();

        for (ProductionVo vo : productionVos) {
            OrgVO orgVO = idMap.get(vo.getOrgId());
            String innerCode = orgVO.getInnerCode();

            String orgName = null;
            for (Map.Entry<String, OrgVO> entry : innerCodeMap.entrySet()) {
                if (innerCode.contains(entry.getKey())) {
                    orgName = entry.getValue().getName();

                    if (map.containsKey(orgName)) {
                        map.get(orgName).add(vo);
                    }else {
                        map.put(orgName, new ArrayList<>());
                        map.get(orgName).add(vo);
                    }
                }
            }
        }

        for (Map.Entry<String, List<ProductionVo>> entry : map.entrySet()) {
            BigDecimal totalFinishTaxMny = entry.getValue().stream().map(ProductionVo::getFinishTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            resultMap.put(entry.getKey(), totalFinishTaxMny);

        }

        return resultMap;
    }

}
