package com.ejianc.business.income.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.income.bean.*;
import com.ejianc.business.income.history.QuoteHistoryVo;
import com.ejianc.business.income.mapper.QuoteMapper;
import com.ejianc.business.income.service.*;
import com.ejianc.business.income.utils.DateUtil;
import com.ejianc.business.income.utils.TreeNodeBUtil;
import com.ejianc.business.income.vo.QuoteClaimVo;
import com.ejianc.business.income.vo.QuoteDetailVo;
import com.ejianc.business.income.vo.QuoteOtherVo;
import com.ejianc.business.income.vo.QuoteVo;
import com.ejianc.business.income.vo.comparator.QuoteDetailComparatorVo;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 甲方报量 服务实现类
 * </p>
 *
 * @author yuezx
 * @since 2020-06-04
 */
@Service("QuoteService")
public class QuoteServiceImpl extends BaseServiceImpl<QuoteMapper, QuoteEntity> implements IQuoteService {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private static final String QUOTE_BILL_CODE = "INCOME_QUOTE";
    @Autowired
    private QuoteMapper quoteMapper;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IQuoteDetailService quoteDetailService;
    @Autowired
    private IQuoteClaimService quoteClaimService;
    @Autowired
    private IQuoteOtherService quoteOtherService;
    @Autowired
    private IContractService contractService;
    @Autowired
    private JedisPool jedisPool;

    @Autowired
    private IClaimService claimService;
    @Autowired
    private IProductionService productionService;
    @Override
    public CommonResponse<QuoteVo> saveOrUpdate(QuoteVo quoteVo) {
//        if (null == quoteVo.getCheckList() || quoteVo.getCheckList().size() < 1) {
//            throw new BusinessException("产值明细不可以为空");
//        }
        //保存时校验合同version是否一致
        ContractEntity contractEntity = contractService.selectById(quoteVo.getContractId());
        if(contractEntity != null && contractEntity.getIsFinish() == 1){
            return CommonResponse.error("该合同已做竣工结算，不能做甲方批量！");
        }

        Jedis jedis = jedisPool.getResource();
        boolean locked = false;
        try {
            locked = RedisTool.tryLock(jedis, String.valueOf(quoteVo.getContractId()), "saveOrUpdate", 1000);
            logger.info("判断单据单据锁结果------" + locked);
            if (quoteVo.getContractVersion() != null && quoteVo.getContractVersion() != 0) {
                if (locked) {
                    Integer version = contractEntity.getVersion() == null ? 0 : contractEntity.getVersion();
                    Integer conVersion = quoteVo.getContractVersion();
                    if (version != conVersion) {
                        return CommonResponse.error("施工合同已被更新，请刷新后重做！");
                    }
                }else{
                    return CommonResponse.error("出现并发操作,请稍后重试！");
                }
            }

            List<QuoteDetailVo> checkVos = quoteVo.getCheckList();
            if(checkVos != null && checkVos.size() > 0){
                Map<Long, Integer> versionMap = new HashMap<>();
                for(QuoteDetailVo detailVo : checkVos){
                    if (detailVo.getProductionVersion() != null && detailVo.getProductionVersion() != 0) {
                        versionMap.put(detailVo.getProductionId(), detailVo.getProductionVersion());
                    }
                }
                if (versionMap != null && versionMap.size() > 0 ) {
                    if (locked) {
                        QueryParam param = new QueryParam();
                        param.getParams().put("id", new Parameter("in", new ArrayList<Long>(versionMap.keySet())));
                        List<ProductionEntity> productionEntityList = productionService.queryList(param, false);
                        for (ProductionEntity productionEntity : productionEntityList) {
                            if ((productionEntity.getVersion() == null ? 0 : productionEntity.getVersion()) != versionMap.get(productionEntity.getId())) {
                                return CommonResponse.error("产值进度已被更新，请刷新后重做！");
                            }
                        }
                    }else{
                        return CommonResponse.error("出现并发操作,请稍后重试！");
                    }
                }
            }

            List<QuoteClaimVo> claimVos = quoteVo.getClaimList();
            if (claimVos != null && claimVos.size() > 0) {
                Map<Long, Integer> versionMap = new HashMap<>();
                for (QuoteClaimVo claimVo : claimVos) {
                    if (claimVo.getClaimVersion() != null && claimVo.getClaimVersion() != 0) {
                        versionMap.put(claimVo.getClaimId(), claimVo.getClaimVersion());
                    }
                }
                if (versionMap != null && versionMap.size() > 0 ) {
                    if (locked) {
                        QueryParam param = new QueryParam();
                        param.getParams().put("id", new Parameter("in", new ArrayList<Long>(versionMap.keySet())));
                        List<ClaimEntity> claimEntities = claimService.queryList(param, false);
                        for (ClaimEntity claimEntity : claimEntities) {
                            if ((claimEntity.getVersion() == null ? 0 : claimEntity.getVersion()) != versionMap.get(claimEntity.getId())) {
                                return CommonResponse.error("变更签证洽商索赔已被更新，请刷新后重做！");
                            }
                        }
                    }else{
                        return CommonResponse.error("出现并发操作,请稍后重试！");
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (locked) {
                RedisTool.releaseLock(jedis, String.valueOf(quoteVo.getContractId()), "saveOrUpdate");
            }
            jedis.close();
        }

        LambdaQueryWrapper<QuoteEntity> lamb = new LambdaQueryWrapper();
        lamb.eq(QuoteEntity::getContractId, quoteVo.getContractId());
        lamb.ne(quoteVo.getId() != null && quoteVo.getId() > 0, QuoteEntity::getId, quoteVo.getId());
        lamb.last("and date_format(quote_date,'%Y-%m')='" + DateUtil.getYearAndMonth(quoteVo.getQuoteDate())+ "'");
        List<QuoteEntity> list = super.list(lamb);
        if(CollectionUtils.isNotEmpty(list)){
            throw new BusinessException("该合同在该月份已制定甲方批量，不允许新增！");
        }

        Date signDate = contractEntity.getSignDate();
        if (null != signDate && quoteVo.getQuoteDate().getTime() < signDate.getTime()) {
            throw new BusinessException("批量日期只能是大于等于施工合同的签约日期");
        }
        Date lastDate = quoteMapper.getLastDate(quoteVo.getContractId());
        if (null != lastDate && quoteVo.getQuoteDate().getTime() < lastDate.getTime()) {
            throw new BusinessException("该合同存在比当前月份更晚的产值统计，不允许新增！");
        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        QuoteEntity entity = null;
        String operateType = null;
        if (quoteVo.getId() != null && quoteVo.getId() > 0) { //修改
            if(StringUtils.isEmpty(quoteVo.getBillCode())){
                quoteVo.setBillCode(null);
            }
            entity = BeanMapper.map(quoteVo, QuoteEntity.class);
            operateType = "edit";
        } else {
            //同一个合同只能存在一个自由态或审批中的单据
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, quoteVo.getContractId()));
            List<Integer> billStatus = new ArrayList<>();
            billStatus.add(0);
            billStatus.add(2);
            billStatus.add(5);
            queryParam.getParams().put("bill_state", new Parameter(QueryParam.IN, billStatus));
            List<QuoteEntity> entities = super.queryList(queryParam, false);
            if (null != entities && entities.size() > 0) {
                throw new BusinessException("当前合同存在非审批通过态的甲方批量，不允许新增");
            }
            //新增
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(QUOTE_BILL_CODE, tenantId);
            if (billCode.isSuccess()) {
                quoteVo.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }

            entity = BeanMapper.map(quoteVo, QuoteEntity.class);
            operateType = "add";
        }
        if (quoteVo.getId() != null && quoteVo.getId() > 0) {
            //修改  校验合同编号是否重复
            LambdaQueryWrapper<QuoteEntity> lambda = Wrappers.<QuoteEntity>lambdaQuery();
            lambda.eq(QuoteEntity::getBillCode, quoteVo.getBillCode());
            lambda.eq(QuoteEntity::getTenantId, tenantId);
            lambda.ne(QuoteEntity::getId, quoteVo.getId());
            List<QuoteEntity> entities = super.list(lambda);
            if (entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        } else {
            //校验合同编号是否重复
            LambdaQueryWrapper<QuoteEntity> lambda = Wrappers.<QuoteEntity>lambdaQuery();
            lambda.eq(QuoteEntity::getTenantId, tenantId);
            lambda.eq(QuoteEntity::getBillCode, quoteVo.getBillCode());
            List<QuoteEntity> entities = super.list(lambda);
            if (entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        }
        super.saveOrUpdate(entity);

        List<QuoteDetailVo> detailsVos = quoteVo.getCheckList();
        List<QuoteDetailEntity> saveOrUpldates = new ArrayList<>();
        List<Long> deleteIds = new ArrayList<>();
        for (QuoteDetailVo detailsVo : detailsVos) {
            if ("add".equals(detailsVo.getRowState())) {
                QuoteDetailEntity detailslist = BeanMapper.map(detailsVo, QuoteDetailEntity.class);
                detailslist.setQuoteId(entity.getId());
                detailslist.setId(null);
                saveOrUpldates.add(detailslist);
            } else if ("edit".equals(detailsVo.getRowState())) {
                QuoteDetailEntity detailslist = BeanMapper.map(detailsVo, QuoteDetailEntity.class);
                saveOrUpldates.add(detailslist);
            } else if ("del".equals(detailsVo.getRowState())) {
                deleteIds.add(detailsVo.getId());
            }
        }
        if (saveOrUpldates.size() > 0) {
            quoteDetailService.saveOrUpdateBatch(saveOrUpldates, saveOrUpldates.size(), false);
            //维护父子级关系
            Map<String, Long> idMap = new HashMap<>();
            for (QuoteDetailEntity cdEntity : saveOrUpldates) {
                idMap.put(cdEntity.getTid(), cdEntity.getId());
            }
            for (QuoteDetailEntity cdEntity : saveOrUpldates) {
                if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            quoteDetailService.saveOrUpdateBatch(saveOrUpldates, saveOrUpldates.size(), false);
        }
        if (deleteIds.size() > 0) {
            quoteDetailService.removeByIds(deleteIds, false);
        }

        List<QuoteClaimVo> clauseVos = quoteVo.getClaimList();
        List<QuoteClaimEntity> clauseList = new ArrayList<>();
        List<Long> deleteClauseIds = new ArrayList<>();
        for (QuoteClaimVo clauseVo : clauseVos) {
            if ("add".equals(clauseVo.getRowState())) {
                QuoteClaimEntity detailslist = BeanMapper.map(clauseVo, QuoteClaimEntity.class);
                detailslist.setQuoteId(entity.getId());
                detailslist.setId(null);
                clauseList.add(detailslist);
            } else if ("edit".equals(clauseVo.getRowState())) {
                QuoteClaimEntity detailslist = BeanMapper.map(clauseVo, QuoteClaimEntity.class);
                clauseList.add(detailslist);
            } else if ("del".equals(clauseVo.getRowState())) {
                deleteClauseIds.add(clauseVo.getId());
            }
        }
        if (clauseList.size() > 0) {
            quoteClaimService.saveOrUpdateBatch(clauseList, clauseList.size(), false);
        }
        if (deleteClauseIds.size() > 0) {
            quoteClaimService.removeByIds(deleteClauseIds, false);
        }

        List<QuoteOtherVo> otherVos = quoteVo.getOtherList();
        List<QuoteOtherEntity> otherList = new ArrayList<>();
        List<Long> deleteOtherIds = new ArrayList<>();
        for (QuoteOtherVo otherVo : otherVos) {
            if ("add".equals(otherVo.getRowState())) {
                QuoteOtherEntity detailslist = BeanMapper.map(otherVo, QuoteOtherEntity.class);
                detailslist.setQuoteId(entity.getId());
                detailslist.setId(null);
                otherList.add(detailslist);
            } else if ("edit".equals(otherVo.getRowState())) {
                QuoteOtherEntity detailslist = BeanMapper.map(otherVo, QuoteOtherEntity.class);
                otherList.add(detailslist);
            } else if ("del".equals(otherVo.getRowState())) {
                deleteOtherIds.add(otherVo.getId());
            }
        }
        if (otherList.size() > 0) {
            quoteOtherService.saveOrUpdateBatch(otherList, otherList.size(), false);
        }
        if (deleteOtherIds.size() > 0) {
            quoteOtherService.removeByIds(deleteOtherIds, false);
        }
        return CommonResponse.success(queryDetail(entity.getId()));
    }

    @Override
    public QuoteVo queryDetail(Long id) {
        QuoteEntity entity = quoteMapper.selectById(id);
        if (entity != null) {
            QuoteVo vo = BeanMapper.map(entity, QuoteVo.class);
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("quoteId", new Parameter(QueryParam.EQ, vo.getId()));
            //queryParam.getOrderMap().put("detailIndex", "asc");
            List<QuoteDetailEntity> details = quoteDetailService.queryList(queryParam, false);
            if (details != null && details.size() > 0) {
                for (QuoteDetailEntity cdEntity : details) {
                    cdEntity.setTid(cdEntity.getId().toString());
                    cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");
                    cdEntity.setRowState("edit");
                }
                List<QuoteDetailVo> resultMapList = BeanMapper.mapList(details, QuoteDetailVo.class);
                //实现排序
                Collections.sort(resultMapList,new QuoteDetailComparatorVo());
                vo.setCheckList(TreeNodeBUtil.buildTree(resultMapList));
            }
            QueryParam parm = new QueryParam();
            parm.getParams().put("quoteId", new Parameter(QueryParam.EQ, vo.getId()));
            List<QuoteClaimEntity> claimEntities = quoteClaimService.queryList(parm, false);
            vo.setClaimList(BeanMapper.mapList(claimEntities, QuoteClaimVo.class));
            List<QuoteOtherEntity> otherEntities = quoteOtherService.queryList(parm, false);
            vo.setOtherList(BeanMapper.mapList(otherEntities, QuoteOtherVo.class));
            return vo;
        }
        return null;
    }

    @Override
    public void deleteQuote(List<QuoteVo> vos) {
        super.removeByIds(vos.stream().map(QuoteVo::getId).collect(Collectors.toList()), false);
    }

    @Override
    public QuoteHistoryVo queryQuoteHistory(Long id) {
        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);
        List<QuoteEntity> quoteEntities = quoteMapper.selectList(new QueryWrapper<QuoteEntity>()
                .eq("contract_id", id)
                .in("bill_state", billStatus).orderByDesc("quote_date", "create_time"));
        ContractEntity contractEntity = contractService.selectById(id);
        QuoteHistoryVo vo = new QuoteHistoryVo();
        vo.setContractId(id);
        vo.setIsFinish(contractEntity.getIsFinish());
        vo.setContractTaxMny(contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny());
        vo.setSumQuoteTaxMny(contractEntity.getSumQuoteTaxMny());
        BigDecimal sumQuoteTaxMny = contractEntity.getSumQuoteTaxMny() == null ? BigDecimal.ZERO : contractEntity.getSumQuoteTaxMny();
        if (contractEntity.getContractTaxMny() == null || contractEntity.getContractTaxMny().compareTo(BigDecimal.ZERO) == 0) {
            vo.setSumQuoteRate(BigDecimal.ZERO);//累计形象进度
        } else {
            vo.setSumQuoteRate(sumQuoteTaxMny.divide(contractEntity.getContractTaxMny(), 8, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100)));
        }
        vo.setQuoteRecord(BeanMapper.mapList(quoteEntities, QuoteVo.class));
        return vo;
    }
}
