package com.ejianc.business.jlprogress.quality.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.jlprogress.quality.bean.DisposeEntity;
import com.ejianc.business.jlprogress.quality.bean.ProcessCheckEntity;
import com.ejianc.business.jlprogress.quality.service.IDisposeService;
import com.ejianc.business.jlprogress.quality.service.IProcessCheckService;
import com.ejianc.business.jlprogress.quality.vo.PassRateVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.Month;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 质量检测-生产交检合格率汇总
 */
@Controller
@RequestMapping("passRate")
public class PassRateController implements Serializable {
    private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IProcessCheckService processCheckService;

    @Autowired
    private IDisposeService disposeService;

    /**
     * @param param
     * @Description queryList 查询列表
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<PassRateVO>> queryList(@RequestBody QueryParam param) {

        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        List<PassRateVO> uniquePassRateVOS = getPassRateVO(param);
        IPage<PassRateVO> pageData = new Page<>(param.getPageIndex(), param.getPageSize(), uniquePassRateVOS.size());
        pageData.setRecords(uniquePassRateVOS);

        return CommonResponse.success("查询列表数据成功！", pageData);
    }

    /**
     * @param param
     * @Description 导出
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {

        List<PassRateVO> uniquePassRateVOS = getPassRateVO(param);

        Map<String, Object> beans = new HashMap<>();
        beans.put("records", uniquePassRateVOS);
        ExcelExport.getInstance().export("PassRate-export.xlsx", beans, response);
    }

    /**
     * 获取List<PassRateVO>
     * @param param
     * @return
     */
    private List<PassRateVO> getPassRateVO(QueryParam param) {
        String startOfMonth = "";
        String endOfMonth = "";
        String endOfMonth26 = "";
        List<ProcessCheckEntity> processCheckEntities = new ArrayList<>();
        List<ProcessCheckEntity> fullScaleProcessCheck = new ArrayList<>();
        List<DisposeEntity> disposeEntities = new ArrayList<>();

        if (param.getParams().containsKey("month")) {
            //2024-07
            String checkMonth = param.getParams().get("month").getValue().toString();
            // 解析月份
            LocalDate date = LocalDate.parse(checkMonth + "-01", DateTimeFormatter.ISO_LOCAL_DATE);
            // 获取上个月的日期
            Month lastMonth = date.minusMonths(1).getMonth();
            Month month = date.getMonth();
            int year = date.minusMonths(1).getYear();
            // 获取上个月的26日
            startOfMonth = LocalDate.of(year, lastMonth, 26).toString();
            // 获取本月的25日
            endOfMonth = LocalDate.of(year, month, 25).toString();
            // 获取本月的26日
            endOfMonth26 = LocalDate.of(year, month, 26).toString();
            param.getParams().remove("month");
            param.getParams().put("month", new Parameter(QueryParam.BETWEEN, startOfMonth, endOfMonth));
        }
        if (param.getParams().containsKey("orgId")) {
            Long orgId = Long.valueOf(param.getParams().get("orgId").getValue().toString());
            //生效态单据
            ArrayList<Integer> passedBill = new ArrayList<>();
            passedBill.add(BillStateEnum.PASSED_STATE.getBillStateCode());
            passedBill.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
            //查询合格率为100%的工序检测记录
            LambdaQueryWrapper<ProcessCheckEntity> checkWrapper = new LambdaQueryWrapper<>();
            checkWrapper.eq(ProcessCheckEntity::getProductOrgId, orgId);
            checkWrapper.eq(ProcessCheckEntity::getQualifiedScale, new BigDecimal(100));
            checkWrapper.in(ProcessCheckEntity::getBillState, passedBill);
            checkWrapper.between(ProcessCheckEntity::getCheckDate, startOfMonth, endOfMonth);
            fullScaleProcessCheck = processCheckService.list(checkWrapper);
            //查询不合格品处置单
            LambdaQueryWrapper<DisposeEntity> disposeWrapper = new LambdaQueryWrapper<>();
            disposeWrapper.eq(DisposeEntity::getProductOrgId, orgId);
            disposeWrapper.in(DisposeEntity::getBillState, passedBill);
            disposeWrapper.between(DisposeEntity::getCreateTime, startOfMonth, endOfMonth26);
            disposeEntities = disposeService.list(disposeWrapper);
        }

        //获取符合条件的不合格品处置单对应的工序检测记录的单据编号
        List<String> checkCodes = disposeEntities.stream().map((item) -> item.getCheckCode()).collect(Collectors.toList());
        if (checkCodes.size() != 0) {
            LambdaQueryWrapper<ProcessCheckEntity> wrapper = new LambdaQueryWrapper<>();
            wrapper.in(ProcessCheckEntity::getBillCode, checkCodes);
            processCheckEntities = processCheckService.list(wrapper);
        }
        //合并工序检测记录结果
        processCheckEntities.addAll(fullScaleProcessCheck);
        //拿到工序检测记录的单据编号
        List<String> billCodes = processCheckEntities.stream().map((item) -> item.getBillCode()).collect(Collectors.toList());
        List<DisposeEntity> checkDisposeEntities = new ArrayList<>();
        if (billCodes.size() != 0) {
            //根据本月的工序检测单据编号过滤,筛选出对应工序检测单在本月的不合格品处置单
            checkDisposeEntities = disposeEntities.stream().filter((item) -> billCodes.contains(item.getCheckCode())).collect(Collectors.toList());
        }
        List<PassRateVO> passRateVOS = new ArrayList<>();
        for (ProcessCheckEntity processCheckEntity : processCheckEntities) {
            PassRateVO passRateVO = new PassRateVO();
            //检验总数
            BigDecimal totalNum = processCheckEntity.getTotalNum();
            //合格率为100时,各处置数量为0
            if (processCheckEntity.getQualifiedScale().compareTo(new BigDecimal(100)) == 0) {
                passRateVO.setId(processCheckEntity.getId());
                passRateVO.setProcessId(processCheckEntity.getProcessId());
                passRateVO.setProcessName(processCheckEntity.getProcessName());
                passRateVO.setProcessTotalNum(totalNum);
                passRateVO.setQualifiedNum(totalNum);
                passRateVO.setConcession(BigDecimal.ZERO);
                passRateVO.setRebuild(BigDecimal.ZERO);
                passRateVO.setRework(BigDecimal.ZERO);
                passRateVO.setWorkWaste(BigDecimal.ZERO);
                passRateVO.setMaterialWaste(BigDecimal.ZERO);
                passRateVO.setQualifiedScale(new BigDecimal(100));
                passRateVO.setQualityLoss(BigDecimal.ZERO);
                passRateVO.setCreateTime(processCheckEntity.getCreateTime());
                passRateVOS.add(passRateVO);
            }else {
                //取不合格品处置单中的财务损失,赋值
                List<DisposeEntity> checkedDisposeEntities = checkDisposeEntities.stream().filter(item -> item.getCheckCode().equals(processCheckEntity.getBillCode())).collect(Collectors.toList());
                for (DisposeEntity disposeEntity : checkedDisposeEntities) {
                    passRateVO.setQualityLoss(disposeEntity.getFinancialLoss());
                }
                //让步接收
                BigDecimal concession = new BigDecimal(checkedDisposeEntities.stream().filter(item -> item.getSchemeName().equals("让步接收")).mapToInt(DisposeEntity::getNoNum).sum());
                //返修
                BigDecimal rebuild = new BigDecimal(checkedDisposeEntities.stream().filter(item -> item.getSchemeName().equals("返修")).mapToInt(DisposeEntity::getNoNum).sum());
                //返工
                BigDecimal rework = new BigDecimal(checkedDisposeEntities.stream().filter(item -> item.getSchemeName().equals("返工")).mapToInt(DisposeEntity::getNoNum).sum());
                //工废
                BigDecimal workWaste = new BigDecimal(checkedDisposeEntities.stream().filter(item -> item.getSchemeName().equals("工废")).mapToInt(DisposeEntity::getNoNum).sum());
                //料废
                BigDecimal materialWaste = new BigDecimal(checkedDisposeEntities.stream().filter(item -> item.getSchemeName().equals("料废")).mapToInt(DisposeEntity::getNoNum).sum());

                //计算合格数量
                BigDecimal qualifiedNum = totalNum.subtract(concession).subtract(rebuild).subtract(rework).subtract(workWaste).subtract(materialWaste);
                //赋值
                passRateVO.setId(processCheckEntity.getId());
                passRateVO.setProcessId(processCheckEntity.getProcessId());
                passRateVO.setProcessName(processCheckEntity.getProcessName());
                passRateVO.setProcessTotalNum(totalNum);
                passRateVO.setQualifiedNum(qualifiedNum);
                passRateVO.setConcession(concession);
                passRateVO.setRebuild(rebuild);
                passRateVO.setRework(rework);
                passRateVO.setWorkWaste(workWaste);
                passRateVO.setMaterialWaste(materialWaste);
                passRateVO.setQualifiedScale(qualifiedNum.divide(totalNum, 8, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100)));
                passRateVO.setCreateTime(processCheckEntity.getCreateTime());
                passRateVOS.add(passRateVO);
            }
        }
        //去重,合并
        List<PassRateVO> uniquePassRateVOS = passRateVOS.stream().collect(Collectors.toMap(PassRateVO::getProcessId, v -> v, (v1, v2) -> {
                    v1.setProcessTotalNum(ensureNotNull(v1.getProcessTotalNum(), BigDecimal.ZERO).add(ensureNotNull(v2.getProcessTotalNum(), BigDecimal.ZERO)));
                    v1.setQualifiedNum(ensureNotNull(v1.getQualifiedNum(), BigDecimal.ZERO).add(ensureNotNull(v2.getQualifiedNum(), BigDecimal.ZERO)));
                    v1.setConcession(ensureNotNull(v1.getConcession(), BigDecimal.ZERO).add(ensureNotNull(v2.getConcession(), BigDecimal.ZERO)));
                    v1.setRebuild(ensureNotNull(v1.getRebuild(), BigDecimal.ZERO).add(ensureNotNull(v2.getRebuild(), BigDecimal.ZERO)));
                    v1.setRework(ensureNotNull(v1.getRework(), BigDecimal.ZERO).add(ensureNotNull(v2.getRework(), BigDecimal.ZERO)));
                    v1.setWorkWaste(ensureNotNull(v1.getWorkWaste(), BigDecimal.ZERO).add(ensureNotNull(v2.getWorkWaste(), BigDecimal.ZERO)));
                    v1.setMaterialWaste(ensureNotNull(v1.getMaterialWaste(), BigDecimal.ZERO).add(ensureNotNull(v2.getMaterialWaste(), BigDecimal.ZERO)));
                    v1.setQualityLoss(ensureNotNull(v1.getQualityLoss(), BigDecimal.ZERO).add(ensureNotNull(v2.getQualityLoss(), BigDecimal.ZERO)));

                    BigDecimal qualifiedNum1 = v1.getProcessTotalNum().subtract(v1.getConcession()).subtract(v1.getRebuild()).subtract(v2.getRework()).subtract(v2.getWorkWaste()).subtract(v2.getMaterialWaste());
                    v1.setQualifiedNum(qualifiedNum1);
                    v1.setQualifiedScale(qualifiedNum1.divide(v1.getProcessTotalNum(), 8, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100)));
                    return v1;
                }
        )).values().stream().collect(Collectors.toList());
        //根据创建时间排序
        if (param.getOrderMap().isEmpty()){
            uniquePassRateVOS.sort(Comparator.comparing(PassRateVO::getCreateTime).reversed());
        }else {
            if (param.getOrderMap().get("createTime").equals("desc")) {
                uniquePassRateVOS.sort(Comparator.comparing(PassRateVO::getCreateTime).reversed());
            } else {
                uniquePassRateVOS.sort(Comparator.comparing(PassRateVO::getCreateTime));
            }
        }
        return uniquePassRateVOS;
    }

    /**
     * 判断值是否为空
     * @param value
     * @param defaultValue
     * @return
     */
    private static BigDecimal ensureNotNull(BigDecimal value, BigDecimal defaultValue) {
        return value != null ? value : defaultValue;
    }
}
