package com.ejianc.business.jlprogress.progress.handler;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.jlprogress.progress.bean.DayFillDetailEntity;
import com.ejianc.business.jlprogress.progress.bean.DayFillEntity;
import com.ejianc.business.jlprogress.progress.service.IDayFillDetailService;
import com.ejianc.business.jlprogress.progress.service.IDayFillService;
import com.ejianc.business.jlprogress.progress.bean.ExecPlanDetailEntity;
import com.ejianc.business.jlprogress.progress.bean.ExecPlanEntity;
import com.ejianc.business.jlprogress.progress.service.IExecPlanDetailService;
import com.ejianc.business.jlprogress.progress.service.IExecPlanService;
import com.ejianc.business.jlprogress.progress.utils.NumberUtil;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yqls
 * @version 1.0
 * @description: 日进度反馈推送执行计划
 * @date 2022/6/16
 */
@Service
public class DayFillHandler implements IPlanHandler {
    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private IDayFillService service;

    @Autowired
    private IDayFillDetailService detailService;

    @Autowired
    private IExecPlanService execService;

    @Autowired
    private IExecPlanDetailService execDetailService;

    /**
     * 计划推送具体实现逻辑, 日进度反馈审批通过后调用
     * 正向推送
     * 1 直接生成执行计划
     * 2 更新执行计划 无则插入,有则增量更新
     * @param id
     * @return
     */
    @Override
    public ExecPlanEntity handle(Long id) {
        DayFillEntity source = service.selectById(id);
        if(source == null) {
            throw new BusinessException("未查询到日进度反馈！");
        }
        QueryWrapper<DayFillDetailEntity> ew = new QueryWrapper<DayFillDetailEntity>()
                .eq("progress_id", id).orderByAsc("tid");
        List<DayFillDetailEntity> sourceList = detailService.list(ew);
        // 执行计划
        ExecPlanEntity data = execService.getOne(new QueryWrapper<ExecPlanEntity>().eq("project_id", source.getProjectId()));
        if(data == null) {
            throw new BusinessException("未查询到项目下生效总计划！");
        }
        List<ExecPlanDetailEntity> detailList = execDetailService.list(new QueryWrapper<ExecPlanDetailEntity>()
                .eq("progress_id", data.getId()).orderByAsc("tid"));
        // 加锁
        Boolean lock = PlanLockUtil.getLock(source.getProjectId());
        if (!lock) {
            throw new BusinessException("获取执行计划锁网络异常，请刷新重试！");
        }
        try {
            // 处理子表
            List<ExecPlanDetailEntity> saveList = this.transferSaveList(sourceList, detailList, source.getFillDate());
            if(CollectionUtils.isNotEmpty(saveList)){
                execDetailService.saveOrUpdateBatch(saveList);
            }
            data.setAllList(detailList);

            // 回写执行计划反馈版本号加一
            Integer fillVersion = data.getFillVersion() == null ? 1 : data.getFillVersion() + 1;
            data.setFillVersion(fillVersion);
            data.setFillTime(new Date());
            execService.saveOrUpdate(data);
        } catch (Exception e) {
            logger.error("推送日进度反馈id-{}给执行计划id-{} 异常，", id, data.getId(), e);
            throw new BusinessException(e.getMessage());
        } finally {
            if (lock) {
                PlanLockUtil.releaseLock(source.getProjectId());
            }
        }
        return data;
    }

    /**
     * 子表转成新增/编辑集合
     * @param sourceList 来源单据明细
     * @param detailList 执行计划明细
     * @return
     */
    private List<ExecPlanDetailEntity> transferSaveList(List<DayFillDetailEntity> sourceList,
            List<ExecPlanDetailEntity> detailList, Date fillDate) {
        List<ExecPlanDetailEntity> saveList = new ArrayList<>();
        Map<String, DayFillDetailEntity> sourceMap = sourceList.stream().filter(x->StringUtils.isNotEmpty(x.getStructCode()))
                .collect(Collectors.toMap(DayFillDetailEntity::getStructCode, x->x));
        detailList.forEach(vo->{
            if(sourceMap.containsKey(vo.getStructCode())){// 编辑覆盖
                DayFillDetailEntity source = sourceMap.get(vo.getStructCode());
                vo.setActualStart(source.getActualStart());// 实际开始
                vo.setActualFinish(source.getActualFinish());// 实际完成
                // 完成时间有值，则预计完成时间置空
                if(vo.getActualFinish() != null){
                    vo.setEstimateFinish(null);
                } else {
                    vo.setEstimateFinish(source.getEstimateFinish());// 预计完成
                }
                vo.setPredictStart(source.getStart());// 预测开始
                vo.setPredictFinish(source.getFinish());// 预测完成
                vo.setPredictDuration(source.getDuration());// 预测工期
                vo.setDiffType(source.getDiffType());// 偏差原因类型取最新生效
                vo.setDiffResson(source.getDiffResson());// 偏差原因取最新生效
                vo.setResourceNum(source.getResourceNum());// 实际资源量
                // 偏差时间=预测完成时间-完成时间
//                vo.setDiffValue(new BigDecimal(DateUtil.getBetweenDays(vo.getPredictFinish(), vo.getPlanFinish())));
                vo.setDiffValue(source.getDiffValue());
                // 资源偏差=实际资源量-需求量
                vo.setDiffNum(ComputeUtil.safeSub(source.getResourceNum(), vo.getNeedNum()));
//                // 完成百分比 =（数据填写时间-实际开始时间 + 1）/（预测完成时间-实际开始时间 + 1）
//                BigDecimal percentComplete = ComputeUtil.bigDecimalPercent(DateUtil.getSubDay(fillDate, vo.getActualStart()),
//                        DateUtil.getSubDay(vo.getPredictFinish(), vo.getActualStart()), 0);
//                // 当任务已填写了实际开始、实际完成时，完成百分比=100%；
//                if(vo.getActualStart() != null && vo.getActualFinish() != null){
//                    percentComplete = new BigDecimal("100");
//                }
//                // 当任务未填写实际开始、预计完成、实际完成时，完成百分比=0；
//                if(vo.getActualStart() == null && vo.getActualFinish() == null && vo.getEstimateFinish() == null){
//                    percentComplete = BigDecimal.ZERO;
//                }
//                if(ComputeUtil.isGreaterThan(percentComplete, new BigDecimal("100"))){
//                    percentComplete = new BigDecimal("100");
//                }
//                vo.setPercentComplete(percentComplete.intValue());
                vo.setPercentComplete(source.getPercentComplete());
//                // 完成工程量=实际完成百分比*工程量
//                BigDecimal finishNum = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(new BigDecimal(vo.getPercentComplete()), new BigDecimal("100")), vo.getPlanNum());
//                vo.setFinishNum(finishNum);
                vo.setFinishNum(source.getFinishNum());
                // 实际开始时间为空，则未开始；实际完成时间为空，则进行中；实际完成时间不为空，则已完成
                Integer finishState = 0;// 未开始
                if(vo.getActualStart() != null){
                    if(vo.getActualFinish() == null){
                        finishState = 2;// 进行中
                    } else {
                        finishState = 1;// 已完成
                    }
                }
                vo.setFinishState(finishState);
                vo.setFillDate(fillDate);
                vo.setInvolveState(source.getInvolveState());
                saveList.add(vo);
            }
        });
        return saveList;
    }

    /**
     * 计划推送回滚实现逻辑, 日进度反馈弃审前调用
     * 1 校验计划是否被使用
     * 2 若使用,不能回滚
     * 若未使用则删除计划并逆向更新执行计划
     * @param id
     * @return
     */
    @Override
    public ExecPlanEntity handleRollback(Long id) {
        DayFillEntity entity = service.selectById(id);
        QueryWrapper wrapper = new QueryWrapper<>();
        wrapper.eq("project_id", entity.getProjectId());
        wrapper.in("bill_state", 1, 3);
        wrapper.ne("id", id);
        wrapper.orderByDesc("fill_version");
        List<DayFillEntity> list = service.list(wrapper);
        DayFillEntity source = list.stream().findFirst().orElse(null);
        if(source != null && entity.getFillVersion() < source.getFillVersion()){
            throw new BusinessException("非该项目下最新生效日进度反馈【" + source.getBillCode() + "】，不允许撤回！");
        }
        List<DayFillDetailEntity> sourceList = new ArrayList<>();
        if(source != null) {
            QueryWrapper<DayFillDetailEntity> ew = new QueryWrapper<DayFillDetailEntity>()
                    .eq("progress_id", source.getId()).orderByAsc("tid");
            sourceList = detailService.list(ew);
        }
        // 执行计划
        ExecPlanEntity data = execService.getOne(new QueryWrapper<ExecPlanEntity>().eq("project_id", entity.getProjectId()));
        if(data == null) {
            throw new BusinessException("未查询到项目下生效总计划！");
        }
        List<ExecPlanDetailEntity> detailList = execDetailService.list(new QueryWrapper<ExecPlanDetailEntity>()
                .eq("progress_id", data.getId()).orderByAsc("tid"));
        if(source == null){
            for(ExecPlanDetailEntity detail : detailList){
                DayFillDetailEntity vo = new DayFillDetailEntity();
                vo.setStructCode(detail.getStructCode());
                vo.setPercentComplete(0);
                vo.setStart(detail.getPlanStart());
                vo.setFinish(detail.getPlanFinish());
                sourceList.add(vo);
            }
        }
        QueryWrapper<DayFillDetailEntity> qw = new QueryWrapper<DayFillDetailEntity>()
                .eq("progress_id", id).orderByAsc("tid");
        List<DayFillDetailEntity> oldList = detailService.list(qw);
        Map<String, DayFillDetailEntity> oldMap = oldList.stream().collect(Collectors.toMap(x->x.getStructCode(), x->x));
        for(DayFillDetailEntity detail : sourceList){
            if(oldMap.containsKey(detail.getStructCode())){
                DayFillDetailEntity old = oldMap.get(detail.getStructCode());
                detail.setFinishNum(ComputeUtil.safeSub(old.getFinishNum(), old.getThisNum()));
                detail.setPercentComplete(NumberUtil.bigDecimalPercent(detail.getFinishNum(), detail.getPlanNum(), 0).intValue());
            }
        }
        // 加锁
        Boolean lock = PlanLockUtil.getLock(entity.getProjectId());
        if (!lock) {
            throw new BusinessException("获取执行计划锁网络异常，请刷新重试！");
        }
        try {
            // 处理子表
            List<ExecPlanDetailEntity> saveList = this.transferSaveList(sourceList, detailList, null);
            if(CollectionUtils.isNotEmpty(saveList)){
                execDetailService.saveOrUpdateBatch(saveList);
            }
            data.setAllList(detailList);

            // 回写执行计划反馈版本号减一
            Integer fillVersion = data.getFillVersion() == null ? 0 : data.getFillVersion() - 1;
            data.setFillVersion(fillVersion);
            data.setFillTime(new Date());
            execService.saveOrUpdate(data);
        } catch (Exception e) {
            logger.error("推送日进度反馈id-{}给执行计划id-{} 异常，", id, data.getId(), e);
            throw new BusinessException(e.getMessage());
        } finally {
            if (lock) {
                PlanLockUtil.releaseLock(entity.getProjectId());
            }
        }
        return data;
    }

    @Override
    public ExecPlanEntity handleUpdate(Long id) {
        return null;
    }
}
