package com.ejianc.business.jlprogress.order.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.jlprogress.order.mapper.SalesMapper;
import com.ejianc.business.jlprogress.order.service.ISalesService;
import com.ejianc.business.jlprogress.order.vo.IncomeContractVO;
import com.ejianc.business.jlprogress.progress.bean.TotalPlanEntity;
import com.ejianc.business.jlprogress.progress.mapper.TotalPlanMapper;
import com.ejianc.business.jlprogress.progress.service.ITotalPlanService;
import com.ejianc.foundation.share.vo.ProjectArchiveVO;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;


/**
 * @Description: 预警控制类
 */
@RestController
@RequestMapping(value = "/warn/")
public class WarnController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String SALES_WARN_NAME = "销售订单未制单";
    private static final String PLAN_WARN_NAME = "计划通知单未制单";
    private static final String YEAR_PLAN_WARN_NAME = "计划分解未制单";

    private static final String SALES_REPORT_URL = "/ejc-jlprogress-frontend/#/sales";
    private static final String PLAN_REPORT_URL = "/ejc-jlprogress-frontend/#/plan";
    private static final String TOTAL_PLAN_URL = "/ejc-jlprogress-frontend/#/totalPlan";
    private static final String YEAR_PLAN_URL = "/ejc-jlprogress-frontend/#/planResolve";


    @Autowired
    private IWarnCenterApi warnCenterApi;

    @Autowired
    private ISalesService salesService;

    @Autowired
    private SalesMapper salesMapper;
    @Autowired
    private TotalPlanMapper totalPlanMapper;

    @Autowired
    private ITotalPlanService totalPlanService;


    /**
     * 已做合同登记 未做销售订单
     * 定时预警
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "salesWarn")
    public CommonResponse<String> salesWarn(@RequestBody JSONObject jsonParam) {
        //1.接收预警参数
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        //2.已做合同，未做销售订单的数据
        List<IncomeContractVO> recordVOList = salesMapper.getNoSales();

        if (CollectionUtils.isEmpty(recordVOList)) {
            logger.error("执行" + SALES_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        logger.info("未做销售订单的数据：{}", JSONObject.toJSONString(recordVOList));

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");

        //3.筛出符合条件的数据发送信息
        for (IncomeContractVO reportVO : recordVOList) {
            String warnLevel = defaultWarnInfo.getString("warnLevel");
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName(reportVO.getProjectName());
            vo.setPcTitle("销售订单");
            vo.setPcUrl(SALES_REPORT_URL);
            vo.setOrgId(reportVO.getOrgId());
            vo.setOrgName(reportVO.getOrgName());
            vo.setTenantId(999999L);
            vo.setWarnLevel(warnLevel);
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

            tmpContent = defaultWarnInfo.getString("earlywarnContent");
            vo.setEarlywarnContent(tmpContent.replaceAll("#contractName#", reportVO.getContractName()));
            resp.add(vo);
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }


    /**
     * 已做合同登记 未做计划通知单
     * 定时预警
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "planWarn")
    public CommonResponse<String> planWarn(@RequestBody JSONObject jsonParam) {
        //1.接收预警参数
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        //2.已做合同，未做销售订单的数据
        List<IncomeContractVO> recordVOList = salesMapper.getNoPlan();

        if (CollectionUtils.isEmpty(recordVOList)) {
            logger.error("执行" + PLAN_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        logger.info("未做计划通知单的数据：{}", JSONObject.toJSONString(recordVOList));

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;

        //3.筛出符合条件的数据发送信息
        for (IncomeContractVO reportVO : recordVOList) {
            String warnLevel = defaultWarnInfo.getString("warnLevel");
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName(reportVO.getProjectName());
            vo.setPcTitle("计划通知单");
            vo.setPcUrl(PLAN_REPORT_URL);
            vo.setOrgId(reportVO.getOrgId());
            vo.setOrgName(reportVO.getOrgName());
            vo.setTenantId(999999L);
            vo.setWarnLevel(warnLevel);
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));



            tmpContent = defaultWarnInfo.getString("earlywarnContent");
            vo.setEarlywarnContent(tmpContent.replaceAll("#contractName#", reportVO.getContractName()));
            resp.add(vo);
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    /**
     * 已做wbs 未做总计划
     * 定时预警
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "totalPlanWarn")
    public CommonResponse<String> totalPlanWarn(@RequestBody JSONObject jsonParam) {
        //1.接收预警参数
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);
        //2.已做合同登记，未做总计划
        List<ProjectArchiveVO> projectList = totalPlanMapper.getNoPlan();

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        //3.筛出符合条件的数据发送信息
        for (ProjectArchiveVO project : projectList) {
            String warnLevel = defaultWarnInfo.getString("warnLevel");
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName(project.getName());
            vo.setPcTitle("项目总计划");
            vo.setPcUrl(TOTAL_PLAN_URL);
            vo.setOrgId(project.getOrgId());
            vo.setOrgName(project.getOrgName());
            vo.setTenantId(999999L);
            vo.setWarnLevel(warnLevel);
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));



            tmpContent = defaultWarnInfo.getString("earlywarnContent");
            vo.setEarlywarnContent(tmpContent.replaceAll("#projectName#", project.getName()));
            resp.add(vo);
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    /**
     * 已做项目总计划 未做计划分解
     * 定时预警
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "yearPlanWarn")
    public CommonResponse<String> yearPlanWarn(@RequestBody JSONObject jsonParam) {
        //1.接收预警参数
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);
        //2.已做总计划 未作计划分解
        List<TotalPlanEntity> projectList = totalPlanService.getNoTotalPlan();
        if (CollectionUtils.isEmpty(projectList)) {
            logger.error("执行" + YEAR_PLAN_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        logger.info("未做计划分解的数据：{}", JSONObject.toJSONString(projectList));

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        //3.筛出符合条件的数据发送信息
        for (TotalPlanEntity totalPlanEntity : projectList) {
            String warnLevel = defaultWarnInfo.getString("warnLevel");
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName(totalPlanEntity.getProjectName());
            vo.setPcTitle("计划分解");
            vo.setPcUrl(YEAR_PLAN_URL);
            vo.setOrgId(totalPlanEntity.getParentOrgId());
            vo.setOrgName(totalPlanEntity.getParentOrgName());
            vo.setTenantId(999999L);
            vo.setWarnLevel(warnLevel);
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));
            tmpContent = defaultWarnInfo.getString("earlywarnContent");
            vo.setEarlywarnContent(tmpContent.replaceAll("#projectName#", totalPlanEntity.getProjectName()));
            resp.add(vo);
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    private CommonResponse<String> sendWarnToTask(List<EarlyWarnTransVO> warnList, Long warnId) {
        logger.info("发送任务：{}", JSONObject.toJSONString(warnList));
        CommonResponse<String> warnResp = warnCenterApi.sendToWarnCenter(warnList, warnId);
        if (!warnResp.isSuccess()) {
            logger.error("回调预警任务服务失败，{}", warnResp.getMsg());
            return CommonResponse.error("回调预警任务服务失败" + warnResp.getMsg());
        }

        return CommonResponse.success("任务接受处理成功！");
    }

    private BigDecimal getBigDecimal(Object obj) {
        return null == obj ? BigDecimal.ZERO : new BigDecimal(obj.toString());
    }

}
