package com.ejianc.business.labor.service.impl;

import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.labor.service.IAmapService;
import com.ejianc.business.labor.vo.DistanceVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.util.HttpTookit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 高德地图接口实现
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@Service(value = "amapService")
public class AmapServiceImpl implements IAmapService {
	private static final String DISTANCE = "https://restapi.amap.com/v3/distance?parameters";
	private static final String REGEO = "https://restapi.amap.com/v3/geocode/regeo?parameters";
	private static final String REGEO_TDT = "http://api.tianditu.gov.cn/geocoder?postStr={'lon':lonStr,'lat':latStr,'ver':1}&type=geocode&tk=";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Value(value = "${amap.key}")
	private String key;
	@Value(value = "${tdt.key}")
	private String key_tdt;

	// 益企云-坐标测距	华康天地图 appkey：39c21ec36cf18710fea6887afe255328
	/**
	 * 默认地球半径
	 */
	private static double EARTH_RADIUS = 6371000;//赤道半径(单位m)

	/**
	 * 转化为弧度(rad)
	 * */
	private static double rad(double d) {
		return d * Math.PI / 180.0;
	}

	/**
	 * @param lon1 第一点的经度
	 * @param lat1 第一点的纬度
	 * @param lon2 第二点的经度
	 * @param lat2 第二点的纬度
	 * @return 返回的距离，单位m
	 * */
	public static double GetDistance(double lon1,double lat1,double lon2, double lat2) {
		double radLat1 = rad(lat1);
		double radLat2 = rad(lat2);
		double a = radLat1 - radLat2;
		double b = rad(lon1) - rad(lon2);
		double s = 2 * Math.asin(Math.sqrt(Math.pow(Math.sin(a / 2), 2) + Math.cos(radLat1) * Math.cos(radLat2) * Math.pow(Math.sin(b / 2), 2)));
		s = s * EARTH_RADIUS;
		s = (double) Math.round(s * 10000) / 10000;
		return s;
	}

	/**
	 * 距离测量
	 *
	 * @param origins     出发点：支持100个坐标对，坐标对见用“| ”分隔；经度和纬度用","分隔
	 * @param destination 目的地： lon，lat（经度，纬度）， “,”分割"，如117.500244, 40.417801经纬度小数点不超过6位
	 * @param type        路径计算的方式和方法：0：直线距离 1：驾车导航距离（仅支持国内坐标）。必须指出，当为1时会考虑路况，故在不同时间请求返回结果可能不同。此策略和驾车路径规划接口的 strategy=0策略基本一致，策略为“ 速度优先，此路线不一定距离最短 ”由于算法差异，无法保证距离测量结果与路径规划结果完全一致。若需要实现高德地图客户端效果，可以考虑使用驾车路径规划接口 3：步行规划距离（仅支持5km之间的距离）
	 *
	 * @return {@link List}<{@link DistanceVO}>
	 */
	@Override
	public List<DistanceVO> distance(String origins, String destination, String type) {
		Assert.hasText(origins, "出发点不能为空！");
		Assert.hasText(destination, "目的地不能为空！");
		Assert.hasText(type, "路径计算的方式和方法不能为空！");

		logger.info("距离测量，入参：origins--{}，destination--{}，type--{}", origins, destination, type);
		String[] originsLonLat = origins.split("\\|");
		List<DistanceVO> distanceVOList = new ArrayList<>();
		for (int i = 0; i < originsLonLat.length; i++) {
			String s = originsLonLat[i];
			String[] lonLat = s.split(",");
			DistanceVO distanceVO = new DistanceVO();
			distanceVO.setOriginId((i+1)+"");
			distanceVO.setDistance(String.valueOf(GetDistance(Double.parseDouble(lonLat[0]), Double.parseDouble(lonLat[1]), Double.parseDouble(destination.split(",")[0]), Double.parseDouble(destination.split(",")[1]))));
			distanceVOList.add(distanceVO);
		}
		return distanceVOList;
	}


	/**
	 * 逆地理编码
	 *
	 * @param location 经纬度坐标
	 *
	 * @return {@link JSONObject}
	 */
	@Override
	public JSONObject regeo(String location) {
		Assert.hasText(location, "经纬度坐标不能为空！");

		logger.info("逆地理编码，入参：location--{}", location);
		String url = REGEO_TDT.replace("lonStr", location.split(",")[0]);
		url = url.replace("latStr", location.split(",")[1]);
		url = url+key_tdt;

		Map<String, String> params = new HashMap<>();
		params.put("postStr", "{'lon':lonStr,'lat':latStr,'ver':1}".replace("lonStr", location.split(",")[0]).replace("latStr", location.split(",")[1]));
		params.put("type", "geocode");
		params.put("tk", key_tdt);

        String response = null;
		try {
			response = HttpTookit.get("http://api.tianditu.gov.cn/geocoder",params,new HashMap<>());
			logger.info("请求天地图 地址-{},参数：{}，响应：{}",url,JSONObject.toJSONString(params),response);
		} catch (GeneralSecurityException e) {
			logger.error("请求天地图失败！",e);
			throw new BusinessException("请求天地图失败！");
		} catch (IOException e) {
			logger.error("请求天地图失败！",e);
			throw new BusinessException("请求天地图失败！");
		}
		if (StringUtils.isEmpty(response)) {
			throw new BusinessException("请求天地图失败！");
		}
		JSONObject jsonObject = JSON.parseObject(response);
		String status = jsonObject.getString("status");
		if (!"0".equals(status)) {
			throw new BusinessException("逆地理编码失败，失败原因：" + jsonObject.getString("msg"));
		}
		return (JSONObject) jsonObject.get("result");
	}
}
