package com.ejianc.business.labor.service.impl;

import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.labor.service.IAmapService;
import com.ejianc.business.labor.vo.DistanceVO;
import com.ejianc.framework.core.exception.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.HashMap;
import java.util.List;

/**
 * 高德地图接口实现
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@Service(value = "amapService")
public class AmapServiceImpl implements IAmapService {
	private static final String DISTANCE = "https://restapi.amap.com/v3/distance?parameters";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Value(value = "${amap.key}")
	private String key;

	/**
	 * 距离测量
	 *
	 * @param origins     出发点：支持100个坐标对，坐标对见用“| ”分隔；经度和纬度用","分隔
	 * @param destination 目的地： lon，lat（经度，纬度）， “,”分割"，如117.500244, 40.417801经纬度小数点不超过6位
	 * @param type        路径计算的方式和方法：0：直线距离 1：驾车导航距离（仅支持国内坐标）。必须指出，当为1时会考虑路况，故在不同时间请求返回结果可能不同。此策略和驾车路径规划接口的 strategy=0策略基本一致，策略为“ 速度优先，此路线不一定距离最短 ”由于算法差异，无法保证距离测量结果与路径规划结果完全一致。若需要实现高德地图客户端效果，可以考虑使用驾车路径规划接口 3：步行规划距离（仅支持5km之间的距离）
	 *
	 * @return {@link List}<{@link DistanceVO}>
	 */
	@Override
	public List<DistanceVO> distance(String origins, String destination, String type) {
		Assert.hasText(origins, "出发点不能为空！");
		Assert.hasText(destination, "目的地不能为空！");
		Assert.hasText(type, "路径计算的方式和方法不能为空！");

		logger.info("距离测量，入参：origins--{}，destination--{}，type--{}", origins, destination, type);
		HashMap<String, Object> paramMap = new HashMap<>();
		paramMap.put("key", key);
		paramMap.put("origins", origins);
		paramMap.put("destination", destination);
		paramMap.put("type", type);


		HttpResponse response = HttpUtil.createGet(DISTANCE)
				.form(paramMap)
				.execute();
		if (!response.isOk()) {
			throw new BusinessException("请求高德地图失败！");
		}
		JSONObject jsonObject = JSON.parseObject(response.body());
		String status = jsonObject.getString("status");
		if ("0".equals(status)) {
			throw new BusinessException("距离测量失败，失败原因：" + jsonObject.getString("info"));
		}
		return JSON.parseArray(JSON.toJSONString(jsonObject.get("results")), DistanceVO.class);
	}
}
