package com.ejianc.business.labor.service.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.labor.bean.AttendanceBindEntity;
import com.ejianc.business.labor.bean.AttendanceEntity;
import com.ejianc.business.labor.bean.AttendanceLogEntity;
import com.ejianc.business.labor.bean.WorkRecordEntity;
import com.ejianc.business.labor.common.dtoMapper.WorkRecordMapper;
import com.ejianc.business.labor.mapper.AttendanceLogMapper;
import com.ejianc.business.labor.service.*;
import com.ejianc.business.labor.utils.MockLogin;
import com.ejianc.business.labor.vo.AttendanceLogVO;
import com.ejianc.business.labor.vo.DistanceVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.usercenter.api.IThirdSystemApi;
import com.ejianc.foundation.usercenter.vo.ThirdSystemVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.EnvironmentTools;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.util.*;
import java.util.concurrent.TimeUnit;


/**
 * 考勤日志
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@Service("attendanceLogService")
public class AttendanceLogServiceImpl extends BaseServiceImpl<AttendanceLogMapper, AttendanceLogEntity> implements IAttendanceLogService {
	private static final String WEIXIN_OAUTH_2_URL = "https://api.weixin.qq.com/sns/oauth2/access_token"; // 获取access_token
	private static final String DEFAULT_TEMPLATE_CODE = "SMS_195335075"; // 默认的登录验证模板
	private static final String CODE = "Weixin";
	private static final String BILL_TYPE = "BT202211000002";
	private static final String START_WORK = "上班打卡";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Resource
	private EnvironmentTools environmentTools;
	@Resource
	private IThirdSystemApi thirdSystemApi;
	@Resource
	private IAmapService amapService;
	@Resource
	private RedisTemplate<String, Object> redisTemplate;
	@Resource
	private MockLogin mockLogin;
	@Resource
	private IAttendanceBindService attendanceBindService;
	@Resource
	private IWorkRecordService workRecordService;
	@Resource
	private IAttachmentApi attachmentApi;
	@Resource
	private IAttendanceService attendanceService;

	/**
	 * 登录
	 *
	 * @param tenantId     租户id
	 * @param projectId    项目id
	 * @param phone        手机号
	 * @param templateCode 短信模版
	 * @param validate     验证码
	 * @param openid       openid
	 */
	public void login(Long tenantId, Long projectId, String phone, String templateCode, String validate, String openid) {
		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");
		Assert.hasText(validate, "验证码不能为空！");
		Assert.hasText(openid, "openid不能为空！");

		// 1、校验验证码
		String checkMessageUrl = environmentTools.getBaseHost() + "/ejc-message-web/no_auth/sms/checkMessage";
		HashMap<String, Object> param = new HashMap<>();
		if (StringUtils.isNoneBlank(templateCode)) {
			param.put("templateCode", templateCode);
		} else {
			param.put("templateCode", DEFAULT_TEMPLATE_CODE);
		}
		param.put("phone", phone);
		param.put("validate", validate);

		HttpResponse response = HttpUtil.createPost(checkMessageUrl)
				.form(param).execute();
		if (!response.isOk()) {
			throw new BusinessException("校验验证码失败！");
		}

		// 2、绑定租户手机号和openid：未绑定的绑定，已绑定需要更新绑定手机号
		String checkBind = checkBind(tenantId, openid);
		if (StringUtils.isBlank(checkBind)) {
			AttendanceBindEntity attendanceBindEntity = new AttendanceBindEntity();
			attendanceBindEntity.setTenantId(tenantId);
			attendanceBindEntity.setPhone(phone);
			attendanceBindEntity.setOpenid(openid);
			attendanceBindService.saveOrUpdateNoES(attendanceBindEntity);
		} else {
			LambdaUpdateWrapper<AttendanceBindEntity> lambdaUpdate = Wrappers.lambdaUpdate();
			lambdaUpdate.eq(AttendanceBindEntity::getOpenid, openid);
			lambdaUpdate.set(AttendanceBindEntity::getPhone, phone);
			attendanceBindService.update(lambdaUpdate);
		}
	}

	/**
	 * 获取openid
	 *
	 * @param code 临时登录凭证
	 *
	 * @return {@link JSONObject}
	 */
	@Override
	public String getOpenId(String code) {
		Assert.hasText(code, "临时登录凭证不能为空！");

		// 1、查询微信公众号配置信息
		CommonResponse<ThirdSystemVO> thirdSystemResponse = thirdSystemApi.getOneByCode(CODE);
		if (!thirdSystemResponse.isSuccess()) {
			throw new BusinessException("查询微信小程序配置信息失败，失败原因：" + thirdSystemResponse.getMsg());
		}
		ThirdSystemVO thirdSystemVO = thirdSystemResponse.getData();
		String configInfo = thirdSystemVO.getConfigInfo();
		JSONObject configJson = JSON.parseObject(configInfo);
		String appId = configJson.getString("appid");
		String secret = configJson.getString("secret");

		// 2、公众号
		HashMap<String, Object> paramMap = new HashMap<>();
		paramMap.put("appid", appId);
		paramMap.put("secret", secret);
		paramMap.put("code", code);
		paramMap.put("grant_type", "authorization_code");

		HttpResponse response = HttpUtil.createGet(WEIXIN_OAUTH_2_URL)
				.form(paramMap).execute();
		if (!response.isOk()) {
			throw new BusinessException("请求失败！");
		}
		JSONObject jsonObject = JSON.parseObject(response.body());
		if (jsonObject.containsKey("errcode")) {
			throw new BusinessException("获取openid失败，失败原因：" + jsonObject.getString("errmsg"));
		}
		return jsonObject.getString("openid");
	}


	/**
	 * 获取微信配置数据
	 *
	 * @param reqFrom 请求来自客户端
	 * @param url     url
	 *
	 * @return {@link JSONObject}
	 */
	@Override
	public JSONObject getWeiXinConfigData(String reqFrom, String url) {
		Assert.hasText(reqFrom, "客户端来源不能为空！");
		Assert.hasText(url, "url地址不能为空！");

		logger.info("获取微信配置数据: req_from = {}，url = {}", reqFrom, url);
		JSONObject res = new JSONObject();
		CommonResponse<ThirdSystemVO> thirdSystemResponse = thirdSystemApi.getOneByCode(reqFrom);
		if (!thirdSystemResponse.isSuccess()) {
			throw new BusinessException("获取微信配置数据失败，失败原因：" + thirdSystemResponse.getMsg());
		}
		ThirdSystemVO thirdSystemVO = thirdSystemResponse.getData();
		if (thirdSystemVO != null) {
			String configInfo = thirdSystemVO.getConfigInfo();
			JSONObject configJson = JSON.parseObject(configInfo);
			String corpAppId = configJson.getString("corp_appId");
			String corpSecret = configJson.getString("corp_secret");
			String accessToken = null;
			String jsapi_ticket = null;
			if ("weixinminprogram".equals(reqFrom)) {
				accessToken = this.getMiniProgramAccessToken(corpAppId, corpSecret);
				String key = url.replace("://", "_").replace("/", "_");
				jsapi_ticket = this.getMiniProgramJsapiTicket(key, accessToken);
			}
			String noncestr = UUID.randomUUID().toString();
			String timestamp = (System.currentTimeMillis() / 1000) + "";
			String signature = "";
			String string1 = "jsapi_ticket=" + jsapi_ticket +
					"&noncestr=" + noncestr +
					"&timestamp=" + timestamp +
					"&url=" + url;
			logger.info("string1 ====> " + string1);
			try {
				MessageDigest crypt = MessageDigest.getInstance("SHA-1");
				crypt.reset();
				crypt.update(string1.getBytes(StandardCharsets.UTF_8));
				signature = byteToHex(crypt.digest());
				System.out.println("signature ====> " + signature);
				res.put("url", url);
				res.put("jsapi_ticket", jsapi_ticket);
				res.put("noncestr", noncestr);
				res.put("timestamp", timestamp);
				res.put("signature", signature);
				res.put("appid", corpAppId);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return res;
	}


	/**
	 * 字节数组转换为十六进制字符串
	 *
	 * @param hash 字节数组
	 *
	 * @return 十六进制字符串
	 */
	private static String byteToHex(final byte[] hash) {
		Formatter formatter = new Formatter();
		for (byte b : hash) {
			formatter.format("%02x", b);
		}
		String result = formatter.toString();
		formatter.close();
		return result;
	}

	/**
	 * 获取小程序访问令牌
	 *
	 * @param appId     appid
	 * @param appSecret app密钥
	 *
	 * @return {@link String}
	 */
	public String getMiniProgramAccessToken(String appId, String appSecret) {
		String key = appId + "_" + appSecret;
		String accessToken = (String) redisTemplate.opsForValue().get(key);
		logger.info("redis----MiniProgramaccessToken=" + accessToken);
		System.out.println("redis----MiniProgramaccessToken=" + accessToken);
		if (StringUtils.isBlank(accessToken)) {
			Map<String, String> param = new HashMap<>();
			String accessResult = null;
			param.put("grant_type", "client_credential");
			param.put("appid", appId);
			param.put("secret", appSecret);
			try {
				accessResult = ReferHttpClientUtils.get("https://api.weixin.qq.com/cgi-bin/token", param, new HashMap<>());
			} catch (GeneralSecurityException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}
			logger.info("MiniProgramaccessTokenaccessResult=" + accessResult);
			System.out.println("MiniProgramaccessTokenaccessResult=" + accessResult);
			if (StringUtils.isNotBlank(accessResult)) {
				JSONObject accessJson = JSON.parseObject(accessResult);
				accessToken = accessJson.getString("access_token");
				String errmsg = accessJson.getString("errcode");
				if ("0".equals(errmsg) && StringUtils.isNotBlank(accessToken)) {
					redisTemplate.opsForValue().set(key, accessToken, 7000, TimeUnit.SECONDS);
				}
			}
		}
		return accessToken;
	}


	/**
	 * 获取小程序Ticket
	 *
	 * @param corpAppId   appid
	 * @param accessToken 访问令牌
	 *
	 * @return {@link String}
	 */
	public String getMiniProgramJsapiTicket(String corpAppId, String accessToken) {
		String key = corpAppId + "_jsapi_ticket";
		String jsapiTicket = (String) redisTemplate.opsForValue().get(key);
		logger.info("redis----jsapiTicket=" + jsapiTicket);
		System.out.println("redis----jsapiTicket=" + jsapiTicket);
		if (StringUtils.isBlank(jsapiTicket)) {
			Map<String, String> param = new HashMap<>();
			String accessResult = null;
			param.put("access_token", accessToken);
			param.put("type", "jsapi");
			try {
				accessResult = ReferHttpClientUtils.get("https://api.weixin.qq.com/cgi-bin/ticket/getticket", param, new HashMap<>());
			} catch (GeneralSecurityException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}
			System.out.println("accessResult=" + accessResult);
			logger.info("accessResult=" + accessResult);
			if (StringUtils.isNotBlank(accessResult)) {
				JSONObject accessJson = JSON.parseObject(accessResult);
				jsapiTicket = accessJson.getString("ticket");
				String errmsg = accessJson.getString("errcode");
				if ("0".equals(errmsg) && StringUtils.isNotBlank(jsapiTicket)) {
					redisTemplate.opsForValue().set(key, jsapiTicket, 7000, TimeUnit.SECONDS);
				}
			}
		}
		return jsapiTicket;
	}

	/**
	 * 租户手机号绑定openid校验
	 *
	 * @param tenantId 租户id
	 * @param openid   openid
	 *
	 * @return {@link String}
	 */
	@Override
	public String checkBind(Long tenantId, String openid) {
		LambdaQueryWrapper<AttendanceBindEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(AttendanceBindEntity::getTenantId, tenantId);
		lambdaQuery.eq(AttendanceBindEntity::getOpenid, openid);
		AttendanceBindEntity attendanceBindEntity = attendanceBindService.getOne(lambdaQuery);
		return attendanceBindEntity.getPhone();
	}


	/**
	 * 给指定手机号发送验证码或者短信
	 *
	 * @param tenantId     租户id
	 * @param projectId    项目id
	 * @param phone        手机号
	 * @param templateCode 短信模板编码
	 * @param signName     短信签名
	 *
	 * @return {@link String}
	 *
	 * @throws Exception 异常
	 */
	@Override
	public String sendMsg(Long tenantId, Long projectId, String phone, String templateCode, String signName) throws Exception {


		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");
		Assert.hasText(templateCode, "短信模板编码不能为空！");
		Assert.hasText(signName, "短信签名不能为空！");

		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getTenantId, tenantId);
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId);
		lambdaQuery.eq(WorkRecordEntity::getPhone, phone);
		WorkRecordEntity workRecordEntity = workRecordService.getOne(lambdaQuery);
		if (workRecordEntity == null) {
			throw new BusinessException("手机号在平台租户花名册中不存在！");
		}

		JSONObject messageParam = new JSONObject();
		messageParam.put("phone", phone);
		if (StringUtils.isBlank(templateCode)) {
			messageParam.put("templateCode", DEFAULT_TEMPLATE_CODE);
		} else {
			messageParam.put("templateCode", templateCode);
		}
		if (StringUtils.isNotBlank(signName)) {
			messageParam.put("signName", signName);
		}
		String messageUrl = environmentTools.getBaseHost() + "/ejc-message-web/no_auth/sms/sendMessage";
		String responseStr = HttpTookit.postByJson(messageUrl, JSON.toJSONString(messageParam));
		logger.info("向手机号[{}]发送验证码结果：[{}]", phone, responseStr);

		CommonResponse<String> response = JSON.parseObject(responseStr, CommonResponse.class);
		if (!response.isSuccess()) {
			throw new BusinessException("短信发送失败");
		}
		return "短信发送成功！";
	}

	/**
	 * 距离测量
	 *
	 * @param projectId   项目id
	 * @param destination 目的地
	 */
	@Override
	public void checkDistance(String projectId, String destination) {
		Assert.hasText(projectId, "项目id不能为空！");
		Assert.hasText(destination, "目的地不能为空！");

		List<DistanceVO> distance = amapService.distance("116.481028,39.989643|114.481028,39.989643|115.481028,39.989643", "114.465302,40.004717", "0");
		logger.info("距离测量--{}", JSON.toJSONString(distance, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

	}

	/**
	 * 校验人脸信息
	 *
	 * @param tenantId  租户id
	 * @param projectId 项目id
	 * @param phone     手机号
	 *
	 * @return {@link Boolean}
	 */
	@Override
	public Boolean checkFace(Long tenantId, Long projectId, String phone) {
		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");

		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getTenantId, tenantId);
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId);
		lambdaQuery.eq(WorkRecordEntity::getPhone, phone);
		WorkRecordEntity workRecordEntity = workRecordService.getOne(lambdaQuery);
		if (workRecordEntity == null) {
			throw new BusinessException("校验人脸信息失败！");
		}
		CommonResponse<List<AttachmentVO>> response = attachmentApi.queryListBySourceId(workRecordEntity.getId(), BILL_TYPE, "face", null);
		if (!response.isSuccess()) {
			throw new BusinessException("获取人脸信息失败！");
		}
		return CollectionUtils.isNotEmpty(response.getData());
	}


	/**
	 * 考勤打卡
	 *
	 * @param tenantId       租户id
	 * @param projectId      项目id
	 * @param phone          手机号
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link AttendanceLogVO}
	 */
	@Override
	public AttendanceLogVO punchCard(Long tenantId, Long projectId, String phone, String punchCardType, String punchCardPlace, Integer remoteFlag) {
		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");
		Assert.hasText(punchCardType, "打卡类型不能为空！");
		Assert.hasText(punchCardPlace, "打卡地点不能为空！");
		Assert.notNull(remoteFlag, "是否是异地打卡不能为空！");

		// 1、新增考勤打卡日志
		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getTenantId, tenantId);
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId);
		lambdaQuery.eq(WorkRecordEntity::getPhone, phone);
		WorkRecordEntity workRecordEntity = workRecordService.getOne(lambdaQuery);
		if (workRecordEntity == null) {
			throw new BusinessException("考勤打卡失败，不存在该人员！");
		}

		AttendanceLogEntity attendanceLogEntity = WorkRecordMapper.INSTANCE.transformAttendanceLogEntity(workRecordEntity);

		attendanceLogEntity.setPunchCardDate(new Date());
		if (START_WORK.equals(punchCardType)) {
			attendanceLogEntity.setStartWorkTime(new Date());
			attendanceLogEntity.setStartWorkPlace(punchCardPlace);
		} else {
			attendanceLogEntity.setEndWorkTime(new Date());
			attendanceLogEntity.setEndWorkPlace(punchCardPlace);
		}
		if (remoteFlag == 0) {
			attendanceLogEntity.setAttendanceStatus("正常");
		} else {
			attendanceLogEntity.setAttendanceStatus("异常");
		}
		attendanceLogEntity.setPunchCardType(punchCardType);
		super.saveOrUpdate(attendanceLogEntity, false);

		// 2、查询考勤打卡记录
		LambdaQueryWrapper<AttendanceEntity> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(AttendanceEntity::getTenantId, tenantId);
		wrapper.eq(AttendanceEntity::getProjectId, projectId);
		wrapper.eq(AttendanceEntity::getPhone, phone);
		AttendanceEntity attendanceEntity = attendanceService.getOne(wrapper);
		if (attendanceEntity == null) {
			AttendanceEntity entity = com.ejianc.business.labor.common.dtoMapper.AttendanceLogMapper.INSTANCE.transformAttendanceEntity(attendanceLogEntity);
			if (entity.getStartWorkTime() != null && entity.getEndWorkTime() != null) {
				entity.setDuration(DateUtil.formatBetween(entity.getStartWorkTime(), entity.getEndWorkTime()));
			} else {
				entity.setDuration(null);
			}
			attendanceService.saveOrUpdate(entity, false);
		} else {
			if (START_WORK.equals(punchCardType)) {
				attendanceEntity.setStartWorkTime(attendanceLogEntity.getStartWorkTime());
				attendanceEntity.setStartWorkPlace(attendanceLogEntity.getStartWorkPlace());
			} else {
				attendanceEntity.setEndWorkTime(attendanceLogEntity.getEndWorkTime());
				attendanceEntity.setEndWorkPlace(attendanceLogEntity.getEndWorkPlace());
			}
			if (attendanceEntity.getStartWorkTime() != null && attendanceEntity.getEndWorkTime() != null) {
				attendanceEntity.setDuration(DateUtil.formatBetween(attendanceEntity.getStartWorkTime(), attendanceEntity.getEndWorkTime()));
			} else {
				attendanceEntity.setDuration(null);
			}
			attendanceEntity.setAttendanceStatus(attendanceLogEntity.getAttendanceStatus());
			attendanceEntity.setRemoteFlag(attendanceLogEntity.getRemoteFlag());
			attendanceService.saveOrUpdate(attendanceEntity, false);
		}

		return BeanMapper.map(attendanceLogEntity, AttendanceLogVO.class);
	}
}
