package com.ejianc.business.labor.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.labor.bean.AttendanceLogEntity;
import com.ejianc.business.labor.service.IAmapService;
import com.ejianc.business.labor.service.IAttendanceLogService;
import com.ejianc.business.labor.vo.AttendanceLogVO;
import com.ejianc.business.labor.vo.WorkRecordVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.io.Serializable;
import java.util.List;

import static com.ejianc.framework.skeleton.template.BaseServiceImpl.changeToQueryWrapper;


/**
 * 考勤日志
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@RestController
@RequestMapping("attendanceLog")
public class AttendanceLogController implements Serializable {
	private static final long serialVersionUID = 5204994771029686422L;
	private static final String BILL_CODE = "Invoice_Open_Apply_Code";//此处需要根据实际修改
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Resource
	private IBillTypeApi billTypeApi;
	@Resource
	private IBillCodeApi billCodeApi;
	@Resource
	private IOrgApi iOrgApi;
	@Resource
	private IAttendanceLogService service;
	@Autowired
	private SessionManager sessionManager;
	@Resource
	private IAmapService amapService;


	/**
	 * 获取微信配置数据
	 *
	 * @param reqFrom 请求来自客户端
	 * @param url     url
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "getWeiXinConfigData")
	public CommonResponse<JSONObject> getWeiXinConfigData(@RequestParam("req_from") String reqFrom,
	                                                      @RequestParam("url") String url) {
		return CommonResponse.success(service.getWeiXinConfigData(reqFrom, url));
	}


	/**
	 * 获取openid
	 *
	 * @param code 临时登录凭证
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "getOpenId")
	public CommonResponse<String> getOpenId(@RequestParam(value = "code") String code) {
		return CommonResponse.success("获取openid成功！", service.getOpenId(code));
	}


	/**
	 * 租户手机号绑定openid校验
	 *
	 * @param tenantId 租户id
	 * @param openid   openid
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@GetMapping(value = "checkBind")
	public CommonResponse<String> checkBind(@RequestParam(value = "tenantId") Long tenantId,
	                                        @RequestParam(value = "openid") String openid) {
		return CommonResponse.success("租户手机号绑定openid校验成功！", service.checkBind(tenantId, openid));
	}


	/**
	 * 发送验证码
	 *
	 * @param tenantId     租户id
	 * @param projectId    项目id
	 * @param phone        手机号
	 * @param templateCode 短信模板编码
	 * @param signName     短信签名
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 *
	 * @throws Exception 异常
	 */
	@GetMapping(value = "sendMessage")
	public CommonResponse<String> sendMessage(@RequestParam(value = "tenantId") Long tenantId,
	                                          @RequestParam(value = "projectId") Long projectId,
	                                          @RequestParam(value = "phone") String phone,
	                                          @RequestParam(value = "templateCode", required = false) String templateCode,
	                                          @RequestParam(value = "signName") String signName) throws Exception {
		return CommonResponse.success(service.sendMsg(tenantId, projectId, phone, templateCode, signName));
	}


	/**
	 * 登录
	 *
	 * @param tenantId     租户id
	 * @param projectId    项目id
	 * @param phone        手机号
	 * @param templateCode 短信模版
	 * @param validate     验证码
	 * @param openid       openid
	 *
	 * @return {@link CommonResponse}<{@link Void}>
	 */
	@GetMapping(value = "login")
	public CommonResponse<Void> login(@RequestParam(value = "tenantId") Long tenantId,
	                                  @RequestParam(value = "projectId") Long projectId,
	                                  @RequestParam(value = "phone") String phone,
	                                  @RequestParam(value = "templateCode") String templateCode,
	                                  @RequestParam(value = "validate") String validate,
	                                  @RequestParam(value = "openid") String openid) {
		service.login(tenantId, projectId, phone, templateCode, validate, openid);
		return CommonResponse.success("登录成功！");
	}


	/**
	 * 校验人脸
	 *
	 * @param jsonObject json对象
	 *
	 * @return {@link CommonResponse}<{@link WorkRecordVO}>
	 */
	@PostMapping(value = "checkFace")
	public CommonResponse<WorkRecordVO> checkFace(@RequestBody JSONObject jsonObject) {
		Long tenantId = jsonObject.getLong("tenantId");
		Long projectId = jsonObject.getLong("projectId");
		String phone = jsonObject.getString("phone");
		String imgBase64A = jsonObject.getString("imgBase64A");
		return service.checkFace(tenantId, projectId, phone, imgBase64A);
	}


	/**
	 * 校验考勤范围
	 *
	 * @param worktimeSetId 考勤设置id
	 * @param destination   当前位置经纬度
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "checkDistance")
	public CommonResponse<JSONObject> checkDistance(@RequestParam(value = "worktimeSetId") String worktimeSetId,
	                                                @RequestParam(value = "destination") String destination) {
		return service.checkDistance(worktimeSetId, destination);
	}


	/**
	 * 校验项目打卡权限
	 *
	 * @param tenantId  租户id
	 * @param projectId 项目id
	 * @param phone     手机号
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@GetMapping(value = "checkPunchCardAuth")
	public CommonResponse<String> checkPunchCardAuth(@RequestParam(value = "tenantId") Long tenantId,
	                                                 @RequestParam(value = "projectId") Long projectId,
	                                                 @RequestParam(value = "phone") String phone) {
		service.checkPunchCardAuth(tenantId, projectId, phone);
		return CommonResponse.success("项目打卡权限校验成功！");
	}


	/**
	 * 校验已存在的打卡记录
	 *
	 * @param tenantId      租户id
	 * @param projectId     项目id
	 * @param phone         手机号
	 * @param punchCardType 打卡类型：上班打卡、下班打卡
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link AttendanceLogVO}>>
	 */
	@GetMapping(value = "checkExistedLog")
	public CommonResponse<List<AttendanceLogVO>> checkExistedLog(@RequestParam(value = "tenantId") Long tenantId,
	                                                             @RequestParam(value = "projectId") Long projectId,
	                                                             @RequestParam(value = "phone") String phone,
	                                                             @RequestParam(value = "punchCardType") String punchCardType) {
		return service.checkExistedLog(tenantId, projectId, phone, punchCardType);
	}


	/**
	 * 考勤打卡
	 *
	 * @param tenantId       租户id
	 * @param projectId      项目id
	 * @param phone          手机号
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link CommonResponse}<{@link AttendanceLogVO}>
	 */
	@GetMapping(value = "/punchCard")
	public CommonResponse<AttendanceLogVO> punchCard(@RequestParam(value = "tenantId") Long tenantId,
	                                                 @RequestParam(value = "projectId") Long projectId,
	                                                 @RequestParam(value = "phone") String phone,
	                                                 @RequestParam(value = "punchCardType") String punchCardType,
	                                                 @RequestParam(value = "punchCardPlace") String punchCardPlace,
	                                                 @RequestParam(value = "remoteFlag") Integer remoteFlag) {
		return CommonResponse.success("考勤打卡成功！", service.punchCard(tenantId, projectId, phone, punchCardType, punchCardPlace, remoteFlag));
	}


	/**
	 * 查询详情
	 *
	 * @param id id
	 *
	 * @return {@link CommonResponse}<{@link AttendanceLogVO}>
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<AttendanceLogVO> queryDetail(Long id) {
		AttendanceLogEntity entity = service.selectById(id);
		AttendanceLogVO vo = BeanMapper.map(entity, AttendanceLogVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}


	/**
	 * 分页查询
	 *
	 * @param param 参数
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link AttendanceLogVO}>>
	 */
	@PostMapping(value = "/queryPage")
	public CommonResponse<IPage<AttendanceLogVO>> queryPage(@RequestBody QueryParam param) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("startWorkPlace");
		fuzzyFields.add("endWorkPlace");

		Object punchCardDate = param.getParams().get("punchCardDate").getValue();
		param.getParams().remove("punchCardDate");

		QueryWrapper<AttendanceLogEntity> queryWrapper = changeToQueryWrapper(param);
		queryWrapper.apply("date_format(punch_card_date,'%Y-%m') = {0}", punchCardDate);

		Page<AttendanceLogEntity> p = new Page<>(param.getPageIndex(), param.getPageSize());
		IPage<AttendanceLogEntity> page = service.page(p, queryWrapper);

		IPage<AttendanceLogVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), AttendanceLogVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}


	/**
	 * 参照
	 *
	 * @param pageNumber   页码
	 * @param pageSize     页面大小
	 * @param condition    条件
	 * @param searchObject 搜索对象
	 * @param searchText   搜索文本
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link AttendanceLogVO}>>
	 */
	@GetMapping(value = "/refAttendanceLogData")
	public CommonResponse<IPage<AttendanceLogVO>> refAttendanceLogData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
	                                                                   String condition,
	                                                                   String searchObject,
	                                                                   String searchText) {
		QueryParam param = new QueryParam();
		param.setPageSize(pageSize);
		param.setPageIndex(pageNumber);
		param.setSearchText(searchText);
		param.setSearchObject(searchObject);
		/** 租户隔离 */
		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		if (StringUtils.isNotEmpty(condition)) {
			/** 处理condition */
			JSONObject _con = JSONObject.parseObject(condition);
		}

		IPage<AttendanceLogEntity> page = service.queryPage(param, false);
		IPage<AttendanceLogVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), AttendanceLogVO.class));

		return CommonResponse.success("查询参照数据成功！", pageData);
	}
}
