package com.ejianc.business.labor.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.labor.service.IFaceService;
import com.ejianc.business.labor.vo.FaceInfoVO;
import com.ejianc.foundation.usercenter.api.IFaceAndIdCardService;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

/**
 * 人脸识别接口实现
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@Service(value = "faceService")
public class FaceServiceImpl implements IFaceService {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	@Resource
	private IFaceAndIdCardService faceAndIdCardService;

	/**
	 * 人脸活体校验
	 *
	 * @param faceInfoVO 人脸信息VO
	 */
	public Boolean checkFaceQualityAndLive(FaceInfoVO faceInfoVO) {
		logger.info("人脸活体校验，入参：{}", JSON.toJSONString(faceInfoVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
		// 1.校验照片是否包含可用人脸
		CommonResponse<Boolean> faceQuality = faceAndIdCardService.checkFaceQuality(faceInfoVO.getImgUrl());
		if (!faceQuality.isSuccess()) {
			logger.info("人脸活体校验，校验照片是否包含可用人脸失败，失败原因：{}", faceQuality.getMsg());
			throw new BusinessException(faceQuality.getMsg());
		}
		// 2.人脸照片活体检查
		CommonResponse<Boolean> live = faceAndIdCardService.liveCheck(faceInfoVO.getImgUrl());
		if (!live.isSuccess()) {
			logger.info("人脸活体校验，校验照片是否包含可用人脸失败，失败原因：{}", live.getMsg());
			throw new BusinessException(live.getMsg());
		}
		return Boolean.TRUE;
	}

	/**
	 * 人脸注册
	 *
	 * @param faceInfoVO 人脸信息VO
	 *
	 * @return {@link Boolean}
	 */
	@Override
	public Boolean faceRegistration(FaceInfoVO faceInfoVO) {
		logger.info("人脸注册，入参：{}", JSON.toJSONString(faceInfoVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

		// 1.人脸活体校验
		this.checkFaceQualityAndLive(faceInfoVO);

		// 2.查询人脸数据库用户是否存在
		CommonResponse<Boolean> getFaceDbUser = faceAndIdCardService.getFaceDbUser(faceInfoVO.getUserId());
		if (!getFaceDbUser.isSuccess()) {
			logger.info("人脸注册，查询人脸数据库用户是否存在失败，失败原因：{}", getFaceDbUser.getMsg());
			throw new BusinessException(getFaceDbUser.getMsg());
		}
		Boolean flag = getFaceDbUser.getData();
		// 人脸数据库用户：1-不存在，创建人脸数据库用户并执行人脸注册;2-已存在，更新人脸数据库用户并添加人脸数据到指定用户；
		if (Boolean.FALSE.equals(flag)) {
			// 3.创建人脸数据库用户
			CommonResponse<Boolean> createFaceDbUser = faceAndIdCardService.createFaceDbUser(faceInfoVO.getUserId(), faceInfoVO.getUserName());
			if (!createFaceDbUser.isSuccess()) {
				logger.info("人脸注册，创建人脸数据库用户失败，失败原因：{}", createFaceDbUser.getMsg());
				throw new BusinessException(createFaceDbUser.getMsg());
			}
		} else {
			// 4.更新人脸数据库用户
			CommonResponse<Boolean> updateFaceDbUser = faceAndIdCardService.updateFaceDbUser(faceInfoVO.getUserId(), faceInfoVO.getUserName());
			if (!updateFaceDbUser.isSuccess()) {
				logger.info("人脸注册，更新人脸数据库用户失败，失败原因：{}", updateFaceDbUser.getMsg());
				throw new BusinessException(updateFaceDbUser.getMsg());
			}
		}

		// 5.添加人脸数据到指定用户
		CommonResponse<Boolean> addFaceToUser = faceAndIdCardService.addFaceToUser(faceInfoVO.getUserId(), faceInfoVO.getImgUrl());
		if (!addFaceToUser.isSuccess()) {
			logger.info("人脸注册，添加人脸数据到指定用户失败，失败原因：{}", addFaceToUser.getMsg());
			throw new BusinessException(addFaceToUser.getMsg());
		}
		return true;
	}
}
