package com.ejianc.business.labor.service.impl;

import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.labor.bean.*;
import com.ejianc.business.labor.common.dtoMapper.WorkRecordMapper;
import com.ejianc.business.labor.mapper.AttendanceLogMapper;
import com.ejianc.business.labor.service.*;
import com.ejianc.business.labor.utils.MockLogin;
import com.ejianc.business.labor.vo.AttendanceLogVO;
import com.ejianc.business.labor.vo.DistanceVO;
import com.ejianc.business.labor.vo.WorkRecordVO;
import com.ejianc.business.labor.vo.WorkerVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.usercenter.api.IFaceAndIdCardService;
import com.ejianc.foundation.usercenter.api.IThirdSystemApi;
import com.ejianc.foundation.usercenter.vo.ThirdSystemVO;
import com.ejianc.foundation.usercenter.vo.UserVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.EnvironmentTools;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.google.common.collect.Sets;
import com.google.common.collect.UnmodifiableIterator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 考勤日志
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@Service("attendanceLogService")
public class AttendanceLogServiceImpl extends BaseServiceImpl<AttendanceLogMapper, AttendanceLogEntity> implements IAttendanceLogService {
	private static final String WEIXIN_OAUTH_2_URL = "https://api.weixin.qq.com/sns/oauth2/access_token"; // 获取access_token
	private static final String DEFAULT_TEMPLATE_CODE = "SMS_195335075"; // 默认的登录验证模板
	private static final String CODE = "Weixin";
	private static final String BILL_TYPE = "BT202211000007";
	private static final String START_WORK = "上班打卡";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Resource
	private EnvironmentTools environmentTools;
	@Resource
	private IThirdSystemApi thirdSystemApi;
	@Resource
	private IAmapService amapService;
	@Resource
	private RedisTemplate<String, Object> redisTemplate;
	@Resource
	private MockLogin mockLogin;
	@Resource
	private IAttendanceBindService attendanceBindService;
	@Resource
	private IWorkerService workerService;
	@Resource
	private IWorkRecordService workRecordService;
	@Resource
	private IProjectTeamService projectTeamService;
	@Resource
	private IAttachmentApi attachmentApi;
	@Resource
	private IAttendanceService attendanceService;
	@Resource
	private IWorktimeSetService worktimeSetService;
	@Resource
	private IWorktimeSetDetailService worktimeSetDetailService;
	@Resource
	private IFaceAndIdCardService faceAndIdCardService;
	@Resource
	private IUserApi userApi;


	/**
	 * 获取openid
	 *
	 * @param code 临时登录凭证
	 *
	 * @return {@link JSONObject}
	 */
	@Override
	public String getOpenId(String code) {
		Assert.hasText(code, "临时登录凭证不能为空！");

		// 1、查询微信公众号配置信息
		CommonResponse<ThirdSystemVO> thirdSystemResponse = thirdSystemApi.getOneByCode(CODE);
		if (!thirdSystemResponse.isSuccess()) {
			throw new BusinessException("查询微信小程序配置信息失败，失败原因：" + thirdSystemResponse.getMsg());
		}
		ThirdSystemVO thirdSystemVO = thirdSystemResponse.getData();
		String configInfo = thirdSystemVO.getConfigInfo();
		JSONObject configJson = JSON.parseObject(configInfo);
		String appId = configJson.getString("appid");
		String secret = configJson.getString("secret");

		// 2、公众号
		HashMap<String, Object> paramMap = new HashMap<>();
		paramMap.put("appid", appId);
		paramMap.put("secret", secret);
		paramMap.put("code", code);
		paramMap.put("grant_type", "authorization_code");

		HttpResponse response = HttpUtil.createGet(WEIXIN_OAUTH_2_URL)
				.form(paramMap).execute();
		if (!response.isOk()) {
			throw new BusinessException("请求失败！");
		}
		JSONObject jsonObject = JSON.parseObject(response.body());
		if (jsonObject.containsKey("errcode")) {
			throw new BusinessException("获取openid失败，失败原因：" + jsonObject.getString("errmsg"));
		}
		return jsonObject.getString("openid");
	}


	/**
	 * 租户手机号绑定openid校验
	 *
	 * @param openid openid
	 *
	 * @return {@link String}
	 */
	@Override
	public String checkBind(String openid) {
		Assert.hasText(openid, "openid不能为空！");

		logger.info("租户手机号绑定openid校验，入参：openid-{}", openid);
		LambdaQueryWrapper<AttendanceBindEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(AttendanceBindEntity::getOpenid, openid);
		AttendanceBindEntity attendanceBindEntity = attendanceBindService.getOne(lambdaQuery);
		if (attendanceBindEntity != null) {
			return attendanceBindEntity.getPhone();
		}
		return null;
	}


	/**
	 * 给指定手机号发送验证码或者短信
	 *
	 * @param phone        手机号
	 * @param templateCode 短信模板编码
	 * @param signName     短信签名
	 *
	 * @return {@link String}
	 *
	 * @throws Exception 异常
	 */
	@Override
	public String sendMsg(String phone, String templateCode, String signName) throws Exception {
		Assert.hasText(phone, "手机号不能为空！");
		Assert.hasText(templateCode, "短信模板编码不能为空！");
		Assert.hasText(signName, "短信签名不能为空！");

		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getPhone, phone);
		List<WorkRecordEntity> workRecordEntityList = workRecordService.list(lambdaQuery);
		if (CollectionUtils.isEmpty(workRecordEntityList)) {
			throw new BusinessException("手机号在平台花名册中不存在！");
		}
		WorkRecordEntity workRecordEntity = workRecordEntityList.get(0);
		Long workerId = workRecordEntity.getWorkerId();
		WorkerEntity workerEntity = workerService.selectById(workerId);
		if (workerEntity == null) {
			throw new BusinessException("手机号在平台花名册中不存在！");
		}

		JSONObject messageParam = new JSONObject();
		messageParam.put("phone", phone);
		if (StringUtils.isBlank(templateCode)) {
			messageParam.put("templateCode", DEFAULT_TEMPLATE_CODE);
		} else {
			messageParam.put("templateCode", templateCode);
		}
		if (StringUtils.isNotBlank(signName)) {
			messageParam.put("signName", signName);
		}
		String messageUrl = environmentTools.getBaseHost() + "/ejc-message-web/no_auth/sms/sendMessage";
		String responseStr = HttpTookit.postByJson(messageUrl, JSON.toJSONString(messageParam));
		logger.info("向手机号[{}]发送验证码结果：[{}]", phone, responseStr);

		CommonResponse<String> response = JSON.parseObject(responseStr, CommonResponse.class);
		if (!response.isSuccess()) {
			throw new BusinessException("短信发送失败");
		}
		return "短信发送成功！";
	}


	/**
	 * 登录
	 *
	 * @param phone        手机号
	 * @param templateCode 短信模版
	 * @param validate     验证码
	 * @param openid       openid
	 */
	public void login(String phone, String templateCode, String validate, String openid) {
		Assert.hasText(phone, "手机号不能为空！");
		Assert.hasText(validate, "验证码不能为空！");
		Assert.hasText(openid, "openid不能为空！");

		// 1、校验验证码
		String checkMessageUrl = environmentTools.getBaseHost() + "/ejc-message-web/no_auth/sms/checkMessage";
		JSONObject jsonObject = new JSONObject();
		if (StringUtils.isNoneBlank(templateCode)) {
			jsonObject.put("templateCode", templateCode);
		} else {
			jsonObject.put("templateCode", DEFAULT_TEMPLATE_CODE);
		}
		jsonObject.put("phone", phone);
		jsonObject.put("validate", validate);

		HttpResponse response = HttpUtil.createPost(checkMessageUrl)
				.body(jsonObject.toJSONString())
				.execute();
		if (!response.isOk()) {
			throw new BusinessException("校验验证码失败！");
		}
		String responseStr = response.body();
		if (StringUtils.isNotBlank(responseStr)) {
			CommonResponse<String> checkResponse = JSON.parseObject(responseStr, CommonResponse.class);
			if (0 != checkResponse.getCode()) {
				logger.info("校验验证码失败，失败原因：{}", checkResponse.getMsg());
				throw new BusinessException(checkResponse.getMsg());
			}
		} else {
			throw new BusinessException("校验验证码失败！");
		}

		// 2、绑定租户手机号和openid：未绑定的绑定，已绑定需要更新绑定手机号
		String checkBind = checkBind(openid);
		if (StringUtils.isBlank(checkBind)) {
			AttendanceBindEntity attendanceBindEntity = new AttendanceBindEntity();
			attendanceBindEntity.setPhone(phone);
			attendanceBindEntity.setOpenid(openid);
			attendanceBindService.saveOrUpdateNoES(attendanceBindEntity);
		} else {
			LambdaUpdateWrapper<AttendanceBindEntity> lambdaUpdate = Wrappers.lambdaUpdate();
			lambdaUpdate.eq(AttendanceBindEntity::getOpenid, openid);
			lambdaUpdate.set(AttendanceBindEntity::getPhone, phone);
			attendanceBindService.update(lambdaUpdate);
		}
	}


	/**
	 * 获取项目
	 *
	 * @param phone 手机号
	 *
	 * @return {@link List}<{@link JSONObject}>
	 */
	@Override
	public List<JSONObject> fetchProject(String phone) {
		Assert.hasText(phone, "手机号不能为空！");

		ArrayList<JSONObject> result = new ArrayList<>();
		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getPhone, phone);
		lambdaQuery.eq(WorkRecordEntity::getEnterExitState, 1);
		List<WorkRecordEntity> workRecordEntityList = workRecordService.list(lambdaQuery);

		if (CollectionUtils.isNotEmpty(workRecordEntityList)) {
			for (WorkRecordEntity workRecord : workRecordEntityList) {
				JSONObject jsonObject = new JSONObject();
				Long projectId = workRecord.getProjectId();
				jsonObject.put("tenantId", workRecord.getTenantId());
				jsonObject.put("projectId", projectId);
				jsonObject.put("projectName", workRecord.getProjectName());

				LambdaQueryWrapper<WorktimeSetEntity> lambdaQueryWrapper = Wrappers.lambdaQuery();
				lambdaQueryWrapper.eq(WorktimeSetEntity::getProjectId, projectId);
				WorktimeSetEntity worktimeSetEntity = worktimeSetService.getOne(lambdaQueryWrapper);
				// 无考勤设置不能切换项目
				jsonObject.put("worktimeSetId", worktimeSetEntity != null ? worktimeSetEntity.getId() : null);
				result.add(jsonObject);
			}
		}
		return result;
	}


	/**
	 * 校验人脸信息设置
	 *
	 * @param tenantId  租户id
	 * @param projectId 项目id
	 * @param phone     手机号
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@Override
	public CommonResponse<String> checkFaceDataSetting(Long tenantId, Long projectId, String phone) {
		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");

		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getTenantId, tenantId);
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId);
		lambdaQuery.eq(WorkRecordEntity::getPhone, phone);
		List<WorkRecordEntity> workRecordEntityList = workRecordService.list(lambdaQuery);
		if (CollectionUtils.isEmpty(workRecordEntityList)) {
			throw new BusinessException("手机号在平台租户花名册中不存在！");
		}
		WorkRecordEntity workRecordEntity = workRecordEntityList.get(0);
		Long workerId = workRecordEntity.getWorkerId();

		CommonResponse<List<AttachmentVO>> response = attachmentApi.queryListBySourceId(workerId, BILL_TYPE, "face", null);
		if (!response.isSuccess() || CollectionUtils.isEmpty(response.getData())) {
			return CommonResponse.error("提示：当前用户未设置人脸信息，无法进行手机打卡，请联系项目管理人员添加；");
		}
		return CommonResponse.success();
	}


	/**
	 * 校验项目打卡权限
	 *
	 * @param tenantId  租户id
	 * @param projectId 项目id
	 * @param phone     手机号
	 */
	@Override
	public void checkPunchCardAuth(Long tenantId, Long projectId, String phone) {
		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");

		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getTenantId, tenantId);
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId);
		lambdaQuery.eq(WorkRecordEntity::getPhone, phone);
		lambdaQuery.eq(WorkRecordEntity::getEnterExitState, 1);
		List<WorkRecordEntity> workRecordEntityList = workRecordService.list(lambdaQuery);
		if (CollectionUtils.isEmpty(workRecordEntityList)) {
			throw new BusinessException("无该项目打卡权限");
		}
	}


	/**
	 * 校验考勤范围
	 *
	 * @param worktimeSetId 考勤设置id
	 * @param destination   当前位置经纬度
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@Override
	public CommonResponse<JSONObject> checkDistance(String worktimeSetId, String destination) {
		Assert.hasText(worktimeSetId, "考勤设置id不能为空！");
		Assert.hasText(destination, "当前位置经纬度不能为空！");

		JSONObject jsonObject = new JSONObject();
		WorktimeSetEntity worktimeSetEntity = worktimeSetService.selectById(worktimeSetId);
		Assert.notNull(worktimeSetEntity, "考勤设置不能为空！");

		List<WorktimeSetDetailEntity> detailList = worktimeSetEntity.getWorktimeSetDetailList();
		if (CollectionUtils.isNotEmpty(detailList)) {
			// 是否允许异地打卡（1:是 0：否）
			Integer offSiteFlag = worktimeSetEntity.getOffSiteFlag();

			String lngLat = detailList.stream().map(WorktimeSetDetailEntity::getLngLat).collect(Collectors.joining("|"));
			List<DistanceVO> distance = amapService.distance(lngLat, destination, "0");
			logger.info("校验考勤范围--{}", JSON.toJSONString(distance, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

			Map<String, DistanceVO> distanceMap = distance.stream().collect(Collectors.toMap(DistanceVO::getDistance, Function.identity(), (key1, key2) -> key2));
			BigDecimal min = distance.stream().map(d -> new BigDecimal(d.getDistance())).min(Comparator.comparing(x -> x)).orElse(null);
			DistanceVO minDistanceVO = distanceMap.get(Objects.requireNonNull(min).toString());
			WorktimeSetDetailEntity worktimeSetDetailEntity = detailList.get(Integer.parseInt(minDistanceVO.getOriginId()) - 1);

			// 判断是否在考勤范围
			if (new BigDecimal(minDistanceVO.getDistance()).compareTo(worktimeSetDetailEntity.getEffectiveRange()) < 0) {
				// 打卡地点
				jsonObject.put("punchCardPlace", worktimeSetDetailEntity.getAddress());
				// 是否是异地打卡（1:是 0：否）
				jsonObject.put("remoteFlag", 0);
				return CommonResponse.success("考勤范围校验成功！", jsonObject);
			} else {
				if (offSiteFlag == 1) {
					JSONObject regeo = amapService.regeo(destination);
					logger.info("校验考勤范围，逆地理编码--{}", regeo);
					String formattedAddress = regeo.getString("formatted_address");
					// 打卡地点
					jsonObject.put("punchCardPlace", formattedAddress);
					// 是否是异地打卡（1:是 0：否）
					jsonObject.put("remoteFlag", 1);
					return CommonResponse.error("当前定位不在项目设定打卡范围内，继续打卡会标记为异常打卡，是否确定打卡？", jsonObject);
				} else {
					return CommonResponse.error("当前定位不在项目设定打卡范围内，不允许打卡", jsonObject);
				}

			}
		}
		return CommonResponse.error("考勤范围校验失败！", jsonObject);
	}


	/**
	 * 校验人脸
	 *
	 * @param tenantId   租户id
	 * @param projectId  项目id
	 * @param phone      手机号
	 * @param imgBase64A 待识别人脸照片base64编码
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@Override
	public CommonResponse<WorkerVO> checkFace(Long tenantId, Long projectId, String phone, String imgBase64A) {
		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");
		Assert.hasText(imgBase64A, "待识别人脸照片不能为空！");

		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getTenantId, tenantId)
				.eq(WorkRecordEntity::getProjectId, projectId)
				.eq(WorkRecordEntity::getPhone, phone)
				.eq(WorkRecordEntity::getEnterExitState, 1);
		WorkRecordEntity workRecordEntity = workRecordService.getOne(lambdaQuery);
		WorkerEntity workerEntity = workerService.selectById(workRecordEntity.getWorkerId());
		if (workerEntity == null) {
			throw new BusinessException("校验人脸信息失败！");
		}
		CommonResponse<List<AttachmentVO>> response = attachmentApi.queryListBySourceId(workerEntity.getId(), BILL_TYPE, "face", null);
		if (!response.isSuccess()) {
			throw new BusinessException("获取人脸信息失败！");
		}
		Assert.notEmpty(response.getData(), "项目花名册人脸照片不能为空！");

		AttachmentVO attachmentVO = response.getData().get(0);
		String truePath = attachmentVO.getTruePath();

		JSONObject jsonObject = new JSONObject();
		jsonObject.put("imgBase64A", imgBase64A);
		jsonObject.put("imgUrlB", truePath);

		CommonResponse<Boolean> res = faceAndIdCardService.compareFace(jsonObject);
		logger.info("校验人脸，人脸识别结果--{}", res);
		if (!res.isSuccess()) {
			throw new BusinessException("人脸识别失败，失败原因：" + res.getMsg());
		}
		Boolean flag = res.getData();
		if (Boolean.TRUE.equals(flag)) {
			return CommonResponse.success("人脸信息校验成功！", BeanMapper.map(workerEntity, WorkerVO.class));
		}
		return CommonResponse.error("人脸信息校验失败！");
	}


	/**
	 * 根据公司花名册id校验人脸
	 *
	 * @param workerId   公司花名册人员id
	 * @param imgBase64A 待识别人脸照片base64编码
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@Override
	public CommonResponse<WorkerVO> checkFaceByWorkerId(Long workerId, String imgBase64A) {
		Assert.notNull(workerId, "公司花名册人员id不能为空！");
		Assert.hasText(imgBase64A, "待识别人脸照片不能为空！");

		WorkerEntity workerEntity = workerService.selectById(workerId);
		if (workerEntity == null) {
			throw new BusinessException("公司花名册人员不存在！");
		}
		CommonResponse<List<AttachmentVO>> response = attachmentApi.queryListBySourceId(workerEntity.getId(), BILL_TYPE, "face", null);
		if (!response.isSuccess()) {
			throw new BusinessException("获取人脸信息失败！");
		}
		Assert.notEmpty(response.getData(), "公司花名册人脸照片不能为空！");

		AttachmentVO attachmentVO = response.getData().get(0);
		String truePath = attachmentVO.getTruePath();

		JSONObject jsonObject = new JSONObject();
		jsonObject.put("imgBase64A", imgBase64A);
		jsonObject.put("imgUrlB", truePath);

		logger.info("校验人脸，入参，imgBase64A--{}，imgUrlB--{}", imgBase64A, truePath);
		CommonResponse<Boolean> res = faceAndIdCardService.compareFace(jsonObject);
		logger.info("校验人脸，人脸识别结果--{}", res);
		if (!res.isSuccess()) {
			throw new BusinessException("人脸识别失败，失败原因：" + res.getMsg());
		}
		Boolean flag = res.getData();
		if (Boolean.TRUE.equals(flag)) {
			return CommonResponse.success("人脸信息校验成功！", BeanMapper.map(workerEntity, WorkerVO.class));
		}
		return CommonResponse.error("人脸信息校验失败！");
	}

	/**
	 * 校验已存在的打卡记录
	 *
	 * @param tenantId      租户id
	 * @param projectId     项目id
	 * @param phone         手机号
	 * @param punchCardType 打卡类型：上班打卡、下班打卡
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link AttendanceLogVO}>>
	 */
	@Override
	public CommonResponse<List<AttendanceLogVO>> checkExistedLog(Long tenantId, Long projectId, String phone, String punchCardType) {
		Assert.notNull(tenantId, "租户id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(phone, "手机号不能为空！");
		Assert.hasText(punchCardType, "打卡类型不能为空！");

		LambdaQueryWrapper<AttendanceLogEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(AttendanceLogEntity::getTenantId, tenantId);
		lambdaQuery.eq(AttendanceLogEntity::getProjectId, projectId);
		lambdaQuery.eq(AttendanceLogEntity::getPhone, phone);
		lambdaQuery.eq(AttendanceLogEntity::getPunchCardType, punchCardType);
		lambdaQuery.eq(AttendanceLogEntity::getPunchCardDate, DateUtil.formatDate(new Date()));
		if (START_WORK.equals(punchCardType)) {
			lambdaQuery.orderByDesc(AttendanceLogEntity::getStartWorkTime);
		} else {
			lambdaQuery.orderByDesc(AttendanceLogEntity::getEndWorkTime);
		}
		List<AttendanceLogEntity> logs = super.list(lambdaQuery);

		if (CollectionUtils.isNotEmpty(logs)) {
			return CommonResponse.error("今天已存在" + punchCardType + "，是否需要更新", BeanMapper.mapList(logs, AttendanceLogVO.class));
		}
		return CommonResponse.success("已存在的记录校验成功！");
	}


	/**
	 * 考勤打卡
	 *
	 * @param workerId       公司花名册人员id
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link AttendanceLogVO}
	 */
	@Override
	public AttendanceLogVO punchCard(Long projectId, Long workerId, String punchCardType, String punchCardPlace, Integer remoteFlag) {
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.notNull(workerId, "公司花名册人员id不能为空！");
		Assert.hasText(punchCardType, "打卡类型不能为空！");
		Assert.hasText(punchCardPlace, "打卡地点不能为空！");
		Assert.notNull(remoteFlag, "是否是异地打卡不能为空！");

		// 1、新增考勤打卡日志
		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId)
				.eq(WorkRecordEntity::getWorkerId, workerId)
				.eq(WorkRecordEntity::getEnterExitState, 1);
		WorkRecordEntity workRecordEntity = workRecordService.getOne(lambdaQuery);

		AttendanceLogEntity attendanceLogEntity = WorkRecordMapper.INSTANCE.transformAttendanceLogEntity(workRecordEntity);

		attendanceLogEntity.setPunchCardDate(new Date());
		if (START_WORK.equals(punchCardType)) {
			attendanceLogEntity.setStartWorkTime(new Date());
			attendanceLogEntity.setStartWorkPlace(punchCardPlace);
		} else {
			attendanceLogEntity.setEndWorkTime(new Date());
			attendanceLogEntity.setEndWorkPlace(punchCardPlace);
		}
		attendanceLogEntity.setRemoteFlag(remoteFlag);
		if (remoteFlag == 0) {
			attendanceLogEntity.setAttendanceStatus("正常");
		} else {
			attendanceLogEntity.setAttendanceStatus("异常");
		}
		attendanceLogEntity.setPunchCardType(punchCardType);
		super.saveOrUpdate(attendanceLogEntity, false);

		// 2、查询考勤打卡记录
		LambdaQueryWrapper<AttendanceEntity> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(AttendanceEntity::getProjectId, projectId);
		wrapper.eq(AttendanceEntity::getWorkerId, workerId);
		wrapper.eq(AttendanceEntity::getPunchCardDate, DateUtil.formatDate(new Date()));
		AttendanceEntity attendanceEntity = attendanceService.getOne(wrapper);
		if (attendanceEntity == null) {
			AttendanceEntity entity = com.ejianc.business.labor.common.dtoMapper.AttendanceLogMapper.INSTANCE.transformAttendanceEntity(attendanceLogEntity);
			entity.setDuration(0L);
			attendanceService.saveOrUpdate(entity, false);
		} else {
			if (START_WORK.equals(punchCardType)) {
				attendanceEntity.setStartWorkTime(attendanceLogEntity.getStartWorkTime());
				attendanceEntity.setStartWorkPlace(attendanceLogEntity.getStartWorkPlace());
				if (attendanceEntity.getEndWorkTime() != null) {
					attendanceEntity.setDuration(DateUtil.between(attendanceEntity.getStartWorkTime(), attendanceEntity.getEndWorkTime(), DateUnit.HOUR));
				}
			} else {
				attendanceEntity.setEndWorkTime(attendanceLogEntity.getEndWorkTime());
				attendanceEntity.setEndWorkPlace(attendanceLogEntity.getEndWorkPlace());
				if (attendanceEntity.getStartWorkTime() != null) {
					attendanceEntity.setDuration(DateUtil.between(attendanceEntity.getStartWorkTime(), attendanceEntity.getEndWorkTime(), DateUnit.HOUR));
				}
			}
			attendanceEntity.setAttendanceStatus(attendanceLogEntity.getAttendanceStatus());
			attendanceEntity.setRemoteFlag(attendanceLogEntity.getRemoteFlag());
			attendanceService.saveOrUpdate(attendanceEntity, false);
		}

		return BeanMapper.map(attendanceLogEntity, AttendanceLogVO.class);
	}


	/**
	 * 内部员工：获取项目下所有在场的劳务人员
	 *
	 * @param param 参数
	 *
	 * @return {@link IPage}<{@link WorkRecordVO}>
	 */
	@Override
	public IPage<WorkRecordVO> internalFetchWorkRecord(QueryParam param) {
		// 模糊搜索
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("name");
		fuzzyFields.add("teamName");
		fuzzyFields.add("projectWorkTypeName");
		// 租户隔离
		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		IPage<WorkRecordEntity> page = workRecordService.queryPage(param);
		Page<WorkRecordVO> result = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		List<WorkRecordVO> records = BeanMapper.mapList(page.getRecords(), WorkRecordVO.class);
		result.setRecords(records);
		return result;
	}


	/**
	 * 内部员工：校验考勤范围
	 *
	 * @param projectId   项目id
	 * @param destination 当前位置经纬度
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@Override
	public CommonResponse<JSONObject> internalCheckDistance(Long projectId, String destination) {
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(destination, "当前位置经纬度不能为空！");

		JSONObject jsonObject = new JSONObject();

		LambdaQueryWrapper<WorktimeSetEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorktimeSetEntity::getTenantId, InvocationInfoProxy.getTenantid())
				.eq(WorktimeSetEntity::getProjectId, projectId);
		WorktimeSetEntity worktimeSetEntity = worktimeSetService.getOne(lambdaQuery);
		Assert.notNull(worktimeSetEntity, "考勤设置不能为空！");

		LambdaQueryWrapper<WorktimeSetDetailEntity> l = Wrappers.lambdaQuery();
		l.eq(WorktimeSetDetailEntity::getWorktimeId, worktimeSetEntity.getId());
		List<WorktimeSetDetailEntity> detailList = worktimeSetDetailService.list(l);
		if (CollectionUtils.isNotEmpty(detailList)) {
			// 是否允许异地打卡（1:是 0：否）
			Integer offSiteFlag = worktimeSetEntity.getOffSiteFlag();

			String lngLat = detailList.stream().map(WorktimeSetDetailEntity::getLngLat).collect(Collectors.joining("|"));
			List<DistanceVO> distance = amapService.distance(lngLat, destination, "0");
			logger.info("校验考勤范围--{}", JSON.toJSONString(distance, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

			Map<String, DistanceVO> distanceMap = distance.stream().collect(Collectors.toMap(DistanceVO::getDistance, Function.identity(), (key1, key2) -> key2));
			BigDecimal min = distance.stream().map(d -> new BigDecimal(d.getDistance())).min(Comparator.comparing(x -> x)).orElse(null);
			DistanceVO minDistanceVO = distanceMap.get(Objects.requireNonNull(min).toString());
			WorktimeSetDetailEntity worktimeSetDetailEntity = detailList.get(Integer.parseInt(minDistanceVO.getOriginId()) - 1);

			// 判断是否在考勤范围
			if (new BigDecimal(minDistanceVO.getDistance()).compareTo(worktimeSetDetailEntity.getEffectiveRange()) < 0) {
				// 打卡地点
				jsonObject.put("punchCardPlace", worktimeSetDetailEntity.getAddress());
				// 是否是异地打卡（1:是 0：否）
				jsonObject.put("remoteFlag", 0);
				return CommonResponse.success("考勤范围校验成功！", jsonObject);
			} else {
				if (offSiteFlag == 1) {
					JSONObject regeo = amapService.regeo(destination);
					logger.info("校验考勤范围，逆地理编码--{}", regeo);
					String formattedAddress = regeo.getString("formatted_address");
					// 打卡地点
					jsonObject.put("punchCardPlace", formattedAddress);
					// 是否是异地打卡（1:是 0：否）
					jsonObject.put("remoteFlag", 1);
					return CommonResponse.error("当前定位不在项目设定打卡范围内，继续打卡会标记为异常打卡，是否确定打卡？", jsonObject);
				} else {
					return CommonResponse.error("当前定位不在项目设定打卡范围内，不允许打卡", jsonObject);
				}

			}
		}
		return CommonResponse.error("考勤范围校验失败！", jsonObject);
	}


	/**
	 * 内部员工：校验人脸
	 *
	 * @param projectId  项目id
	 * @param imgBase64A 待识别人脸照片base64编码
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@Override
	public CommonResponse<WorkerVO> internalCheckFace(Long projectId, String imgBase64A) {
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(imgBase64A, "待识别人脸照片不能为空！");

		// 通过人脸获取人员ids
		byte[] imgBytes = Base64.getDecoder().decode(imgBase64A);
		CommonResponse<List<Long>> response = faceAndIdCardService.getUserByFaceImgByte(imgBytes);
		if (!response.isSuccess()) {
			return CommonResponse.error("打卡失败，无法识别人脸");
		}
		Set<Long> ids = new HashSet<>(response.getData());
		logger.info("通过人脸获取人员ids--{}", JSON.toJSONString(ids));

		// 项目在场人员
		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getTenantId, InvocationInfoProxy.getTenantid());
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId);
		lambdaQuery.eq(WorkRecordEntity::getEnterExitState, 1);
		List<WorkRecordEntity> workRecordEntityList = workRecordService.list(lambdaQuery);
		if (CollectionUtils.isEmpty(workRecordEntityList)) {
			return CommonResponse.error("项目无在场人员");
		}

		// 取交集
		Set<Long> workerIds = workRecordEntityList.stream().map(WorkRecordEntity::getWorkerId).collect(Collectors.toSet());
		Map<Long, WorkRecordEntity> workRecordEntityMap = workRecordEntityList.stream().collect(Collectors.toMap(WorkRecordEntity::getWorkerId, Function.identity(), (key1, key2) -> key2));
		Sets.SetView<Long> workerId = Sets.intersection(ids, workerIds);
		if (workerId.isEmpty()) {
			return CommonResponse.error("打卡失败，无法识别人脸");
		}

		// 获取对应人员信息
		UnmodifiableIterator<Long> iterator = workerId.iterator();
		WorkRecordEntity workRecordEntity = workRecordEntityMap.get(iterator.next());
		WorkerEntity workerEntity = workerService.selectById(workRecordEntity.getWorkerId());

		return CommonResponse.success("人脸信息校验成功！", BeanMapper.map(workerEntity, WorkerVO.class));
	}


	/**
	 * 内部员工：校验已存在的打卡记录
	 *
	 * @param workerId      项目花名册人员id
	 * @param projectId     项目id
	 * @param punchCardType 打卡类型：上班打卡、下班打卡
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link AttendanceLogVO}>>
	 */
	@Override
	public CommonResponse<List<AttendanceLogVO>> internalCheckExistedLog(Long workerId, Long projectId, String punchCardType) {
		Assert.notNull(workerId, "项目花名册人员id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(punchCardType, "打卡类型不能为空！");

		LambdaQueryWrapper<AttendanceLogEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(AttendanceLogEntity::getTenantId, InvocationInfoProxy.getTenantid())
				.eq(AttendanceLogEntity::getProjectId, projectId)
				.eq(AttendanceLogEntity::getWorkerId, workerId)
				.eq(AttendanceLogEntity::getPunchCardType, punchCardType)
				.eq(AttendanceLogEntity::getPunchCardDate, DateUtil.formatDate(new Date()));
		if (START_WORK.equals(punchCardType)) {
			lambdaQuery.orderByDesc(AttendanceLogEntity::getStartWorkTime);
		} else {
			lambdaQuery.orderByDesc(AttendanceLogEntity::getEndWorkTime);
		}
		List<AttendanceLogEntity> logs = super.list(lambdaQuery);

		if (CollectionUtils.isNotEmpty(logs)) {
			return CommonResponse.error("今天已存在" + punchCardType + "，是否需要更新", BeanMapper.mapList(logs, AttendanceLogVO.class));
		}
		return CommonResponse.success("已存在的记录校验成功！");
	}


	/**
	 * 内部员工：考勤打卡
	 *
	 * @param workerId       项目花名册人员id
	 * @param projectId      项目id
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link AttendanceLogVO}
	 */
	@Override
	public AttendanceLogVO internalPunchCard(Long workerId, Long projectId, String punchCardType, String punchCardPlace, Integer remoteFlag) {
		Assert.notNull(workerId, "项目花名册人员id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(punchCardType, "打卡类型不能为空！");
		Assert.hasText(punchCardPlace, "打卡地点不能为空！");
		Assert.notNull(remoteFlag, "是否是异地打卡不能为空！");

		// 1、新增考勤打卡日志
		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId)
				.eq(WorkRecordEntity::getWorkerId, workerId)
				.eq(WorkRecordEntity::getEnterExitState, 1);
		WorkRecordEntity workRecordEntity = workRecordService.getOne(lambdaQuery);

		AttendanceLogEntity attendanceLogEntity = WorkRecordMapper.INSTANCE.transformAttendanceLogEntity(workRecordEntity);

		attendanceLogEntity.setPunchCardDate(new Date());
		if (START_WORK.equals(punchCardType)) {
			attendanceLogEntity.setStartWorkTime(new Date());
			attendanceLogEntity.setStartWorkPlace(punchCardPlace);
		} else {
			attendanceLogEntity.setEndWorkTime(new Date());
			attendanceLogEntity.setEndWorkPlace(punchCardPlace);
		}
		attendanceLogEntity.setRemoteFlag(remoteFlag);
		if (remoteFlag == 0) {
			attendanceLogEntity.setAttendanceStatus("正常");
		} else {
			attendanceLogEntity.setAttendanceStatus("异常");
		}
		attendanceLogEntity.setPunchCardType(punchCardType);
		super.saveOrUpdate(attendanceLogEntity, false);

		// 2、查询考勤打卡记录
		LambdaQueryWrapper<AttendanceEntity> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(AttendanceEntity::getTenantId, InvocationInfoProxy.getTenantid());
		wrapper.eq(AttendanceEntity::getProjectId, projectId);
		wrapper.eq(AttendanceEntity::getWorkerId, workerId);
		wrapper.eq(AttendanceEntity::getPunchCardDate, DateUtil.formatDate(new Date()));
		AttendanceEntity attendanceEntity = attendanceService.getOne(wrapper);
		if (attendanceEntity == null) {
			AttendanceEntity entity = com.ejianc.business.labor.common.dtoMapper.AttendanceLogMapper.INSTANCE.transformAttendanceEntity(attendanceLogEntity);
			entity.setDuration(0L);
			attendanceService.saveOrUpdate(entity, false);
		} else {
			if (START_WORK.equals(punchCardType)) {
				attendanceEntity.setStartWorkTime(attendanceLogEntity.getStartWorkTime());
				attendanceEntity.setStartWorkPlace(attendanceLogEntity.getStartWorkPlace());
				if (attendanceEntity.getEndWorkTime() != null) {
					attendanceEntity.setDuration(DateUtil.between(attendanceEntity.getStartWorkTime(), attendanceEntity.getEndWorkTime(), DateUnit.HOUR));
				}
			} else {
				attendanceEntity.setEndWorkTime(attendanceLogEntity.getEndWorkTime());
				attendanceEntity.setEndWorkPlace(attendanceLogEntity.getEndWorkPlace());
				if (attendanceEntity.getStartWorkTime() != null) {
					attendanceEntity.setDuration(DateUtil.between(attendanceEntity.getStartWorkTime(), attendanceEntity.getEndWorkTime(), DateUnit.HOUR));
				}
			}
			attendanceEntity.setAttendanceStatus(attendanceLogEntity.getAttendanceStatus());
			attendanceEntity.setRemoteFlag(attendanceLogEntity.getRemoteFlag());
			attendanceService.saveOrUpdate(attendanceEntity, false);
		}

		return BeanMapper.map(attendanceLogEntity, AttendanceLogVO.class);
	}


	/**
	 * 班组长：获取项目下当前用户所在班组（通过班组长手机号与登录用户手机号校验）的在场用户
	 *
	 * @param param 参数
	 *
	 * @return {@link IPage}<{@link WorkRecordVO}>
	 */
	@Override
	public IPage<WorkRecordVO> leaderFetchWorkRecord(QueryParam param) {
		CommonResponse<UserVO> response = userApi.findUserByUserId(InvocationInfoProxy.getUserid());
		if (!response.isSuccess()) {
			throw new BusinessException("获取项目下当前用户所在班组的在场用户失败，失败原因" + response.getMsg());
		}
		UserVO userVO = response.getData();
		String userMobile = userVO.getUserMobile();
		if (StringUtils.isBlank(userMobile)) {
			throw new BusinessException("获取项目下当前用户所在班组的在场用户失败，失败原因：当前用户在系统内未维护手机号！");
		}

		// 通过当前登录人手机号查班组ids
		LambdaQueryWrapper<ProjectTeamEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(ProjectTeamEntity::getPhone, userMobile);
		List<ProjectTeamEntity> projectTeamEntityList = projectTeamService.list(lambdaQuery);
		if (CollectionUtils.isEmpty(projectTeamEntityList)) {
			return new Page<>();
		}
		Set<Long> ids = projectTeamEntityList.stream().map(ProjectTeamEntity::getId).collect(Collectors.toSet());

		// 模糊搜索
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("name");
		fuzzyFields.add("teamName");
		fuzzyFields.add("projectWorkTypeName");
		// 租户隔离
		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		QueryWrapper<WorkRecordEntity> queryWrapper = changeToQueryWrapper(param);
		LambdaQueryWrapper<WorkRecordEntity> l = Wrappers.lambdaQuery();
		l.eq(WorkRecordEntity::getTenantId, InvocationInfoProxy.getTenantid())
				.eq(param.getParams().containsKey("projectId"), WorkRecordEntity::getProjectId, param.getParams().get("projectId").getValue())
				.eq(WorkRecordEntity::getEnterExitState, 1)
				.in(WorkRecordEntity::getProjectTeamId, ids);
		Page<WorkRecordEntity> p = new Page<>(param.getPageIndex(), param.getPageSize());
		IPage<WorkRecordEntity> page = workRecordService.page(p, queryWrapper);

		Page<WorkRecordVO> result = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		List<WorkRecordVO> records = BeanMapper.mapList(page.getRecords(), WorkRecordVO.class);
		result.setRecords(records);
		return result;
	}


	/**
	 * 班组长：校验考勤范围
	 *
	 * @param projectId   项目id
	 * @param destination 当前位置经纬度
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@Override
	public CommonResponse<JSONObject> leaderCheckDistance(Long projectId, String destination) {
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(destination, "当前位置经纬度不能为空！");

		JSONObject jsonObject = new JSONObject();

		LambdaQueryWrapper<WorktimeSetEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorktimeSetEntity::getTenantId, InvocationInfoProxy.getTenantid())
				.eq(WorktimeSetEntity::getProjectId, projectId);
		WorktimeSetEntity worktimeSetEntity = worktimeSetService.getOne(lambdaQuery);
		Assert.notNull(worktimeSetEntity, "考勤设置不能为空！");

		LambdaQueryWrapper<WorktimeSetDetailEntity> l = Wrappers.lambdaQuery();
		l.eq(WorktimeSetDetailEntity::getWorktimeId, worktimeSetEntity.getId());
		List<WorktimeSetDetailEntity> detailList = worktimeSetDetailService.list(l);
		if (CollectionUtils.isNotEmpty(detailList)) {
			// 是否允许异地打卡（1:是 0：否）
			Integer offSiteFlag = worktimeSetEntity.getOffSiteFlag();

			String lngLat = detailList.stream().map(WorktimeSetDetailEntity::getLngLat).collect(Collectors.joining("|"));
			List<DistanceVO> distance = amapService.distance(lngLat, destination, "0");
			logger.info("校验考勤范围--{}", JSON.toJSONString(distance, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

			Map<String, DistanceVO> distanceMap = distance.stream().collect(Collectors.toMap(DistanceVO::getDistance, Function.identity(), (key1, key2) -> key2));
			BigDecimal min = distance.stream().map(d -> new BigDecimal(d.getDistance())).min(Comparator.comparing(x -> x)).orElse(null);
			DistanceVO minDistanceVO = distanceMap.get(Objects.requireNonNull(min).toString());
			WorktimeSetDetailEntity worktimeSetDetailEntity = detailList.get(Integer.parseInt(minDistanceVO.getOriginId()) - 1);

			// 判断是否在考勤范围
			if (new BigDecimal(minDistanceVO.getDistance()).compareTo(worktimeSetDetailEntity.getEffectiveRange()) < 0) {
				// 打卡地点
				jsonObject.put("punchCardPlace", worktimeSetDetailEntity.getAddress());
				// 是否是异地打卡（1:是 0：否）
				jsonObject.put("remoteFlag", 0);
				return CommonResponse.success("考勤范围校验成功！", jsonObject);
			} else {
				if (offSiteFlag == 1) {
					JSONObject regeo = amapService.regeo(destination);
					logger.info("校验考勤范围，逆地理编码--{}", regeo);
					String formattedAddress = regeo.getString("formatted_address");
					// 打卡地点
					jsonObject.put("punchCardPlace", formattedAddress);
					// 是否是异地打卡（1:是 0：否）
					jsonObject.put("remoteFlag", 1);
					return CommonResponse.error("当前定位不在项目设定打卡范围内，继续打卡会标记为异常打卡，是否确定打卡？", jsonObject);
				} else {
					return CommonResponse.error("当前定位不在项目设定打卡范围内，不允许打卡", jsonObject);
				}

			}
		}
		return CommonResponse.error("考勤范围校验失败！", jsonObject);
	}


	/**
	 * 班组长：校验人脸
	 *
	 * @param projectId  项目id
	 * @param imgBase64A 待识别人脸照片base64编码
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@Override
	public CommonResponse<WorkerVO> leaderCheckFace(Long projectId, String imgBase64A) {
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(imgBase64A, "待识别人脸照片不能为空！");

		// 通过人脸获取人员ids
		byte[] imgBytes = Base64.getDecoder().decode(imgBase64A);
		CommonResponse<List<Long>> response = faceAndIdCardService.getUserByFaceImgByte(imgBytes);
		if (!response.isSuccess()) {
			return CommonResponse.error("打卡失败，无法识别人脸");
		}
		Set<Long> ids = new HashSet<>(response.getData());
		logger.info("通过人脸获取人员ids--{}", JSON.toJSONString(ids));

		// 项目在场人员
		QueryParam param = new QueryParam();
		param.setPageSize(-1);
		param.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
		IPage<WorkRecordVO> page = leaderFetchWorkRecord(param);
		List<WorkRecordVO> records = page.getRecords();
		if (CollectionUtils.isEmpty(records)) {
			return CommonResponse.error("项目无在场人员");
		}

		// 取交集
		Set<Long> workerIds = records.stream().map(WorkRecordVO::getWorkerId).collect(Collectors.toSet());
		Map<Long, WorkRecordVO> workRecordVOMap = records.stream().collect(Collectors.toMap(WorkRecordVO::getWorkerId, Function.identity(), (key1, key2) -> key2));
		Sets.SetView<Long> workerId = Sets.intersection(ids, workerIds);
		if (workerId.isEmpty()) {
			return CommonResponse.error("无法识别该人员");
		}

		// 获取对应人员信息
		UnmodifiableIterator<Long> iterator = workerId.iterator();
		WorkRecordVO workRecordVO = workRecordVOMap.get(iterator.next());
		WorkerEntity workerEntity = workerService.selectById(workRecordVO.getWorkerId());

		return CommonResponse.success("人脸信息校验成功！", BeanMapper.map(workerEntity, WorkerVO.class));
	}


	/**
	 * 班组长：校验已存在的打卡记录
	 *
	 * @param workerId      项目花名册人员id
	 * @param projectId     项目id
	 * @param punchCardType 打卡类型：上班打卡、下班打卡
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link AttendanceLogVO}>>
	 */
	@Override
	public CommonResponse<List<AttendanceLogVO>> leaderCheckExistedLog(Long workerId, Long projectId, String punchCardType) {
		Assert.notNull(workerId, "项目花名册人员id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(punchCardType, "打卡类型不能为空！");

		LambdaQueryWrapper<AttendanceLogEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(AttendanceLogEntity::getTenantId, InvocationInfoProxy.getTenantid())
				.eq(AttendanceLogEntity::getProjectId, projectId)
				.eq(AttendanceLogEntity::getWorkerId, workerId)
				.eq(AttendanceLogEntity::getPunchCardType, punchCardType)
				.eq(AttendanceLogEntity::getPunchCardDate, DateUtil.formatDate(new Date()));
		if (START_WORK.equals(punchCardType)) {
			lambdaQuery.orderByDesc(AttendanceLogEntity::getStartWorkTime);
		} else {
			lambdaQuery.orderByDesc(AttendanceLogEntity::getEndWorkTime);
		}
		List<AttendanceLogEntity> logs = super.list(lambdaQuery);

		if (CollectionUtils.isNotEmpty(logs)) {
			return CommonResponse.error("今天已存在" + punchCardType + "，是否需要更新", BeanMapper.mapList(logs, AttendanceLogVO.class));
		}
		return CommonResponse.success("已存在的记录校验成功！");
	}


	/**
	 * 班组长：考勤打卡
	 *
	 * @param workerId       项目花名册人员id
	 * @param projectId      项目id
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link AttendanceLogVO}
	 */
	@Override
	public AttendanceLogVO leaderPunchCard(Long workerId, Long projectId, String punchCardType, String punchCardPlace, Integer remoteFlag) {
		Assert.notNull(workerId, "项目花名册人员id不能为空！");
		Assert.notNull(projectId, "项目id不能为空！");
		Assert.hasText(punchCardType, "打卡类型不能为空！");
		Assert.hasText(punchCardPlace, "打卡地点不能为空！");
		Assert.notNull(remoteFlag, "是否是异地打卡不能为空！");

		// 1、新增考勤打卡日志
		LambdaQueryWrapper<WorkRecordEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(WorkRecordEntity::getProjectId, projectId)
				.eq(WorkRecordEntity::getWorkerId, workerId)
				.eq(WorkRecordEntity::getEnterExitState, 1);
		WorkRecordEntity workRecordEntity = workRecordService.getOne(lambdaQuery);

		AttendanceLogEntity attendanceLogEntity = WorkRecordMapper.INSTANCE.transformAttendanceLogEntity(workRecordEntity);

		attendanceLogEntity.setPunchCardDate(new Date());
		if (START_WORK.equals(punchCardType)) {
			attendanceLogEntity.setStartWorkTime(new Date());
			attendanceLogEntity.setStartWorkPlace(punchCardPlace);
		} else {
			attendanceLogEntity.setEndWorkTime(new Date());
			attendanceLogEntity.setEndWorkPlace(punchCardPlace);
		}
		attendanceLogEntity.setRemoteFlag(remoteFlag);
		if (remoteFlag == 0) {
			attendanceLogEntity.setAttendanceStatus("正常");
		} else {
			attendanceLogEntity.setAttendanceStatus("异常");
		}
		attendanceLogEntity.setPunchCardType(punchCardType);
		super.saveOrUpdate(attendanceLogEntity, false);

		// 2、查询考勤打卡记录
		LambdaQueryWrapper<AttendanceEntity> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(AttendanceEntity::getTenantId, InvocationInfoProxy.getTenantid());
		wrapper.eq(AttendanceEntity::getProjectId, projectId);
		wrapper.eq(AttendanceEntity::getWorkerId, workerId);
		wrapper.eq(AttendanceEntity::getPunchCardDate, DateUtil.formatDate(new Date()));
		AttendanceEntity attendanceEntity = attendanceService.getOne(wrapper);
		if (attendanceEntity == null) {
			AttendanceEntity entity = com.ejianc.business.labor.common.dtoMapper.AttendanceLogMapper.INSTANCE.transformAttendanceEntity(attendanceLogEntity);
			entity.setDuration(0L);
			attendanceService.saveOrUpdate(entity, false);
		} else {
			if (START_WORK.equals(punchCardType)) {
				attendanceEntity.setStartWorkTime(attendanceLogEntity.getStartWorkTime());
				attendanceEntity.setStartWorkPlace(attendanceLogEntity.getStartWorkPlace());
				if (attendanceEntity.getEndWorkTime() != null) {
					attendanceEntity.setDuration(DateUtil.between(attendanceEntity.getStartWorkTime(), attendanceEntity.getEndWorkTime(), DateUnit.HOUR));
				}
			} else {
				attendanceEntity.setEndWorkTime(attendanceLogEntity.getEndWorkTime());
				attendanceEntity.setEndWorkPlace(attendanceLogEntity.getEndWorkPlace());
				if (attendanceEntity.getStartWorkTime() != null) {
					attendanceEntity.setDuration(DateUtil.between(attendanceEntity.getStartWorkTime(), attendanceEntity.getEndWorkTime(), DateUnit.HOUR));
				}
			}
			attendanceEntity.setAttendanceStatus(attendanceLogEntity.getAttendanceStatus());
			attendanceEntity.setRemoteFlag(attendanceLogEntity.getRemoteFlag());
			attendanceService.saveOrUpdate(attendanceEntity, false);
		}

		return BeanMapper.map(attendanceLogEntity, AttendanceLogVO.class);
	}
}
