package com.ejianc.business.market.controller.api;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.market.bean.ProjectRegisterEntity;
import com.ejianc.business.market.service.IProjectRegisterService;
import com.ejianc.business.market.vo.FilterOrgIdVo;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.market.vo.SupplierCooperateVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/api/project/")
public class ProjectApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IProjectRegisterService projectRegisterService;
    @Autowired
    private SessionManager sessionManager;
    private final static String INDEX_NAME_PAY_CONTRACT = "pay_contract";
    @Autowired(required = false)
    private RestHighLevelClient client;
    private final static Integer QUERY_TIMEOUT = 60;

    @GetMapping("queryProjectBySourceId")
    public CommonResponse<ProjectRegisterVO> queryProjectBySourceId(@RequestParam(value = "sourceId") String sourceId) {
        ProjectRegisterVO vo = new ProjectRegisterVO();
        QueryWrapper<ProjectRegisterEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("dr", 0);
        queryWrapper.eq("source_id", sourceId);
        ProjectRegisterEntity entity = projectRegisterService.getOne(queryWrapper);
        if (entity != null) {
            vo = BeanMapper.map(entity, ProjectRegisterVO.class);
        }
        return CommonResponse.success("查询成功！", vo);
    }

    @GetMapping("queryProjectDetail")
    public CommonResponse<ProjectRegisterVO> queryProjectDetail(@RequestParam(value = "projectId") Long projectId) {
        ProjectRegisterVO vo = new ProjectRegisterVO();
        ProjectRegisterEntity entity = projectRegisterService.selectById(projectId);
        if (entity != null) {
            vo = BeanMapper.map(entity, ProjectRegisterVO.class);
        }
        return CommonResponse.success("查询成功！", vo);
    }


    /**
     * @Author mrsir_wxp
     * @Date 2020/8/21 根据项目部id查询项目信息
     * @Description queryProjectByProjectDepartmentId
     * @Param [sourceId]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.market.vo.ProjectRegisterVO>
     */
    @GetMapping("queryProjectByProjectDepartmentId")
    public CommonResponse<ProjectRegisterVO> queryProjectByProjectDepartmentId(@RequestParam(value = "projectDepartmentId") Long projectDepartmentId) {
        ProjectRegisterVO vo = new ProjectRegisterVO();
        QueryWrapper<ProjectRegisterEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("project_department_id", projectDepartmentId);
        ProjectRegisterEntity entity = projectRegisterService.getOne(queryWrapper);
        if (entity != null) {
            vo = BeanMapper.map(entity, ProjectRegisterVO.class);
        }
        return CommonResponse.success("查询成功！", vo);
    }

    /**
     * 根据项目ID查询项目列表
     *
     * @param
     * @return
     */
    @GetMapping("queryProjectByIds")
    public CommonResponse<List<ProjectRegisterVO>> queryProjectByIds(@RequestParam(value = "ids") List<Long> ids) {
        List<ProjectRegisterVO> vos = new ArrayList<>();
        QueryWrapper<ProjectRegisterEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("dr", 0);
        queryWrapper.in("id", ids);
        List<ProjectRegisterEntity> entitys = projectRegisterService.list(queryWrapper);
        if (entitys != null) {
            vos = BeanMapper.mapList(entitys, ProjectRegisterVO.class);
        }
        return CommonResponse.success("查询成功！", vos);
    }

    /**
     * 根据条件查询对应项目ID列表
     *
     * @param projectRange 1-全部项目，2-在建项目，3-完工项目，4-本年新签项目
     * @return
     */
    @GetMapping(value = "getProjectIdsByProperties")
    CommonResponse<List<Long>> getProjectIdsByProperties(@RequestParam(value = "projectRange") int projectRange) {
        QueryParam queryParam = new QueryParam();
        //查询本租户的项目
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        //启用状态
        queryParam.getParams().put("validStatus", new Parameter(QueryParam.EQ, 0));
        boolean queryNewSigned = false;
        if (projectRange == 2 || projectRange == 3) {
            queryParam.getParams().put("status", new Parameter(QueryParam.EQ, projectRange));
        } else if (projectRange == 4) {
            queryNewSigned = true;
        }
        return CommonResponse.success(projectRegisterService.queryProIdsByProperties(queryParam, queryNewSigned));
    }

    /**
     * 根据条件查询对应项目ID列表
     *
     * @param projectRange 1-全部项目，2-在建项目，3-完工项目，4-本年新签项目
     * @return
     */
    @GetMapping(value = "getProjectIdsByEngineeringType")
    CommonResponse<List<Long>> getProjectIdsByEngineeringType(@RequestParam(value = "projectRange") int projectRange, @RequestParam(value = "engineeringType", required = false) Long engineeringType) {
        QueryParam queryParam = new QueryParam();
        //查询本租户的项目
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        //启用状态
        queryParam.getParams().put("validStatus", new Parameter(QueryParam.EQ, 0));
        boolean queryNewSigned = false;
        if (projectRange == 2 || projectRange == 3) {
            queryParam.getParams().put("status", new Parameter(QueryParam.EQ, projectRange));
        } else if (projectRange == 4) {
            queryNewSigned = true;
        }
        if (null != engineeringType) {
            queryParam.getParams().put("engineeringType", new Parameter(QueryParam.EQ, engineeringType));
        }
        return CommonResponse.success(projectRegisterService.queryProIdsByProperties(queryParam, queryNewSigned));
    }

    /**
     * 供方协同---用户获取有权限的项目，为空代表全部项目的权限
     *
     * @return
     */
    @GetMapping(value = "queryProjectByUserId")
    public CommonResponse<List<ProjectRegisterVO>> queryProjectByUserId(@RequestParam(value = "userId") Long userId) {
        return projectRegisterService.queryProjectByUserId(userId);
    }

    /***
     * @Author mrsir_wxp
     * @Date 2021/2/1 查询本下启用的项目，参数为需要查询的条数，默认十条
     * @Description queryProjectList
     * @Param [num]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.util.List < com.ejianc.business.market.vo.ProjectRegisterVO>>
     */
    @GetMapping(value = "queryProjectList")
    public CommonResponse<List<ProjectRegisterVO>> queryProjectList(@RequestParam(value = "num", defaultValue = "10") int num) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageSize(num);
        queryParam.setPageIndex(1);
        //查询本租户的项目
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        CommonResponse<List<OrgVO>> response = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        List<OrgVO> orgVOList = response.getData();
        if (!response.isSuccess()) {
            return CommonResponse.error("本下组织查询失败！");
        }
        queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, orgVOList.stream().map(OrgVO::getId).collect(Collectors.toList())));
        /** 项目状态：0-启用 */
        queryParam.getParams().put("validStatus", new Parameter(QueryParam.EQ, 0));
        /** 创建时间倒序 */
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);
        IPage<ProjectRegisterEntity> page = projectRegisterService.queryPage(queryParam, false);
        System.out.println("查询本下启用的项目，参数为需要查询的条数，默认十条 page.getTotal = " + page.getTotal());
        List<ProjectRegisterVO> res = BeanMapper.mapList(page.getRecords(), ProjectRegisterVO.class);
        System.out.println("查询本下启用的项目，参数为需要查询的条数，默认十条 res.size = " + res.size());
        return CommonResponse.success(res);
    }


    @PostMapping(value = "queryProjectPage")
    public CommonResponse<Page<ProjectRegisterVO>> queryProjectPage(@RequestBody QueryParam queryParam) {
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("projectManagerName");
        //查询本租户的项目
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        /** 数据隔离 本下 */
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        logger.info(">>>>>>>>>>>>>>>>>>>>>>>>>>>>queryProjectPage>>>>>>>>>>>>>>>>>>>>>>>>>");
        logger.info("authOrgIds:{}", authOrgIds);
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            logger.info("authResponse.getData().size():{}", authOrgIds, authResponse.getData() == null ? null : authResponse.getData().size());
            queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        logger.info("<<<<<<<<<<<<<<<<<<<<<<<<<<<<queryProjectPage<<<<<<<<<<<<<<<<<<<<<<<<<");

        /** 项目状态：0-启用 */
        queryParam.getParams().put("validStatus",new Parameter(QueryParam.EQ, 0));
        /** 创建时间倒序 */
        queryParam.getOrderMap().putIfAbsent("createTime", QueryParam.DESC);
        IPage<ProjectRegisterEntity> pageData = projectRegisterService.queryPage(queryParam,false);
        Page<ProjectRegisterVO> result = new Page<>(pageData.getCurrent(), pageData.getSize(), pageData.getTotal());
        if(pageData.getTotal()>0){
            List<ProjectRegisterVO> list = BeanMapper.mapList(pageData.getRecords(),ProjectRegisterVO.class);
            result.setRecords(list);
        }
        return CommonResponse.success(result);
    }

    @PostMapping(value = "queryProjectListByStatus")
    public CommonResponse<List<Long>> queryProjectListByStatus(@RequestBody ProjectRegisterVO vo) {

        QueryParam queryParam = new QueryParam();

        List<Long> orgIds = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));
        //查询本租户的项目
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("status", new Parameter(QueryParam.EQ, vo.getStatus()));
        List<ProjectRegisterEntity> projectRegisterEntities = projectRegisterService.queryList(queryParam);

        List<Long> collect = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(projectRegisterEntities)) {
            collect = projectRegisterEntities.stream().map(i -> i.getId()).collect(Collectors.toList());
        }
        return CommonResponse.success(collect);


    }

    /**
     * 查询本下所有满足匹配条件的项目列表
     *
     * @param filterOrgVo
     * @return
     */
    @PostMapping(value = "queryFilterProjects")
    public CommonResponse<List<ProjectRegisterVO>> queryFilterProjects(@RequestBody FilterOrgIdVo filterOrgVo) {
        List<ProjectRegisterVO> resp = new ArrayList<>();
        Long orgId = filterOrgVo.getOrgId() == null ? InvocationInfoProxy.getOrgId() : filterOrgVo.getOrgId();
        List<Long> orgIds = new ArrayList<>();
        QueryParam queryParam = new QueryParam();

        CommonResponse<OrgVO> orgResp = orgApi.getOneById(orgId);
        if (!orgResp.isSuccess()) {
            return CommonResponse.error("获取组织信息失败！");
        }
        if (OrgVO.ORG_TYPE_DEPARTMENT.equals(orgResp.getData().getOrgType())) { //项目部
            queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.EQ, orgId));
        } else { //非项目部组织
            CommonResponse<List<OrgVO>> withoutProjectDeptRes = orgApi.findChildrenByParentIdWithoutProjectDept(orgId);
            if (withoutProjectDeptRes.isSuccess() && CollectionUtils.isNotEmpty(withoutProjectDeptRes.getData())) {
                orgIds.addAll(withoutProjectDeptRes.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
            }
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));
        }

        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        if (filterOrgVo.getProjectStatus() != null) {
            if (filterOrgVo.getProjectStatus().contains(",")) {
                //多个状态逗号分隔
                queryParam.getParams().put("status", new Parameter(QueryParam.IN, filterOrgVo.getProjectStatus()));
            } else {
                queryParam.getParams().put("status", new Parameter(QueryParam.EQ, filterOrgVo.getProjectStatus()));
            }
        }
        if (filterOrgVo.getDateIn() != null) {
            String dateIn = filterOrgVo.getDateIn();
            if (dateIn.equals("near4Year")) {
                //查询近四年的数据
                queryParam.getParams().put("", new Parameter(QueryParam.SQL, " year(sign_date)>year(now())-3"));
            } else if (!(dateIn.equals("all"))) {
                queryParam.getParams().put("", new Parameter(QueryParam.SQL, " year(sign_date)=" + filterOrgVo.getDateIn()));
            }
        }
        if (StringUtils.isNotBlank(filterOrgVo.getSearchText())) {
            queryParam.setSearchText(filterOrgVo.getSearchText());
            if (CollectionUtils.isNotEmpty(filterOrgVo.getSearchFields())) {
                queryParam.getFuzzyFields().addAll(filterOrgVo.getSearchFields());
            } else {
                //默认模糊匹配根据名称搜索
                queryParam.getFuzzyFields().add("name");
            }
        }

        List<ProjectRegisterEntity> projectRegisterEntities = projectRegisterService.queryList(queryParam);
        if (CollectionUtils.isNotEmpty(projectRegisterEntities)) {
            resp = BeanMapper.mapList(projectRegisterEntities, ProjectRegisterVO.class);
        }
        return CommonResponse.success(resp);
    }

    /**
     * 返回一个orgIds 根据条件过滤掉对应的orgId
     *
     * @param filterOrgVo
     * @return
     */
    @PostMapping(value = "queryFilterOrgIds")
    public CommonResponse<List<Long>> queryFilterOrgIds(@RequestBody FilterOrgIdVo filterOrgVo) {
        Long orgId = filterOrgVo.getOrgId() == null ? InvocationInfoProxy.getOrgId() : filterOrgVo.getOrgId();
        List<Long> orgIds = new ArrayList<>();
        CommonResponse<List<OrgVO>> withoutProjectDeptRes = orgApi.findChildrenByParentIdWithoutProjectDept(orgId);
        if (withoutProjectDeptRes.isSuccess() && CollectionUtils.isNotEmpty(withoutProjectDeptRes.getData())) {
            orgIds.addAll(withoutProjectDeptRes.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        }

        QueryParam queryParam = new QueryParam();
        CommonResponse<List<OrgVO>> response = orgApi.findChildrenByParentId(orgId);
        List<OrgVO> orgVOList = response.getData();
        if (!response.isSuccess()) {
            return CommonResponse.error("本下组织查询失败！");
        }
        queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, orgVOList.stream().map(OrgVO::getId).collect(Collectors.toList())));

        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("valid_status", new Parameter(QueryParam.EQ, 0));

        //状态处理
        if (filterOrgVo.getProjectStatus() != null) {
            String projectStatus = filterOrgVo.getProjectStatus();
            if (!(projectStatus.equals("all"))) {
                queryParam.getParams().put("status", new Parameter(QueryParam.EQ, filterOrgVo.getProjectStatus()));
            }
        } else {
            queryParam.getParams().put("status", new Parameter(QueryParam.EQ, 2));
        }
        if (filterOrgVo.getDateIn() != null) {
            String dateIn = filterOrgVo.getDateIn();
            if (dateIn.equals("near4Year")) {
                //查询近四年的数据
                queryParam.getParams().put("", new Parameter(QueryParam.SQL, " year(sign_date)>year(now())-3"));
            } else if (!(dateIn.equals("all"))) {
                queryParam.getParams().put("", new Parameter(QueryParam.SQL, " year(sign_date)=" + filterOrgVo.getDateIn()));
            }
        } else {
            //默认今年
            queryParam.getParams().put("", new Parameter(QueryParam.SQL, " year(sign_date)>year(now())"));
        }
        List<ProjectRegisterEntity> projectRegisterEntities = projectRegisterService.queryList(queryParam);
        List<Long> projectDepartmentIds = projectRegisterEntities.stream().map(i -> i.getProjectDepartmentId()).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(projectDepartmentIds)) {
            orgIds.addAll(projectDepartmentIds);
        }
        if (CollectionUtils.isEmpty(orgIds)) {
            orgIds.add(1L);
        }
        return CommonResponse.success(orgIds);
    }

    /**
     * @Author zhangwx
     * @Date 2021/06/07 根据组织主键查询本下所有项目
     * @Description queryProjectByProjectDepartmentId
     * @Param [orgId]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.util.List < com.ejianc.business.market.vo.ProjectRegisterVO>>
     */
    @GetMapping(value = "queryChildrenProjectByOrgId")
    CommonResponse<List<ProjectRegisterVO>> queryChildrenProjectByOrgId(@RequestParam(value = "orgId") Long orgId) {
        List<Long> departmentIdList = new ArrayList<>();
        List<ProjectRegisterVO> result = new ArrayList<>();
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(orgId);
        if (childOrgResp.isSuccess() && CollectionUtils.isNotEmpty(childOrgResp.getData())) {
            for (OrgVO orgVO : childOrgResp.getData()) {
                if (null != orgVO.getOrgType() && orgVO.getOrgType() == 5) {
                    departmentIdList.add(orgVO.getId());
                }
            }
        }
        if (CollectionUtils.isNotEmpty(departmentIdList)) {
            LambdaQueryWrapper<ProjectRegisterEntity> queryWrapper = new LambdaQueryWrapper();
            queryWrapper.in(ProjectRegisterEntity::getProjectDepartmentId, departmentIdList);
            List<ProjectRegisterEntity> list = projectRegisterService.list(queryWrapper);
            if (CollectionUtils.isNotEmpty(list)) {
                result = BeanMapper.mapList(list, ProjectRegisterVO.class);
            }
        }
        return CommonResponse.success(result);
    }

    /**
     * 根据状态返回可用的 org id
     *
     * @param status
     * @return
     */
    @PostMapping(value = "queryFilterOrgIdsBuStatus")
    public CommonResponse<List<Long>> queryFilterOrgIdsBuStatus(@RequestParam(value = "status") String status) {
        Long orgId = InvocationInfoProxy.getOrgId();
        List<Long> orgIds = new ArrayList<>();
        CommonResponse<List<OrgVO>> withoutProjectDeptRes = orgApi.findChildrenByParentIdWithoutProjectDept(orgId);
        if (withoutProjectDeptRes.isSuccess() && CollectionUtils.isNotEmpty(withoutProjectDeptRes.getData())) {
            orgIds.addAll(withoutProjectDeptRes.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        }

        QueryParam queryParam = new QueryParam();
        CommonResponse<List<OrgVO>> response = orgApi.findChildrenByParentId(orgId);
        List<OrgVO> orgVOList = response.getData();
        if (!response.isSuccess()) {
            return CommonResponse.error("本下组织查询失败！");
        }
        queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, orgVOList.stream().map(OrgVO::getId).collect(Collectors.toList())));

        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("status", new Parameter(QueryParam.EQ, status));
        List<ProjectRegisterEntity> projectRegisterEntities = projectRegisterService.queryList(queryParam);
        List<Long> projectDepartmentIds = projectRegisterEntities.stream().map(i -> i.getProjectDepartmentId()).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(projectDepartmentIds)) {
            orgIds.addAll(projectDepartmentIds);
        }
        if (CollectionUtils.isEmpty(orgIds)) {
            orgIds.add(1L);
        }
        return CommonResponse.success(orgIds);
    }

    /**
     * 根据组织主键查询本下项目地址，过滤条件项目状态、关注类型
     *
     * @return
     */
    @GetMapping(value = "queryProjects")
    CommonResponse<List<Map>> queryProjects(@RequestParam(value = "orgId") Long orgId,
                                            @RequestParam(value = "statusList", required = false) List<String> statusList,
                                            @RequestParam(value = "focusTypeList", required = false) List<String> focusTypeList,
                                            @RequestParam(value = "engineeringTypeList", required = false) List<Long> engineeringTypeList,
                                            @RequestParam(value = "searchText", required = false) String searchText) {
        List<Long> departmentIdList = new ArrayList<>();
        List<Map> result = new ArrayList<>();
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(orgId);
        if (childOrgResp.isSuccess() && CollectionUtils.isNotEmpty(childOrgResp.getData())) {
            for (OrgVO orgVO : childOrgResp.getData()) {
                if (null != orgVO.getOrgType() && orgVO.getOrgType() == 5) {
                    departmentIdList.add(orgVO.getId());
                }
            }
        }

        if (CollectionUtils.isNotEmpty(departmentIdList)) {
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("dr", new Parameter(QueryParam.EQ, 0));
            queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIdList));
            if (statusList != null && statusList.size() > 0) {
                queryParam.getParams().put("status", new Parameter(QueryParam.IN, statusList));
            }
            if (focusTypeList != null && focusTypeList.size() > 0) {
                queryParam.getParams().put("focusType", new Parameter(QueryParam.IN, focusTypeList));
            }
            if (engineeringTypeList != null && engineeringTypeList.size() > 0) {
                queryParam.getParams().put("engineeringType", new Parameter(QueryParam.IN, engineeringTypeList));
            }
            if (StringUtils.isNotBlank(searchText)) {
                queryParam.setSearchText(searchText);
                queryParam.getFuzzyFields().add("name");
                queryParam.getFuzzyFields().add("projectManagerName");
            }
            queryParam.getOrderMap().put("createTime", "desc");
            List<ProjectRegisterEntity> list = projectRegisterService.queryList(queryParam);
            if (CollectionUtils.isNotEmpty(list)) {
                for (ProjectRegisterEntity entity : list) {
                    Map map = new HashMap();
                    map.put("id", entity.getId());
                    map.put("name", entity.getName());
                    map.put("lat", entity.getLat());
                    map.put("lng", entity.getLng());
                    map.put("focusType", entity.getFocusType());
                    map.put("projectDepartmentId", entity.getProjectDepartmentId());
                    map.put("engineeringTypeName", entity.getEngineeringTypeName());
                    map.put("projectManagerName", entity.getProjectManagerName());
                    result.add(map);
                }
            }
        }
        return CommonResponse.success(result);
    }

    @PostMapping(value = "querySupplierCooperate")
    public CommonResponse<Map<Long, List<SupplierCooperateVO>>> querySupplierCooperate(@RequestBody List<Long> supplierIds) {
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));
        boolQuery.must(QueryBuilders.termsQuery("supplierId", supplierIds));
        SearchRequest searchRequest = new SearchRequest(INDEX_NAME_PAY_CONTRACT);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        List<SupplierCooperateVO> list = new ArrayList<>();
        Map<Long, List<SupplierCooperateVO>> map = null;
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, SupplierCooperateVO.class));
            }
            map = list.stream().collect(Collectors.groupingBy(SupplierCooperateVO::getSupplierId));
        } catch (IOException e) {
            e.printStackTrace();
        }

        return CommonResponse.success(map);
    }

    @PostMapping(value = "queryAllProj")
    public CommonResponse<List<ProjectRegisterVO>> queryAllProj(@RequestBody QueryParam queryParam) {
        Parameter tenantIdListPram = queryParam.getParams().get("tenantIdList");
        //传入租户ID集合时查询传入租户的项目
        if (null != tenantIdListPram && null != tenantIdListPram.getValue()) {
            queryParam.getParams().put("tenantId", new Parameter(QueryParam.IN, tenantIdListPram.getValue()));
        }else {
            //查询本租户的项目
            queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        }
        queryParam.getParams().remove("tenantIdList");

        Parameter withoutOrgFlag = queryParam.getParams().get("withoutOrgFlag");
        if (null == withoutOrgFlag || BooleanUtils.isFalse((Boolean) withoutOrgFlag.getValue())) {
            /** 数据隔离 本下 */
            UserContext userContext = sessionManager.getUserContext();
            String authOrgIds = userContext.getAuthOrgIds();
            logger.info(">>>>>>>>>>>>>>>>>>>>>>>>>>>>queryProjectPage>>>>>>>>>>>>>>>>>>>>>>>>>");
            logger.info("authOrgIds:{}", authOrgIds);
            if (StringUtils.isNotEmpty(authOrgIds)) {
                CommonResponse<List<OrgVO>> authResponse =
                        orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                                collect(Collectors.toList()));
                logger.info("authResponse.getData().size():{}", authOrgIds, authResponse.getData() == null ? null : authResponse.getData().size());
                queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
            } else {
                queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
            }
            logger.info("<<<<<<<<<<<<<<<<<<<<<<<<<<<<queryProjectPage<<<<<<<<<<<<<<<<<<<<<<<<<");
        }
        queryParam.getParams().remove("withoutOrgFlag");

        List<ProjectRegisterEntity> projList = projectRegisterService.queryList(queryParam, false);
        return CommonResponse.success(CollectionUtils.isNotEmpty(projList) ? BeanMapper.mapList(projList, ProjectRegisterVO.class) : null);
    }


    @PostMapping("getProjectMapByCodes")
    CommonResponse<Map<String, ProjectRegisterVO>> getProjectMapByCodes(@RequestBody List<String> codes) {
        Map<String, ProjectRegisterVO> map = new HashMap<>();
        if (CollectionUtils.isNotEmpty(codes)) {
            LambdaQueryWrapper<ProjectRegisterEntity> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(ProjectRegisterEntity::getTenantId, InvocationInfoProxy.getTenantid());
            queryWrapper.in(ProjectRegisterEntity::getCode, codes);
            List<ProjectRegisterEntity> list = projectRegisterService.list(queryWrapper);
            List<ProjectRegisterVO> projectRegisterVOS = BeanMapper.mapList(list, ProjectRegisterVO.class);
            if (CollectionUtils.isNotEmpty(projectRegisterVOS)) {
                map = projectRegisterVOS.stream().collect(Collectors.toMap(ProjectRegisterVO::getCode, Function.identity(), (p1, p2) -> p1));
            }
        }
        return CommonResponse.success("查询成功！", map);
    }


    @GetMapping("changeProjectStage")
    CommonResponse<String> changeProjectStage(@RequestParam(value = "projectId") Long projectId, @RequestParam(value = "projectStage") String projectStage) {
        LambdaUpdateWrapper<ProjectRegisterEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(ProjectRegisterEntity::getId, projectId);
        updateWrapper.set(ProjectRegisterEntity::getProjectStage, projectStage);
        if ("完工".equals(projectStage)) {
            updateWrapper.set(ProjectRegisterEntity::getStatus, "3");
        }
        projectRegisterService.update(updateWrapper);
        return CommonResponse.success("操作成功！");
    }


}
