package com.ejianc.business.market.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.market.bean.ProjectRegisterEntity;
import com.ejianc.business.market.service.IProjectRegisterService;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.market.vo.SynchroDataParam;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.usercenter.api.IThirdSystemApi;
import com.ejianc.foundation.usercenter.vo.ThirdSystemVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;

/**
 * 同步第三方系统项目信息服务
 *
 * @author CJ
 * @Description:
 * @date 2020/8/5 20:11
 */
@RestController
@RequestMapping(value = "/sync/project/")
public class SynchroDataController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IProjectRegisterService projectRegisterService;

    @Autowired
    private IThirdSystemApi thirdSystemApi;

    @PostMapping(value = "baseInfo")
    public CommonResponse<String> syncProject(@RequestBody SynchroDataParam data) {
        logger.info("收到待同步项目部信息：【{}】", JSONObject.toJSONString(data));
        ProjectRegisterVO project = transToProjectVO(data);

        switch (data.getSyncType()) {
            case 1:
                //新增
            case 2:
                //更新
                projectRegisterService.saveSyncProject(project);
                break;
            case 3:
                //更新项目状态
                projectRegisterService.changeValidStatus(project);
                break;
        }

        return CommonResponse.success("同步项目信息成功！");
    }

    private ProjectRegisterVO transToProjectVO(SynchroDataParam project) {
        ProjectRegisterVO vo = new ProjectRegisterVO();
        Integer[] typeArr = new Integer[]{1, 2, 3};
        boolean materialVerification = false;

        if (StringUtils.isBlank(project.getId())) {
            logger.error("同步项目信息失败，项目业务主键【id】为空。");
            throw new BusinessException("同步项目信息失败，项目业务主键【id】为空。");
        }

        if (!Arrays.asList(typeArr).contains(project.getSyncType())) {
            logger.error("同步项目信息失败，错误的同步类型【syncType: {}】", project.getSyncType());
            throw new BusinessException("同步项目信息失败，错误的同步类型【syncType: " + project.getSyncType() + "】");
        }
        if (StringUtils.isBlank(project.getSystemCode())) {
            logger.error("同步项目信息失败，数据来源系统编码【systemCode】为空");
            throw new BusinessException("同步项目信息失败，数据来源系统编码【systemCode】为空");
        }

        CommonResponse<ThirdSystemVO> thirdSystemResp = thirdSystemApi.getOneByCode(project.getSystemCode());
        if (!thirdSystemResp.isSuccess()) {
            logger.error("同步项目信息失败，查询编码【systemCode: {}】对应的系统信息失败。", project.getSystemCode());
            throw new BusinessException("同步项目信息失败，查询编码【systemCode: " + project.getSystemCode() + "】对应的系统信息失败。");
        }
        ThirdSystemVO system = thirdSystemResp.getData();
        if (null == system) {
            logger.error("同步项目信息失败，编码【systemCode: {}】无匹配的系统信息。", project.getSystemCode());
            throw new BusinessException("同步项目信息失败，编码【systemCode: " + project.getSystemCode() + "】无匹配的系统信息。");
        }
        vo.setTenantId(system.getTenantId());
        vo.setSystemCode(system.getCode());
        vo.setSystemId(system.getId());

        /* modify by CJ 根据sourceId来做唯一甄别 */
        QueryWrapper<ProjectRegisterEntity> query = new QueryWrapper<>();
        query.eq("source_id", project.getId());
        query.eq("dr", BaseVO.DR_UNDELETE);
        query.eq("tenant_id", vo.getTenantId());
        ProjectRegisterEntity dbEntity = projectRegisterService.getOne(query);

        //非停用
        if (3 != project.getSyncType()) {
            if ("1".equals(project.getType())) {
                //使用材料核销
                materialVerification = true;
            }

            //项目类型设置为自营
            project.setType("1");

            if (StringUtils.isBlank(project.getName())) {
                logger.error("项目同步失败，项目名称为空【name】为空");
                throw new BusinessException("项目同步失败，项目名称为空【name】为空");
            }

            if (StringUtils.isBlank(project.getOrgId())) {
                logger.error("项目同步失败，项目所属组织Id【orgId】为空。");
                throw new BusinessException("项目同步失败，项目所属组织Id【orgId】为空。");
            }

            CommonResponse<OrgVO> orgResp = orgApi.findBySourceId(project.getOrgId());
            if (!orgResp.isSuccess()) {
                logger.error("项目同步失败，项目所属组织失败或组织在系统中不存在。");
                throw new BusinessException("项目同步失败，项目所属组织失败或组织在系统中不存在。");
            }

            OrgVO org = orgResp.getData();

            if(null == dbEntity && (2 == project.getSyncType().intValue())) {
                logger.info("服务调用方式为更新，但是该项目信息在系统中不存在，将服务方式改为新增！");
                project.setSyncType(1);

            } else if (null != dbEntity) {
                vo = BeanMapper.map(dbEntity, ProjectRegisterVO.class);

                vo.setTenantId(system.getTenantId());
                vo.setSystemCode(system.getCode());
                vo.setSystemId(system.getId());
            }

            vo.setOrgName(org.getName());
            vo.setOrgId(org.getId());
            vo.setName(project.getName());
            vo.setType(project.getType());
            vo.setSourceId(project.getId());
            vo.setMaterialVerification(materialVerification);
            if(StringUtils.isNotBlank(project.getCode())) {
                vo.setCode(project.getCode());
            }
            //编码重复校验
            if(null != dbEntity && !dbEntity.getCode().equals(vo.getCode())) {
                //编码重复校验
                if (projectRegisterService.codeUsedCheck(vo)) {
                    logger.error("项目同步失败，项目编码重复。");
                    throw new BusinessException("项目同步失败，项目编码重复。");
                }
            }
        } else {
            if(null == dbEntity) {
                logger.error("项目停用失败，系统中不存在匹配的项目[sourceId-{},code-{}]。", project.getId(), project.getCode());
                throw new BusinessException("项目停用失败，项目在系统中不存在或已被删除。");
            }
            //项目停用
            vo = BeanMapper.map(dbEntity, ProjectRegisterVO.class);
            vo.setValidStatus("1");
        }

        //       Modify by wxh 增加关注程度对照
        if(StringUtils.isNotBlank(project.getRailtype())){
            vo.setFocusType(transFocusType(project.getRailtype()));
        }
//        end

        // 新项目标识，根据标识过滤物资档案
        if(StringUtils.isNotEmpty(project.getNewState())){
            vo.setNewState(project.getNewState());
        }

        return vo;
    }

    /**
     * 关注级别转换 wxh
     * 重要程度   1-局重点关注  2-二级单位关注  3-非重点项目  4-局重点监控 5-二级单位重点监控
     * lp字段含义：0-普通项目 1-工程局重点关注 2-工程局重点监控 3-二级单位重点关注 4-二级单位重点监控
     *  对照：pm>>lp:  1>>1  2>>3   3>>0   4>>2  5>>4
     * @param railtype
     * @return
     */
    private String transFocusType(String railtype) {
        String focusType = null;
        switch (railtype) {
            case "1":
                focusType = "1";
                break;
            case "2":
                focusType = "3";
                break;
            case "3":
                focusType = "0";
                break;
            case "4":
                focusType = "2";
                break;
            case "5":
                focusType = "4";
                break;
            default:
                focusType = "0";
        }
        return focusType;
    }
}
