package com.ejianc.foundation.message.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.message.bean.MessageEntity;
import com.ejianc.foundation.message.bean.MsgTypeEntity;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.usercenter.vo.UserVO;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.foundation.message.service.IMessageService;
import com.ejianc.foundation.message.service.IMsgTypeService;
import com.ejianc.foundation.message.vo.MessageVO;
import com.ejianc.foundation.message.vo.PageParam;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;

import javax.servlet.http.HttpServletResponse;

@RestController
@RequestMapping("/message/")
public class MessageController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IMessageService messageService;

    @Autowired
    private IAttachmentApi attachmentApi;

    @Autowired
    private IUserApi userApi;

    @Autowired
    private IMsgTypeService msgTypeService;
    
    /**
     * 消息列表页数据查询
     *
     * @return
     */
    @PostMapping("page")
    public CommonResponse<JSONObject> pageList(@RequestBody PageParam pageParam) {
        JSONObject resp = new JSONObject();
        JSONObject data = new JSONObject();
        List<MessageVO> voList = null;
        pageParam.setTenantId(InvocationInfoProxy.getTenantid());
        long count = messageService.queryCount(pageParam);
        if(count > 0) {
            voList = messageService.queryList(pageParam);
        }else{
        	voList = new ArrayList<>();
        }
        data.put("total", count);
        data.put("pageSize", pageParam.getPageSize());
        data.put("pageNum", pageParam.getPageNum());
        data.put("records", voList);
        resp.put("data", data);
        return CommonResponse.success(resp);
    }

    /**
     * 删除用户接收到的消息，并删除对应的未读消息记录
     *
     * @param messageIds
     * @return
     */
    @PostMapping("delMessages")
    public CommonResponse<String> delMessages(@RequestBody List<MessageVO> messageList) {
    	messageService.delMessage(messageList);
        return CommonResponse.success("删除成功！");
    }

    /**
     * 将指定的用户未读消息设置成已读状态
     *
     * @return
     */
    @PostMapping("updateMessagesStatus")
    public CommonResponse<String> setMsgReadedStatus(@RequestBody List<Long> messageIds) {
        messageService.updateMessageStatus(messageIds);
        return CommonResponse.success("操作成功！");
    }


    /**
     * 根据Id查询用户通讯消息
     *
     * @param msgId
     * @return
     */
    @GetMapping(value = "getUserMsg")
    public CommonResponse<JSONObject> getUserMsg(@RequestParam(value = "msgId") Long msgId) {
        JSONObject resp = null;
        MessageEntity messageEntity = messageService.selectById(msgId);
        resp = BeanMapper.map(messageEntity, JSONObject.class);
        CommonResponse<List<AttachmentVO>> attachResp = attachmentApi.queryListBySourceId(msgId, null, null, null);
        if(attachResp.isSuccess()) {
            resp.put("attachmentList", attachResp.getData());
        } else {
            logger.error("查询消息【id-{}】对应附件列表失败，原因：{}！", msgId, attachResp.getMsg());
        }

        CommonResponse<UserVO> uservoResp = userApi.queryUserByUserCode(messageEntity.getCreateUserCode());
        if(!uservoResp.isSuccess()) {
            logger.error("查询消息【id-{}，userCode-{}】创建人失败，原因：{}！", msgId, messageEntity.getCreateUserCode(), attachResp.getMsg());
        } else {
            UserVO user = uservoResp.getData();
            if(null != user) {
                resp.put("sendUserName", user.getUserName());
                resp.put("sendUserMobile", user.getUserMobile());
                resp.put("sendUserEmail", user.getUserEmail());
                resp.put("sendUserAvator", user.getAvator());
            }
        }

        return CommonResponse.success(resp);
    }

    /**
     * 将全部未读消息设置成已读状态
     *
     * @return
     */
    @PostMapping("updateAllStatus")
    public CommonResponse<String> updateAllStatus() {
    	messageService.updateAllMessageStatus();
    	return CommonResponse.success("操作成功！");
    }
    
    /**
     * 查询我的消息总数
     * 
     * @return
     */
    @GetMapping("unread/count")
    public CommonResponse<Long> queryMyUnreadCount() {
    	Long userId = InvocationInfoProxy.getUserid();
    	PageParam pageParam = new PageParam();
    	pageParam.setStatus(0);
    	pageParam.setReceiver(userId);
        pageParam.setTenantId(InvocationInfoProxy.getTenantid());
    	
    	long count = messageService.queryCount(pageParam);
    	
    	return CommonResponse.success(count);
    }

    @GetMapping("sendMessage")
    public CommonResponse<String> sendMessage(@RequestParam String message) {
        if(StringUtils.isBlank(message)) {
            return CommonResponse.error("参数错误，消息发送失败！");
        }
        logger.info("收到消息-【{}】, 待初步验证参数发送消息队列！", message);

        try {
            PushMsgParameter pushMsgParameter = JSONObject.parseObject(message, PushMsgParameter.class);
            if(StringUtils.isBlank(pushMsgParameter.getMsgType())) {
                return CommonResponse.error("消息类型不能为空");
            }
            if(StringUtils.isBlank(pushMsgParameter.getSubject()) && StringUtils.isBlank(pushMsgParameter.getContent())) {
                return CommonResponse.error("消息标题和消息内容不能同时为空");
            }
            if(StringUtils.isBlank(pushMsgParameter.getTenantId())) {
                return CommonResponse.error("租户ID不能为空");
            }
            messageService.sendMessage(pushMsgParameter);
        } catch (Exception e) {
            logger.info("消息-【{}】发送消息队列异常", e);
            e.printStackTrace();
            return CommonResponse.success("消息发送失败，失败原因：{}", e.getMessage());
        }

        return CommonResponse.success("消息发送成功！");
    }

    @GetMapping("queryMsgType")
    public CommonResponse<List<MsgTypeEntity>> queryMsgType() {
    	QueryWrapper<MsgTypeEntity> queryWrapper = new QueryWrapper<>();
    	queryWrapper.eq("dr", 0);
		List<MsgTypeEntity> list = msgTypeService.list(queryWrapper);
        return CommonResponse.success(list);
    }

    /**
     * @Description 导出
     * @param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody PageParam pageParam, HttpServletResponse response) {
        List<MessageVO> voList = null;
        pageParam.setTenantId(InvocationInfoProxy.getTenantid());
        Long count = messageService.queryCount(pageParam);
        if(count > 0) {
            pageParam.setPageNum(0);
            pageParam.setPageSize(Math.toIntExact(count));
            voList = messageService.queryList(pageParam);
            for (MessageVO messageVO : voList) {
                String msgType = messageVO.getMsgType();
                switch (messageVO.getMsgType()){
                    case "task": msgType = "任务";break;
                    case "notice": msgType = "提醒";break;
                    case "earlywarning": msgType = "预警";break;
                    case "bpmtask": msgType = "审批待办";break;
                    case "bpmurge": msgType = "流程催办提醒";break;
                    case "bpmpass": msgType = "审批通过提醒";break;
                    case "bpmstop": msgType = "流程终止提醒";break;
                    case "bpmunapprove": msgType = "流程弃审提醒";break;
                    case "bpmreject": msgType = "流程驳回提醒";break;
                    case "bpmcc": msgType = "审批抄送提醒";break;
                    case "commentcc": msgType = "评论抄送提醒";break;
                    case "elecsignwithdrawal": msgType = "电子签章撤回提醒";break;
                    case "contractsign": msgType = "合同签署待办提醒";break;
                    case "ordershelf": msgType = "订单上架提醒";break;
                }
                messageVO.setMsgType(msgType);
            }
        }else{
            voList = new ArrayList<>();
        }
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", voList);
        ExcelExport.getInstance().export("message-export.xlsx", beans, response);
    }
}
