package com.ejianc.business.home.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.ejianc.business.finance.vo.ReceiveVO;
import com.ejianc.business.home.mapper.HomePortalMapper;
import com.ejianc.business.home.service.IHomePortalService;
import com.ejianc.business.home.util.EJCDateUtil;
import com.ejianc.business.home.util.JSONUtils;
import com.ejianc.business.home.util.ListCallable;
import com.ejianc.business.home.vo.MonthProductVO;
import com.ejianc.business.home.vo.ProjectTermNumVO;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.market.vo.StartWorkReportVO;
import com.ejianc.business.oa.bean.WeeklyReportEntity;
import com.ejianc.business.oa.service.IWeeklyReportService;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.Utils;
import com.ejianc.framework.skeleton.template.BaseEntity;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;


/**
 * <p>
 * 门户首页 服务实现类
 * </p>
 *
 * @author yqls
 * @since 2022-04-28
 */
@Service("ContractService")
public class HomePortalServiceImpl extends BaseServiceImpl<HomePortalMapper, BaseEntity> implements IHomePortalService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    @Autowired
    private IProjectApi projectApi;

    @Autowired
    private IWeeklyReportService weekService;

    @Override
    public Map<String, List<MonthProductVO>> monthProductMny(String status) {
        String startMonth = EJCDateUtil.getYear() + "-01-01";
        String endMonth = EJCDateUtil.getYear() + "-12-31";
        QueryParam param = new QueryParam();
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));// 生效

        ExecutorService threadPool = Executors.newFixedThreadPool(10);
        QueryParam param1 = Utils.deepCopy(param);// 深拷贝
        param1.getParams().put("reportMonth", new Parameter(QueryParam.BETWEEN, startMonth + "," + endMonth));
        QueryParam param2 = Utils.deepCopy(param);// 深拷贝
        param2.setPageIndex(0);
        param2.setPageSize(-1);
        param2.getParams().put("confirmTime", new Parameter(QueryParam.BETWEEN, startMonth + "," + endMonth));
        Future<JSONArray> future1 = ListCallable.excute(threadPool, param1, weekService);// 周报
        Future<JSONArray> future2 = ListCallable.excute(threadPool, BASE_HOST + "ejc-finance-web/receive/queryList", param2);// 收款登记

        List<WeeklyReportEntity> weekList = new ArrayList<>();// 周报
        List<ReceiveVO> receiveList = new ArrayList<>();// 收款登记
        try {
            weekList = JSONUtils.json2List(future1.get().toJSONString(), WeeklyReportEntity.class);
            receiveList = JSONUtils.json2List(future2.get().toJSONString(), ReceiveVO.class);
        } catch (Exception e) {
            logger.error("查询数据异常", e);
        } finally {
            threadPool.shutdown();
        }

        Set<Long> projectIds = weekList.stream().map(x->x.getProjectId()).collect(Collectors.toSet());
        projectIds.addAll(receiveList.stream().map(x->x.getProjectId()).collect(Collectors.toSet()));
        CommonResponse<List<ProjectRegisterVO>> resp = projectApi.queryProjectByIds(new ArrayList<>(projectIds));
        Map<Long, ProjectRegisterVO> projectMap = new HashMap<>();
        if(resp.isSuccess() && null != resp.getData()){
            List<ProjectRegisterVO> projectList = resp.getData();
            if(StringUtils.isNotEmpty(status)){
                projectList = projectList.stream().filter(x->status.equals(x.getStatus())).collect(Collectors.toList());
            }
            projectMap = projectList.stream().collect(Collectors.toMap(x->x.getId(), x->x));
        }

        List<MonthProductVO> list = new ArrayList<>();
        for(WeeklyReportEntity data : weekList){
            MonthProductVO vo = new MonthProductVO();
            vo.setMonth(EJCDateUtil.format(data.getReportMonth(), EJCDateUtil.MONTH));
            vo.setProductMny(data.getFinishOutputMny());
            vo.setUserNum(data.getManagePersonNum());
            if(!projectMap.containsKey(data.getProjectId())){
                continue;
            }
            ProjectRegisterVO project = projectMap.get(data.getProjectId());
            if(null == project.getProjectDirector()){
                continue;
            }
            vo.setProjectDirector(project.getProjectDirector());
            vo.setProjectDirectorName(project.getProjectDirectorName());
            vo.setSourceType("week");
            list.add(vo);
        }
        for(ReceiveVO data : receiveList){
            MonthProductVO vo = new MonthProductVO();
            vo.setMonth(EJCDateUtil.format(data.getConfirmTime(), EJCDateUtil.MONTH));
            vo.setReceiveMny(data.getReceiveMny());
            if(!projectMap.containsKey(data.getProjectId())){
                continue;
            }
            ProjectRegisterVO project = projectMap.get(data.getProjectId());
            if(null == project.getProjectDirector()){
                continue;
            }
            vo.setProjectDirector(project.getProjectDirector());
            vo.setProjectDirectorName(project.getProjectDirectorName());
            vo.setSourceType("receive");
            list.add(vo);
        }
        if(CollectionUtils.isEmpty(list)){
            return new HashMap<>();
        }

        Map<Long, Map<String, List<MonthProductVO>>> map = list.stream().collect(Collectors.groupingBy(x -> x.getProjectDirector(),
                        Collectors.groupingBy(x->x.getMonth())));

        List<String> monthData = EJCDateUtil.getMonthBetween(startMonth, endMonth);
        List<MonthProductVO> resultList = new LinkedList<>();
        for (String month : monthData) {
            for(Long userId : map.keySet()){
                MonthProductVO vo = new MonthProductVO();
                vo.setProjectDirector(userId);
                vo.setMonth(EJCDateUtil.monthToString(month));
                Map<String, List<MonthProductVO>> monthMap = map.get(userId);
                MonthProductVO data = monthMap.values().stream().flatMap(Collection::stream).findFirst().get();
                vo.setProjectDirectorName(data.getProjectDirectorName());
                BigDecimal productMny = BigDecimal.ZERO;
                BigDecimal receiveMny = BigDecimal.ZERO;
                BigDecimal userNum = BigDecimal.ZERO;
                if(monthMap.containsKey(month)){
                    List<MonthProductVO> voList = monthMap.get(month);
                    productMny = voList.stream().map(MonthProductVO::getProductMny).reduce(BigDecimal.ZERO, ComputeUtil::safeAdd);
                    receiveMny = voList.stream().map(MonthProductVO::getReceiveMny).reduce(BigDecimal.ZERO, ComputeUtil::safeAdd);
                    BigDecimal sumNum = voList.stream().map(MonthProductVO::getUserNum).reduce(BigDecimal.ZERO, ComputeUtil::safeAdd);
                    int weekNum = (int) voList.stream().filter(x->"week".equals(x.getSourceType())).count();
                    userNum = ComputeUtil.safeDiv(sumNum, ComputeUtil.toBigDecimal(weekNum));
                }
                vo.setProductMny(ComputeUtil.scale(productMny, 1));
                vo.setReceiveMny(ComputeUtil.scale(ComputeUtil.safeDiv(receiveMny, ComputeUtil.toBigDecimal("100")), 1));
                vo.setUserNum(ComputeUtil.scale(userNum, 2));
                BigDecimal userMny = ComputeUtil.safeDiv(vo.getProductMny(), vo.getUserNum());
                vo.setUserMny(ComputeUtil.scale(userMny, 1));
                resultList.add(vo);
            }
        }
        Map<String, List<MonthProductVO>> result = resultList.stream().collect(Collectors.groupingBy(x->x.getProjectDirectorName()));
        return result;
    }

    @Override
    public ProjectTermNumVO projectTerm() {
        String startMonth = EJCDateUtil.getYear() + "-01-01";
        String endMonth = EJCDateUtil.getYear() + "-12-31";
        QueryParam param = new QueryParam();
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));// 生效

        ExecutorService threadPool = Executors.newFixedThreadPool(10);
        QueryParam param1 = Utils.deepCopy(param);// 深拷贝
        param1.getParams().put("reportMonth", new Parameter(QueryParam.BETWEEN, startMonth + "," + endMonth));
        QueryParam param2 = Utils.deepCopy(param);// 深拷贝
        param2.setPageIndex(1);
        param2.setPageSize(-1);
        QueryParam param3 = Utils.deepCopy(param2);// 深拷贝
        Future<JSONArray> future1 = ListCallable.excute(threadPool, param1, weekService);// 周报
        Future<JSONArray> future2 = ListCallable.excute(threadPool, BASE_HOST + "ejc-income-web/contract/pageList", param2);// 施工合同
        Future<JSONArray> future3 = ListCallable.excute(threadPool, BASE_HOST + "ejc-market-web/startWorkReport/queryList", param3);// 开工报告

        List<WeeklyReportEntity> weekList = new ArrayList<>();// 周报
        List<ContractVo> incomeList = new ArrayList<>();// 施工合同
        List<StartWorkReportVO> startList = new ArrayList<>();// 开工报告
        try {
            weekList = JSONUtils.json2List(future1.get().toJSONString(), WeeklyReportEntity.class);
            incomeList = JSONUtils.json2List(future2.get().toJSONString(), ContractVo.class);
            startList = JSONUtils.json2List(future3.get().toJSONString(), StartWorkReportVO.class);
        } catch (Exception e) {
            logger.error("查询数据异常", e);
        } finally {
            threadPool.shutdown();
        }

        Map<Long, WeeklyReportEntity> weekMap = weekList.stream().collect(Collectors.toMap(x->x.getProjectId(), x->x,
                (v1, v2)->v1.getCreateTime().getTime() < v2.getCreateTime().getTime() ? v2 : v1));
        Map<Long, ContractVo> incomeMap = incomeList.stream().collect(Collectors.toMap(x->x.getProjectId(), x->x,
                (v1, v2)->v1.getCreateTime().getTime() < v2.getCreateTime().getTime() ? v2 : v1));
        Map<Long, StartWorkReportVO> startMap = startList.stream().collect(Collectors.toMap(x->x.getProjectId(), x->x,
                (v1, v2)->v1.getCreateTime().getTime() < v2.getCreateTime().getTime() ? v2 : v1));
        Set<Long> projectIds = new HashSet<>(weekMap.keySet());
        projectIds.addAll(new HashSet<>(incomeMap.keySet()));
        projectIds.addAll(new HashSet<>(startMap.keySet()));

        ProjectTermNumVO result = new ProjectTermNumVO();
        result.setFinishLessThree(0);
        result.setFinishLessSix(0);
        result.setFinishGreaterSix(0);
        result.setBuildingLessThree(0);
        result.setBuildingLessSix(0);
        result.setBuildingGreaterSix(0);
        for(Long projectId : projectIds){
            BigDecimal weeklyProgress = BigDecimal.ZERO;
            WeeklyReportEntity week = new WeeklyReportEntity();
            if(weekMap.containsKey(projectId)){
                week = weekMap.get(projectId);
                weeklyProgress = week.getWeeklyProgress();
            }
            Integer term = 0;
            if(incomeMap.containsKey(projectId)){
                term = incomeMap.get(projectId).getTerm();
                term = null != term ? term : 0;
            }
            Date estimateEndDate = null;
            if(startMap.containsKey(projectId)){
                estimateEndDate = startMap.get(projectId).getEstimateEndDate();
            }

            Integer overTerm = EJCDateUtil.daysBetween(week.getStartDate(), week.getPreFinishDate());
            if(null != estimateEndDate){
                overTerm = EJCDateUtil.daysBetween(week.getStartDate(), estimateEndDate) - term;
            }

            if(ComputeUtil.isLessThan(weeklyProgress, ComputeUtil.toBigDecimal("80"))){
                overTerm = overTerm - term;
                if(overTerm <= 30){
                    result.setBuildingLessThree(result.getBuildingLessThree() + 1);
                } else if(overTerm <= 60){
                    result.setBuildingLessSix(result.getBuildingLessSix() + 1);
                } else {
                    result.setBuildingGreaterSix(result.getBuildingGreaterSix() + 1);
                }
            } else {
                if(overTerm <= 30){
                    result.setFinishLessThree(result.getFinishLessThree() + 1);
                } else if(overTerm <= 60){
                    result.setFinishLessSix(result.getFinishLessSix() + 1);
                } else {
                    result.setFinishGreaterSix(result.getFinishGreaterSix() + 1);
                }
            }
        }
        return result;
    }

    public static void main(String[] args) {
        String startMonth = EJCDateUtil.getYear() + "-01-01";
        String endMonth = EJCDateUtil.getYear() + "-12-31";
        System.out.println(startMonth + " " + endMonth);
        System.out.println(EJCDateUtil.getMonthBetween(startMonth, endMonth));
        List<MonthProductVO> voList = new ArrayList<>();
        BigDecimal productMny = voList.stream().map(MonthProductVO::getProductMny).reduce(BigDecimal.ZERO, ComputeUtil::safeAdd);
        System.out.println(productMny.toPlainString());
    }

}
