package com.ejianc.business.oa.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.budget.api.IBudgetProjectProApi;
import com.ejianc.business.budget.vo.BudgetProjectProParamControlVO;
import com.ejianc.business.budget.vo.BudgetProjectProQuantityAndMnyVO;
import com.ejianc.business.income.api.IIncomeContractApi;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.labor.api.IAttendanceApi;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.oa.bean.WeeklyReportEntity;
import com.ejianc.business.oa.mapper.WeeklyReportMapper;
import com.ejianc.business.oa.service.IWeeklyReportService;
import com.ejianc.business.oa.vo.WeeklyReportVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.WeekFields;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 周报
 *
 * @author generator
 */
@Service("weeklyReportService")
public class WeeklyReportServiceImpl extends BaseServiceImpl<WeeklyReportMapper, WeeklyReportEntity> implements IWeeklyReportService {

    private static final String BILL_CODE = "WEEKLY_REPORT";

    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IProjectApi projectApi;
    @Autowired
    private IUserApi userApi;
    @Autowired
    private IAttendanceApi attendanceApi;
    @Autowired
    private IBudgetProjectProApi budgetProjectProApi;
    @Autowired
    private IOrgApi iOrgApi;
    @Autowired
    private SessionManager sessionManager;
    @Autowired
    private IIncomeContractApi incomeContractApi;

    @Override
    public WeeklyReportVO saveOrUpdate(WeeklyReportVO weeklyReportVO) {
        LambdaQueryWrapper<WeeklyReportEntity> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(WeeklyReportEntity::getProjectId, weeklyReportVO.getProjectId());
        queryWrapper.eq(WeeklyReportEntity::getWeeklyNum, weeklyReportVO.getWeeklyNum());
        queryWrapper.ne(null != weeklyReportVO.getId(), WeeklyReportEntity::getId, weeklyReportVO.getId());
        if (count(queryWrapper) > 0) {
            throw new BusinessException("已经存在本周周报，请调整!");
        }
        WeeklyReportEntity entity = BeanMapper.map(weeklyReportVO, WeeklyReportEntity.class);
        if (entity.getId() == null || entity.getId() == 0) {
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), weeklyReportVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        saveOrUpdate(entity, false);
        return BeanMapper.map(entity, WeeklyReportVO.class);
    }

    @Override
    public WeeklyReportVO afterProject(Long id, Long projectId, Integer weeklyNum) {
        WeeklyReportVO reportVO = new WeeklyReportVO();
        LambdaQueryWrapper<WeeklyReportEntity> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(WeeklyReportEntity::getProjectId, projectId);
        queryWrapper.lt(null != weeklyNum, WeeklyReportEntity::getWeeklyNum, weeklyNum);
        queryWrapper.ne(null != id, WeeklyReportEntity::getId, id);
        queryWrapper.orderByDesc(WeeklyReportEntity::getWeeklyNum);
        queryWrapper.last("limit 1");
        WeeklyReportEntity entity = getOne(queryWrapper);
        if (null != entity) {
            reportVO.setLastWeeklyProgress(entity.getWeeklyProgress());
            reportVO.setPreFinishDate(entity.getPreFinishDate());
            reportVO.setStartDate(entity.getStartDate());
        }else {
            reportVO.setLastWeeklyProgress(BigDecimal.ZERO);
        }
        if (null != weeklyNum) {
            LocalDate today = LocalDate.now();
            int year = today.getYear();

            LocalDate startOfWeek = getStartOfWeek(year, weeklyNum);
            LocalDate endOfWeek = getEndOfWeek(startOfWeek);

            CommonResponse<Integer> response = attendanceApi.queryNumByProjAndQj(startOfWeek.toString(), endOfWeek.toString(), projectId);
            if (response.isSuccess() && null != response.getData() && !Integer.valueOf(0).equals(response.getData())) {
                int total =  response.getData();
                int dayOfWeekNumber = today.getDayOfWeek().getValue();
                reportVO.setConstructionPersonNum(BigDecimal.valueOf((total + dayOfWeekNumber - 1) / dayOfWeekNumber));
            }else {
                reportVO.setConstructionPersonNum(BigDecimal.ZERO);
            }
        }
        CommonResponse<ProjectRegisterVO> response = projectApi.queryProjectDetail(projectId);
        if (response.isSuccess() && null != response.getData()) {
            ProjectRegisterVO registerVO = response.getData();
            //查询项目成员
            CommonResponse<List<Map<String, Object>>> membersResp = userApi.getEmployeeList(registerVO.getProjectDepartmentId());
            reportVO.setManagePersonNum(membersResp.isSuccess() && CollectionUtils.isNotEmpty(membersResp.getData()) ? BigDecimal.valueOf(membersResp.getData().size()) : BigDecimal.ZERO);
        }

        CommonResponse<ContractVo> contResponse = incomeContractApi.searchContract(projectId);
        if (contResponse.isSuccess() && null != contResponse.getData()) {
            reportVO.setFinishDate(contResponse.getData().getEndDate());
        }

        BudgetProjectProParamControlVO controlVO = new BudgetProjectProParamControlVO();
        controlVO.setProjectId(projectId);
        controlVO.setCostType(-1);
        CommonResponse<BudgetProjectProQuantityAndMnyVO> commonResponse = budgetProjectProApi.fetchQuantityAndMny(controlVO);
        if (commonResponse.isSuccess() && null != commonResponse.getData()) {
            reportVO.setPreCostMny(ComputeUtil.safeDiv(commonResponse.getData().getBudgetTaxMny(), BigDecimal.valueOf(10000)));
        }else {
            reportVO.setPreCostMny(BigDecimal.ZERO);
        }
        return reportVO;
    }

    @Override
    public IPage<WeeklyReportVO> queryReportList(QueryParam param) {
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getOrderMap().put("projectDirector", "desc");
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter("in", authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter("in", iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        IPage<WeeklyReportEntity> page = queryPage(param, false);
        IPage<WeeklyReportVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<WeeklyReportVO> reportVOS = BeanMapper.mapList(page.getRecords(), WeeklyReportVO.class);

        if (CollectionUtils.isNotEmpty(reportVOS)) {
            Map<Long, List<WeeklyReportVO>> map = reportVOS.stream().collect(Collectors.groupingBy(WeeklyReportVO::getProjectDirector));
            Set<Long> matchSet = new HashSet<>();
            for (WeeklyReportVO reportVO : reportVOS) {
                if(!matchSet.contains(reportVO.getProjectDirector())){
                    List<WeeklyReportVO> list = map.get(reportVO.getProjectDirector());
                    reportVO.setRowSpan(list.size());
                    BigDecimal areaOutputMny = BigDecimal.ZERO;
                    BigDecimal areaPersonEfficiency = BigDecimal.ZERO;
                    for (WeeklyReportVO weeklyVO : list) {
                        areaOutputMny = ComputeUtil.safeAdd(areaOutputMny, weeklyVO.getFinishOutputMny());
                        areaPersonEfficiency = ComputeUtil.safeAdd(areaPersonEfficiency, weeklyVO.getManagePersonNum());
                    }
                    reportVO.setAreaOutputMny(areaOutputMny);
                    reportVO.setAreaPersonEfficiency(ComputeUtil.safeDiv(areaOutputMny, areaPersonEfficiency));
                    matchSet.add(reportVO.getProjectDirector());
                }else {
                    reportVO.setRowSpan(0);
                }
                reportVO.setNormalRowSpan(1);
            }
        }
        pageData.setRecords(reportVOS);
        return pageData;
    }

    @Override
    public IPage<WeeklyReportVO> queryProblemList(QueryParam param) {
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getOrderMap().put("projectDirector", "desc");
        param.getOrderMap().put("weeklyNum", "asc");
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter("in", authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter("in", iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        IPage<WeeklyReportEntity> page = queryPage(param, false);
        IPage<WeeklyReportVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<WeeklyReportVO> reportVOS = BeanMapper.mapList(page.getRecords(), WeeklyReportVO.class);

        if (CollectionUtils.isNotEmpty(reportVOS)) {
            Map<Long, List<WeeklyReportVO>> map = reportVOS.stream().collect(Collectors.groupingBy(WeeklyReportVO::getProjectDirector));
            Set<Long> matchSet = new HashSet<>();
            for (WeeklyReportVO reportVO : reportVOS) {
                if(!matchSet.contains(reportVO.getProjectDirector())){
                    List<WeeklyReportVO> list = map.get(reportVO.getProjectDirector());
                    reportVO.setRowSpan(list.size());
                    matchSet.add(reportVO.getProjectDirector());
                }else {
                    reportVO.setRowSpan(0);
                }
                reportVO.setNormalRowSpan(1);
            }
        }
        pageData.setRecords(reportVOS);
        return pageData;
    }

    public static LocalDate getStartOfWeek(int year, int weekNumber) {
        LocalDate firstDayOfYear = LocalDate.of(year, 1, 1);
        WeekFields weekFields = WeekFields.of(Locale.getDefault());
        return firstDayOfYear.with(weekFields.weekOfYear(), weekNumber)
                .with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY));
    }

    public static LocalDate getEndOfWeek(LocalDate startOfWeek) {
        return startOfWeek.with(TemporalAdjusters.nextOrSame(DayOfWeek.SUNDAY));
    }
}
