package com.ejianc.business.oa.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.budget.api.IBudgetProjectProApi;
import com.ejianc.business.budget.vo.BudgetProjectProParamControlVO;
import com.ejianc.business.budget.vo.BudgetProjectProQuantityAndMnyVO;
import com.ejianc.business.income.api.IIncomeContractApi;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.labor.api.IAttendanceApi;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.oa.bean.WeeklyReportEntity;
import com.ejianc.business.oa.bean.WeeklyReportManagerEntity;
import com.ejianc.business.oa.controller.query.WeeklyReportQuery;
import com.ejianc.business.oa.mapper.WeeklyReportMapper;
import com.ejianc.business.oa.service.IWeeklyReportManagerService;
import com.ejianc.business.oa.service.IWeeklyReportService;
import com.ejianc.business.oa.vo.ProjectDurationVO;
import com.ejianc.business.oa.vo.WeeklyReportManagerVO;
import com.ejianc.business.oa.vo.WeeklyReportVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.WeekFields;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 周报
 *
 * @author generator
 */
@Service("weeklyReportService")
public class WeeklyReportServiceImpl extends BaseServiceImpl<WeeklyReportMapper, WeeklyReportEntity> implements IWeeklyReportService {

    private static final String BILL_CODE = "WEEKLY_REPORT";

    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IProjectApi projectApi;
    @Autowired
    private IUserApi userApi;
    @Autowired
    private IAttendanceApi attendanceApi;
    @Autowired
    private IBudgetProjectProApi budgetProjectProApi;
    @Autowired
    private IOrgApi iOrgApi;
    @Autowired
    private SessionManager sessionManager;
    @Autowired
    private IIncomeContractApi incomeContractApi;
    @Autowired
    private WeeklyReportMapper weeklyReportMapper;

    @Autowired
    private IWeeklyReportManagerService weeklyReportManagerService;




    @Override
    public WeeklyReportVO saveOrUpdate(WeeklyReportVO weeklyReportVO) {
        LambdaQueryWrapper<WeeklyReportEntity> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(WeeklyReportEntity::getProjectId, weeklyReportVO.getProjectId());
        queryWrapper.eq(WeeklyReportEntity::getWeeklyNum, weeklyReportVO.getWeeklyNum());
        queryWrapper.ne(null != weeklyReportVO.getId(), WeeklyReportEntity::getId, weeklyReportVO.getId());
        if (count(queryWrapper) > 0) {
            throw new BusinessException("已经存在本周周报，请调整!");
        }
        WeeklyReportEntity entity = BeanMapper.map(weeklyReportVO, WeeklyReportEntity.class);
        if (entity.getId() == null || entity.getId() == 0) {
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), weeklyReportVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        saveOrUpdate(entity, false);
        return BeanMapper.map(entity, WeeklyReportVO.class);
    }

    @Override
    public WeeklyReportVO afterProject(Long id, Long projectId, Integer weeklyNum) {
        WeeklyReportVO reportVO = new WeeklyReportVO();
        LambdaQueryWrapper<WeeklyReportEntity> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(WeeklyReportEntity::getProjectId, projectId);
        // 第一周存在跨年获取不到上周进度问题修复
        queryWrapper.lt(null != weeklyNum && weeklyNum.intValue() != 1, WeeklyReportEntity::getWeeklyNum, weeklyNum);
        queryWrapper.ne(null != id, WeeklyReportEntity::getId, id);
        queryWrapper.orderByDesc(WeeklyReportEntity::getWeeklyNum);
        queryWrapper.last("limit 1");
        WeeklyReportEntity entity = getOne(queryWrapper);
        if (null != entity) {
            reportVO.setLastWeeklyProgress(entity.getWeeklyProgress());
            reportVO.setPreFinishDate(entity.getPreFinishDate());
            reportVO.setStartDate(entity.getStartDate());
            reportVO.setYearPlanMny(entity.getYearPlanMny());
        }else {
            reportVO.setLastWeeklyProgress(BigDecimal.ZERO);
        }
        if (null != weeklyNum) {
            LocalDate today = LocalDate.now();
            int year = today.getYear();

            LocalDate startOfWeek = getStartOfWeek(year, weeklyNum);
            LocalDate endOfWeek = getEndOfWeek(startOfWeek);

            CommonResponse<Integer> response = attendanceApi.queryNumByProjAndQj(startOfWeek.toString(), endOfWeek.toString(), projectId);
            if (response.isSuccess() && null != response.getData() && !Integer.valueOf(0).equals(response.getData())) {
                int total =  response.getData();
                int dayOfWeekNumber = today.getDayOfWeek().getValue();
                reportVO.setConstructionPersonNum(BigDecimal.valueOf((total + dayOfWeekNumber - 1) / dayOfWeekNumber));
            }else {
                reportVO.setConstructionPersonNum(BigDecimal.ZERO);
            }
        }
        CommonResponse<ProjectRegisterVO> response = projectApi.queryProjectDetail(projectId);
        if (response.isSuccess() && null != response.getData()) {
            ProjectRegisterVO registerVO = response.getData();
            //查询项目成员
            CommonResponse<List<Map<String, Object>>> membersResp = userApi.getEmployeeList(registerVO.getProjectDepartmentId());
            reportVO.setManagePersonNum(membersResp.isSuccess() && CollectionUtils.isNotEmpty(membersResp.getData()) ? BigDecimal.valueOf(membersResp.getData().size()) : BigDecimal.ZERO);
        }

        CommonResponse<ContractVo> contResponse = incomeContractApi.searchContract(projectId);
        if (contResponse.isSuccess() && null != contResponse.getData()) {
            reportVO.setFinishDate(contResponse.getData().getEndDate());
            reportVO.setCostTotalMny(contResponse.getData().getContractTaxMny());
            reportVO.setContractTempMny(contResponse.getData().getContractTempMny());
        }

        BudgetProjectProParamControlVO controlVO = new BudgetProjectProParamControlVO();
        controlVO.setProjectId(projectId);
        controlVO.setCostType(-1);
        CommonResponse<BudgetProjectProQuantityAndMnyVO> commonResponse = budgetProjectProApi.fetchQuantityAndMny(controlVO);
        if (commonResponse.isSuccess() && null != commonResponse.getData()) {
            reportVO.setPreCostMny(ComputeUtil.safeDiv(commonResponse.getData().getBudgetTaxMny(), BigDecimal.valueOf(10000)));
        }else {
            reportVO.setPreCostMny(BigDecimal.ZERO);
        }
        return reportVO;
    }

    @Override
    public IPage<WeeklyReportVO> queryReportList( WeeklyReportQuery query) {
        IPage<WeeklyReportEntity> page = new Page<>(query.getPageNum(), query.getPageSize());
        List<WeeklyReportEntity> records = new ArrayList<>();
        query.setTenantId(InvocationInfoProxy.getTenantid());
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
            query.setOrgIdList(authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        } else {
            query.setOrgIdList(iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        }
        //是否查询全部
        if (query.getDisAblePage()){
            records = baseMapper.queryReportList(null, query);
        }else {
            records = baseMapper.queryReportList(page, query);
        }
        page.setRecords(records);

        IPage<WeeklyReportVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<WeeklyReportVO> reportVOS = BeanMapper.mapList(page.getRecords(), WeeklyReportVO.class);

        if (CollectionUtils.isNotEmpty(reportVOS)) {
            Map<Long, List<WeeklyReportVO>> map = reportVOS.stream().collect(Collectors.groupingBy(WeeklyReportVO::getProjectDirector));
            Set<Long> matchSet = new HashSet<>();
            for (WeeklyReportVO reportVO : reportVOS) {
                if(!matchSet.contains(reportVO.getProjectDirector())){
                    List<WeeklyReportVO> list = map.get(reportVO.getProjectDirector());
                    reportVO.setRowSpan(list.size());
                    BigDecimal areaOutputMny = BigDecimal.ZERO;
                    BigDecimal areaPersonEfficiency = BigDecimal.ZERO;
                    for (WeeklyReportVO weeklyVO : list) {
                        areaOutputMny = ComputeUtil.safeAdd(areaOutputMny, weeklyVO.getFinishOutputMny());
                        areaPersonEfficiency = ComputeUtil.safeAdd(areaPersonEfficiency, weeklyVO.getManagePersonNum());
                    }
                    reportVO.setAreaOutputMny(areaOutputMny);
                    reportVO.setAreaPersonEfficiency(ComputeUtil.safeDiv(areaOutputMny, areaPersonEfficiency));
                    matchSet.add(reportVO.getProjectDirector());
                }else {
                    reportVO.setRowSpan(0);
                }
                reportVO.setNormalRowSpan(1);
            }
        }

        //判断当前查询的结果是否为空
        if (!ObjectUtils.isEmpty(reportVOS)){

            //当前结果不为空、查询周报报表对应的管理人员明细
            //获取当前的周报报表id
            List<Long> idList = reportVOS.stream().map(WeeklyReportVO::getId).collect(Collectors.toList());

            //查询周报报表对应的管理人员明细
            List<WeeklyReportManagerVO> managerList = weeklyReportManagerService.lambdaQuery() //调用MP的链式查询 lambda式
                    .in(WeeklyReportManagerEntity::getPid, idList) //条件为 周报人员明细表的pid in 周报信息表id集合
                    .list() // 获取集合
                    .stream()
                    .map(entity -> BeanMapper.map(entity, WeeklyReportManagerVO.class)) //将管理人员实体转换成管理人员VO 以便后续的业务处理
                    .collect(Collectors.toList());

            //将查询出的管理人员设置到 每个周报对象的人员明细列表中
            //遍历周报报表列表
            reportVOS.forEach(weeklyReportVO -> {

                //遍历管理人员明细列表
                managerList.forEach(managerEntity -> {

                    //比较管理人员明细对象的pid 是否与 周报报表对象的id相等,若相等则将管理人员明细对象设置到 周报报表对象的管理人员明细列表中
                    if (Objects.equals(managerEntity.getPid(),weeklyReportVO.getId())){

                        //相等的情况
                        weeklyReportVO.getManagerList().add(managerEntity);
                    }
                });
            });


            //获取项目id集合
            List<Long> projectIdList = reportVOS.stream().map(WeeklyReportVO::getProjectId).collect(Collectors.toList());

            //查询项目工期
            Map<Long,ProjectDurationVO> projectDurationMap = baseMapper.queryProjectDurationMap(projectIdList);
            //判断项目工期是否为空
            if (!ObjectUtils.isEmpty(projectDurationMap)){

                //项目工期不为空、设置每个周报报表的项目工期信息
                reportVOS.forEach(report ->{

                    //获取每个周报对应的项目开工时间、竣工时间
                    //通过项目id取出该项目的项目工期对象
                    ProjectDurationVO projectDurationVO = projectDurationMap.get(report.getProjectId());

                    //判断项目工期对象是否为空
                    if (!ObjectUtils.isEmpty(projectDurationVO)){

                        //不为空、取出项目开工时间、竣工时间
                        String startWorkDate = projectDurationVO.getStartWorkDate();
                        String endWorkDate = projectDurationVO.getEndWorkDate();

                        //设置项目的开工时间、竣工时间
                        report.setStartWorkDate(startWorkDate);
                        report.setEndWorkDate(endWorkDate);
                    }
                });
            }
        }

        pageData.setRecords(reportVOS);
        return pageData;
    }

    @Override
    public IPage<WeeklyReportVO> queryProblemList(QueryParam param) {
        Map<String, Parameter> params = param.getParams();
        params.put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        params.put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getOrderMap().put("projectDirector", "desc");
        param.getOrderMap().put("weeklyNum", "asc");
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
            params.put("orgId", new Parameter("in", authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            params.put("orgId", new Parameter("in", iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        if (params.containsKey("yyear") && params.get("yyear").getValue() != null) {
            param.getParams().put(QueryParam.SQL, new Parameter(QueryParam.SQL, " YEAR(report_month) = " + params.get("yyear").getValue().toString()));
            params.remove("yyear");
        }
        QueryWrapper<Object> wrapper = changeToQueryWrapper(param);
        Page<WeeklyReportVO> page = new Page<>(param.getPageIndex(), param.getPageSize());
        List<WeeklyReportVO> reportVOS = baseMapper.queryPageJson(page, wrapper);
        IPage<WeeklyReportVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        if (CollectionUtils.isNotEmpty(reportVOS)) {
            Map<Long, List<WeeklyReportVO>> map = reportVOS.stream().collect(Collectors.groupingBy(WeeklyReportVO::getProjectDirector));
            Set<Long> matchSet = new HashSet<>();
            for (WeeklyReportVO reportVO : reportVOS) {
                if(!matchSet.contains(reportVO.getProjectDirector())){
                    List<WeeklyReportVO> list = map.get(reportVO.getProjectDirector());
                    reportVO.setRowSpan(list.size());
                    matchSet.add(reportVO.getProjectDirector());
                }else {
                    reportVO.setRowSpan(0);
                }
                reportVO.setNormalRowSpan(1);
            }
        }
        pageData.setRecords(reportVOS);
        return pageData;
    }




    public static LocalDate getStartOfWeek(int year, int weekNumber) {
        LocalDate firstDayOfYear = LocalDate.of(year, 1, 1);
        WeekFields weekFields = WeekFields.of(Locale.getDefault());
        return firstDayOfYear.with(weekFields.weekOfYear(), weekNumber)
                .with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY));
    }

    public static LocalDate getEndOfWeek(LocalDate startOfWeek) {
        return startOfWeek.with(TemporalAdjusters.nextOrSame(DayOfWeek.SUNDAY));
    }


    @Override
    public Map<Long,ProjectDurationVO> selectStartWorkReportTimeAndTermByProjectId(List<Long>  projectId) {
        Map<Long,ProjectDurationVO> map=baseMapper.queryProjectDurationMap(projectId);
        return map;
    }

    @Override
    public Integer queryWeeklyNum(WeeklyReportQuery query) {
        query.setTenantId(InvocationInfoProxy.getTenantid());
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
            query.setOrgIdList(authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        } else {
            query.setOrgIdList(iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        }
        return baseMapper.queryWeeklyNum(query);
    }

}
