package com.ejianc.business.procBid.utils;

import org.apache.poi.ss.usermodel.*;

import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.RegionUtil;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.core.io.ClassPathResource;

import javax.servlet.http.HttpServletResponse;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.*;

import static org.apache.poi.ss.usermodel.PatternFormatting.SOLID_FOREGROUND;

public class ExportExcelUtil {

    /**
     * 导出表格
     * @param fileName 文件名称
     * @param data 数据
     * @param headerMap 表头别名
     * @param response 响应对象
     */
    public static void export(String fileName,List<List<Object>> data,Map<String, List<CellModel>> headerMap,HttpServletResponse response){
        export(null,0,fileName,data,headerMap,response);
    }

    /**
     * 导出Excel文件的方法
     * 参考网址1： https://blog.csdn.net/q1424966670/article/details/135644180 、
     *
     * @param templatePath 模板文件路径
     * @param skipRows     跳过的行数
     * @param fileName     导出文件名
     * @param data         数据列表
     * @param cellTitleMap 表头信息映射
     * @param response     HTTP响应对象
     */
    public static void export(String templatePath, Integer skipRows, String fileName, List<List<Object>> data, Map<String, List<CellModel>> cellTitleMap, HttpServletResponse response) {
        Workbook workbook = null;
        OutputStream out = null;
        Sheet sheet = null;
        try {
            // 根据模板路径判断是否使用模板
            if (templatePath != null && !templatePath.isEmpty()) {
                ClassPathResource resource = new ClassPathResource("excel/" + templatePath);
                if (!resource.exists()) {
                    // 如果文件不存在，抛出异常并提供详细信息
                    throw new RuntimeException("模板文件不存在: " + resource.getPath());
                }

                try (InputStream inputStream = resource.getInputStream()) {
                    // 使用模板文件的输入流创建Workbook
                    workbook = WorkbookFactory.create(inputStream);
                } catch (Exception e) {
                    // 异常处理时，提供更多的上下文信息
                    throw new RuntimeException("加载模板文件时出错: " + resource.getPath(), e);
                }

                // 获取模板文件的第一个工作表
                sheet = workbook.getSheetAt(0);
            } else {
                // 如果没有模板，创建一个新的工作簿和工作表
                workbook = new XSSFWorkbook();
                sheet = workbook.createSheet("sheet1");

                // 定义表头单元格样式
                XSSFCellStyle cellStyle = (XSSFCellStyle)workbook.createCellStyle();
                // 创建 RGB 颜色 (153, 204, 255) 浅蓝色
                XSSFColor lightBlueColor = new XSSFColor(new java.awt.Color(153, 204, 255));
                cellStyle.setFillForegroundColor(lightBlueColor);// 设置单元格的背景色为指定的 RGB 颜色
                cellStyle.setFillPattern(SOLID_FOREGROUND);// 背景色填充样式
                cellStyle.setAlignment(CellStyle.ALIGN_CENTER);
                cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
                cellStyle.setWrapText(true);
                cellStyle.setBorderBottom(CellStyle.BORDER_THIN);
                cellStyle.setBorderLeft(CellStyle.BORDER_THIN);
                cellStyle.setBorderRight(CellStyle.BORDER_THIN);
                cellStyle.setBorderTop(CellStyle.BORDER_THIN);

                // 定义表头单元格字体
                Font font = workbook.createFont();
                font.setFontHeightInPoints((short) 14);
                font.setFontName("宋体");
                font.setBoldweight(Font.BOLDWEIGHT_BOLD);
                cellStyle.setFont(font);

                // 创建表头
                for (int t = 0; t <= skipRows; t++) {
                    Row row = sheet.createRow(t);
                    List<CellModel> cellNameList = cellTitleMap.get(String.valueOf(t));

                    for (CellModel cellModel : cellNameList) {
                        if (cellModel.getStartColumn() != null) {
                            Cell cell = row.createCell(cellModel.getStartColumn());
                            cell.setCellValue(cellModel.getCellName());
                            cell.setCellStyle(cellStyle);
                        }
                    }

                    // 设置除第一行单元格样式（行合并单元格样式设置）
                    if (t != 0) {
                        int lastCellNum = sheet.getRow(0).getLastCellNum();
                        for (int cellIndex = 0; cellIndex < lastCellNum; cellIndex++) {
                            Cell cell = sheet.getRow(t).getCell(cellIndex);
                            if (cell == null) {
                                cell = sheet.getRow(t).createCell(cellIndex);
                                cell.setCellValue("");
                            }
                            cell.setCellStyle(cellStyle);
                        }
                    }

                    // 合并单元格
                    for (CellModel cellModel : cellNameList) {
                        if (cellModel.getStartRow() != null) {
                            CellRangeAddress region = new CellRangeAddress(cellModel.getStartRow(),
                                    cellModel.getEndRow(), cellModel.getStartColumn(), cellModel.getEndColumn());
                            sheet.addMergedRegion(region);

                            // 设置合并后边框
                            RegionUtil.setBorderBottom(1, region, sheet, workbook);
                            RegionUtil.setBorderLeft(1, region, sheet, workbook);
                            RegionUtil.setBorderRight(1, region, sheet, workbook);
                            RegionUtil.setBorderTop(1, region, sheet, workbook);
                        }

                        if (cellModel.getWidth() != null) {
                            sheet.setColumnWidth(cellModel.getStartColumn(), cellModel.getWidth() * 256);
                        } else {
                            sheet.setColumnWidth(cellModel.getStartColumn(), cellModel.getCellName().getBytes().length * 256);
                        }
                    }
                }
            }

            // 定义数据内容单元格样式
            CellStyle cellDataStyle = workbook.createCellStyle();
            cellDataStyle.setAlignment(CellStyle.ALIGN_CENTER);
            cellDataStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
            cellDataStyle.setWrapText(true);
            cellDataStyle.setBorderBottom(CellStyle.BORDER_THIN);
            cellDataStyle.setBorderLeft(CellStyle.BORDER_THIN);
            cellDataStyle.setBorderRight(CellStyle.BORDER_THIN);
            cellDataStyle.setBorderTop(CellStyle.BORDER_THIN);

            // 计算数据开始的行号
            int startRow = skipRows + 1;
            // 导出具体的数据 使用Object进行类型装换
            for (List<Object> rowData : data) {
                Row rowValue = sheet.createRow(startRow);
                for (int i = 0; i < rowData.size(); i++) {
                    Object value = rowData.get(i);
                    Cell cell = rowValue.createCell(i);
                    if (value != null) {
                        if (value instanceof String) {
                            cell.setCellValue((String) value);
                        } else if (value instanceof Double) {
                            cell.setCellValue((Double) value);
                        } else if (value instanceof Long) {
                            cell.setCellValue((Long) value);
                        } else if (value instanceof Integer) {
                            cell.setCellValue((Integer) value);
                        } else if (value instanceof Boolean) {
                            cell.setCellValue((Boolean) value);
                        } else if (value instanceof Date) {
                            cell.setCellValue((Date) value);
                            cell.setCellStyle(cellDataStyle);
                        } else if (value instanceof Float) {
                            cell.setCellValue((Float) value);
                        } else if (value instanceof BigDecimal) {
                            cell.setCellValue(((BigDecimal) value).doubleValue());
                        } else {
                            cell.setCellValue(value.toString());
                        }
                    } else {
                        cell.setCellValue("");
                    }
                    cell.setCellStyle(cellDataStyle);
                }
                startRow++;
            }
            // 设置响应头以下载文件
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(fileName, "UTF-8"));
            response.setCharacterEncoding("UTF-8");

            // 将工作簿写入输出流
            out = response.getOutputStream();
            workbook.write(out);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            // 关闭资源
            if (workbook != null) {
                try {
                    workbook.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * 导出Excel文件的方法
     *
     * @param templatePath 模板文件路径
     * @param skipRows     跳过的行数
     * @param fileName     导出文件名
     * @param data         数据列表
     * @param cellTitleMap 表头信息映射
     * @param response     HTTP响应对象
     */
    public static void export(String templatePath, Integer skipRows, String fileName, Iterable<?> data, Map<String, List<CellModel>> cellTitleMap, HttpServletResponse response) {
        Workbook workbook = null;
        OutputStream out = null;
        Sheet sheet = null;
        try {
            // 根据模板路径判断是否使用模板
            if (templatePath != null && !templatePath.isEmpty()) {
                ClassPathResource resource = new ClassPathResource("excel/" + templatePath);
                if (!resource.exists()) {
                    // 如果文件不存在，抛出异常并提供详细信息
                    throw new RuntimeException("模板文件不存在: " + resource.getPath());
                }

                try (InputStream inputStream = resource.getInputStream()) {
                    // 使用模板文件的输入流创建Workbook
                    workbook = WorkbookFactory.create(inputStream);
                } catch (Exception e) {
                    // 异常处理时，提供更多的上下文信息
                    throw new RuntimeException("加载模板文件时出错: " + resource.getPath(), e);
                }

                // 获取模板文件的第一个工作表
                sheet = workbook.getSheetAt(0);
            } else {
                // 如果没有模板，创建一个新的工作簿和工作表
                workbook = new XSSFWorkbook();
                sheet = workbook.createSheet("sheet1");

                // 定义表头单元格样式 工作簿格式不同会导致设置背景色变为全黑  https://blog.csdn.net/weixin_40251892/article/details/113997173
                XSSFCellStyle cellStyle = (XSSFCellStyle)workbook.createCellStyle();
                // 创建 RGB 颜色 (153, 204, 255) 浅蓝色
                XSSFColor lightBlueColor = new XSSFColor(new java.awt.Color(153, 204, 255));
                cellStyle.setFillForegroundColor(lightBlueColor);// 设置单元格的背景色为指定的 RGB 颜色
                cellStyle.setFillPattern(SOLID_FOREGROUND);// 背景色填充样式
                cellStyle.setAlignment(CellStyle.ALIGN_CENTER);
                cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
                cellStyle.setWrapText(true);
                cellStyle.setBorderBottom(CellStyle.BORDER_THIN);
                cellStyle.setBorderLeft(CellStyle.BORDER_THIN);
                cellStyle.setBorderRight(CellStyle.BORDER_THIN);
                cellStyle.setBorderTop(CellStyle.BORDER_THIN);

                // 定义表头单元格字体
                Font font = workbook.createFont();
                font.setFontHeightInPoints((short) 14);
                font.setFontName("宋体");
                font.setBoldweight(Font.BOLDWEIGHT_BOLD);
                cellStyle.setFont(font);

                // 创建表头
                for (int t = 0; t <= skipRows; t++) {
                    Row row = sheet.createRow(t);
                    List<CellModel> cellNameList = cellTitleMap.get(String.valueOf(t));

                    for (CellModel cellModel : cellNameList) {
                        if (cellModel.getStartColumn() != null) {
                            Cell cell = row.createCell(cellModel.getStartColumn());
                            cell.setCellValue(cellModel.getCellName());
                            cell.setCellStyle(cellStyle);
                        }
                    }

                    if (t != 0) {
                        int lastCellNum = sheet.getRow(0).getLastCellNum();
                        for (int cellIndex = 0; cellIndex < lastCellNum; cellIndex++) {
                            Cell cell = sheet.getRow(t).getCell(cellIndex);
                            if (cell == null) {
                                cell = sheet.getRow(t).createCell(cellIndex);
                                cell.setCellValue("");
                            }
                            cell.setCellStyle(cellStyle);
                        }
                    }

                    for (CellModel cellModel : cellNameList) {
                        if (cellModel.getStartRow() != null) {
                            CellRangeAddress region = new CellRangeAddress(cellModel.getStartRow(),
                                    cellModel.getEndRow(), cellModel.getStartColumn(), cellModel.getEndColumn());
                            sheet.addMergedRegion(region);
                            RegionUtil.setBorderBottom(1, region, sheet, workbook);
                            RegionUtil.setBorderLeft(1, region, sheet, workbook);
                            RegionUtil.setBorderRight(1, region, sheet, workbook);
                            RegionUtil.setBorderTop(1, region, sheet, workbook);
                        }

                        if (cellModel.getWidth() != null) {
                            sheet.setColumnWidth(cellModel.getStartColumn(), cellModel.getWidth() * 256);
                        } else {
                            sheet.setColumnWidth(cellModel.getStartColumn(), cellModel.getCellName().getBytes().length * 256);
                        }
                    }
                }
            }

            // 定义数据内容单元格样式
            CellStyle cellDataStyle = workbook.createCellStyle();
            cellDataStyle.setAlignment(CellStyle.ALIGN_CENTER);
            cellDataStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
            cellDataStyle.setWrapText(true);
            cellDataStyle.setBorderBottom(CellStyle.BORDER_THIN);
            cellDataStyle.setBorderLeft(CellStyle.BORDER_THIN);
            cellDataStyle.setBorderRight(CellStyle.BORDER_THIN);
            cellDataStyle.setBorderTop(CellStyle.BORDER_THIN);

            // 计算数据开始的行号
            int startRow = skipRows + 1;
            Iterator<?> iterator = data.iterator();

            if (iterator.hasNext()) {
                Object firstItem = iterator.next();
                Class<?> clazz = firstItem.getClass();
                Field[] fields = clazz.getDeclaredFields();

                // 使所有字段可访问
                for (Field field : fields) {
                    field.setAccessible(true);
                }

                int rowIndex = startRow;
                do {
                    Object item = firstItem;

                    if (rowIndex > startRow) {
                        if (!iterator.hasNext()) break;
                        item = iterator.next();
                    }

                    // 创建一行
                    Row row = sheet.createRow(rowIndex++);
                    int fieldIndex = 0;

                    // 填充数据
                    for (Field field : fields) {
                        Object value = null;
                        try {
                            value = field.get(item);
                        } catch (IllegalAccessException e) {
                            e.printStackTrace();
                            continue;
                        }

                        Cell cell = row.createCell(fieldIndex++);
                        if (value != null) {
                            if (value instanceof String) {
                                cell.setCellValue((String) value);
                            } else if (value instanceof Double) {
                                cell.setCellValue((Double) value);
                            } else if (value instanceof Long) {
                                cell.setCellValue((Long) value);
                            } else if (value instanceof Integer) {
                                cell.setCellValue((Integer) value);
                            } else if (value instanceof Boolean) {
                                cell.setCellValue((Boolean) value);
                            } else if (value instanceof Date) {
                                cell.setCellValue((Date) value);
                                cell.setCellStyle(cellDataStyle);
                            } else if (value instanceof Float) {
                                cell.setCellValue((Float) value);
                            } else if (value instanceof BigDecimal) {
                                cell.setCellValue(((BigDecimal) value).doubleValue());
                            } else {
                                cell.setCellValue(value.toString());
                            }
                        } else {
                            cell.setCellValue("");
                        }
                        cell.setCellStyle(cellDataStyle);
                    }
                } while (iterator.hasNext());
            }

            // 设置响应头以下载文件
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(fileName, "UTF-8"));
            response.setCharacterEncoding("UTF-8");

            // 将工作簿写入输出流
            out = response.getOutputStream();
            workbook.write(out);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            // 关闭资源
            if (workbook != null) {
                try {
                    workbook.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }
}
