package com.ejianc.business.procBid.utils;

import java.time.*;
import java.time.temporal.ChronoUnit;
import java.util.Date;

/**
 * 日期工具类（Java8+ 新API实现，天级精度天数计算）
 */
public class DateUtils {

    /**
     * 计算计划定标时间与当前时间的剩余天数（自然日，仅天级精度）
     * @param planBidTime 计划定标时间（可为null）
     * @param currentTime 当前时间
     * @return 剩余天数（已过期返回0，非负整数）
     */
    public static int getDaysBetween(Date planBidTime, Date currentTime) {
        // 1. 空值处理：计划定标时间为空，返回0（可根据业务调整为null）
        if (planBidTime == null || currentTime == null) {
            return 0;
        }

        // 2. 转换为LocalDate（自动忽略时分秒，仅保留年月日，保证天级精度）
        LocalDate planDate = convertToLocalDate(planBidTime);
        LocalDate currentDate = convertToLocalDate(currentTime);

        // 3. 计算两个日期的天数差（planDate - currentDate）
        long daysDiff = ChronoUnit.DAYS.between(currentDate, planDate);

        // 4. 边界处理：已过期则返回0，否则返回计算的天数
        return daysDiff < 0 ? 0 : (int) daysDiff;
    }

    /**
     * 将java.util.Date转换为LocalDate（指定东八区时区，避免时区偏差）
     */
    private static LocalDate convertToLocalDate(Date date) {
        return date.toInstant()
                .atZone(ZoneId.of("Asia/Shanghai")) // 固定东八区，适配业务场景
                .toLocalDate();
    }

}
