package com.ejianc.business.othprice.service.impl;

import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.othprice.bean.OtherInquiryEntity;
import com.ejianc.business.othprice.bean.OtherQuoteEntity;
import com.ejianc.business.othprice.consts.InquiryStateEnum;
import com.ejianc.business.othprice.mapper.OtherQuoteMapper;
import com.ejianc.business.othprice.service.IOtherInquiryService;
import com.ejianc.business.othprice.service.IOtherQuoteDetailService;
import com.ejianc.business.othprice.service.IOtherQuoteService;
import com.ejianc.business.othprice.vo.OtherQuoteVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.EnvironmentTools;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

/**
 * 报价记录
 *
 * @author baipengyan
 */
@Service("otherQuoteService")
public class OtherQuoteServiceImpl extends BaseServiceImpl<OtherQuoteMapper, OtherQuoteEntity> implements IOtherQuoteService {
	private static final Logger logger = LoggerFactory.getLogger(OtherQuoteServiceImpl.class);
	private static final String DEFAULT_TEMPLATE_CODE = "SMS_195335074";
	private static final String TEMPLATE_CODE = "templateCode";
	private static final String VERIFY_CODE_ERROR = "验证码校验失败！";
	@Resource
	private IOtherInquiryService otherInquiryService;
	@Resource
	private EnvironmentTools environmentTools;
	@Resource
	private IOtherQuoteDetailService otherQuoteDetailService;


	/**
	 * 校验当前时间是否超过询价单报价截止时间
	 *
	 * @param inquiryId 询价单id
	 */
	@Override
	public void checkQuoteEndDate(Long inquiryId) {
		Assert.notNull(inquiryId, "询价单id不能为空！");

		OtherInquiryEntity materialInquiryEntity = otherInquiryService.selectById(inquiryId);
		if (materialInquiryEntity == null) {
			throw new BusinessException("该询价单不存在！");
		}
		boolean after = new Date().after(materialInquiryEntity.getQuoteEndDate());
		//未开标状态需要校验，开标即使截止也可报价
		if (after && 0 == materialInquiryEntity.getBidOpenState()) {
			throw new BusinessException("当前时间已超过报价截止时间！");
		}
	}

	/**
	 * 指定供应商登录、所有供应商登录校验查看权限
	 *
	 * @param inquiryId   询价单id
	 * @param phoneNumber 电话号码
	 */
	@Override
	public void checkViewAuthority(Long inquiryId, String phoneNumber) {
		Assert.notNull(inquiryId, "询价单id不能为空！");
		Assert.hasText(phoneNumber, "电话号码不能为空！");

		OtherInquiryEntity materialInquiryEntity = otherInquiryService.selectById(inquiryId);
		if (materialInquiryEntity == null) {
			throw new BusinessException("该询价单不存在！");
		}

		// 查看权限
		if (materialInquiryEntity.getViewAuthority() == 1) {
			// 1、校验该手机号是否在询价单表体指定供应商手机号列表里
			materialInquiryEntity.getOtherInquirySupplierList()
					.stream()
					.filter(materialInquirySupplierEntity -> phoneNumber.equals(materialInquirySupplierEntity.getTelephone()))
					.findFirst()
					.orElseThrow(() -> new BusinessException("该手机号无权限请切换其他号码重试！"));
		}
	}

	/**
	 * 指定供应商报价校验报价权限
	 *
	 * @param inquiryId   询价单id
	 * @param phoneNumber 电话号码
	 */
	@Override
	public void checkQuotationAuthority(Long inquiryId, String phoneNumber) {
		Assert.notNull(inquiryId, "询价单id不能为空！");
		Assert.hasText(phoneNumber, "电话号码不能为空！");

		OtherInquiryEntity materialInquiryEntity = otherInquiryService.selectById(inquiryId);
		if (materialInquiryEntity == null) {
			throw new BusinessException("该询价单不存在！");
		}

		// 报价权限
		if (materialInquiryEntity.getQuotationAuthority() == 1) {
			materialInquiryEntity.getOtherInquirySupplierList()
					.stream()
					.filter(materialInquirySupplierEntity -> phoneNumber.equals(materialInquirySupplierEntity.getTelephone()))
					.findFirst()
					.orElseThrow(() -> new BusinessException("该手机号无报价权限，请切换登录手机号"));
		}
	}

	/**
	 * 生成验证码
	 *
	 * @param phoneNumber  电话号码
	 * @param templateCode 模板代码
	 *
	 * @throws Exception 异常
	 */
	public void sendVerifyCode(String phoneNumber, String templateCode) throws Exception {
		Assert.hasText(phoneNumber, "电话号码不能为空！");

		JSONObject messageParam = new JSONObject();
		messageParam.put("phone", phoneNumber);
		if (StringUtils.isBlank(templateCode)) {
			messageParam.put(TEMPLATE_CODE, DEFAULT_TEMPLATE_CODE);
		} else {
			messageParam.put(TEMPLATE_CODE, templateCode);
		}

		String messageUrl = environmentTools.getBaseHost() + "/ejc-message-web/no_auth/sms/sendMessage";
		String responseStr = HttpTookit.postByJson(messageUrl, JSON.toJSONString(messageParam));
		logger.info("向手机号[{}]发送验证码结果：[{}]", phoneNumber, responseStr);
		CommonResponse<String> response = JSON.parseObject(responseStr, CommonResponse.class);
		if (!response.isSuccess()) {
			throw new BusinessException("短信发送失败！");
		}
	}

	/**
	 * 校验验证码
	 *
	 * @param phoneNumber  电话号码
	 * @param validate     验证码
	 * @param templateCode 模板代码
	 */
	public void checkVerifyCode(String phoneNumber, String validate, String templateCode) {
		Assert.hasText(phoneNumber, "电话号码不能为空！");
		Assert.hasText(validate, "验证码不能为空！");

		JSONObject jsonObject = new JSONObject();
		if (StringUtils.isBlank(templateCode)) {
			jsonObject.put(TEMPLATE_CODE, DEFAULT_TEMPLATE_CODE);
		} else {
			jsonObject.put(TEMPLATE_CODE, templateCode);
		}
		jsonObject.put("phone", phoneNumber);
		jsonObject.put("validate", validate);

		String checkMessageUrl = environmentTools.getBaseHost() + "/ejc-message-web/no_auth/sms/checkMessage";
		HttpResponse response = HttpUtil.createPost(checkMessageUrl)
				.body(jsonObject.toJSONString())
				.execute();
		if (!response.isOk()) {
			throw new BusinessException(VERIFY_CODE_ERROR);
		}
		String responseStr = response.body();
		if (StringUtils.isBlank(responseStr)) {
			throw new BusinessException(VERIFY_CODE_ERROR);
		}
		CommonResponse<String> checkResponse = JSON.parseObject(responseStr, CommonResponse.class);
		if (0 != checkResponse.getCode()) {
			logger.info("校验验证码失败，失败原因：{}", checkResponse.getMsg());
			throw new BusinessException(checkResponse.getMsg());
		}
	}

	/**
	 * 获取验证码
	 *
	 * @param inquiryId   询价单id
	 * @param phoneNumber 电话号码
	 */
	@Override
	public void fetchVerifyCode(Long inquiryId, String phoneNumber) throws Exception {
		Assert.notNull(inquiryId, "询价单id不能为空！");
		Assert.hasText(phoneNumber, "电话号码不能为空！");

		OtherInquiryEntity materialInquiryEntity = otherInquiryService.selectById(inquiryId);
		if (materialInquiryEntity == null) {
			throw new BusinessException("该询价单不存在！");
		}
		// 2、发送验证码
		this.sendVerifyCode(phoneNumber, null);
	}

	/**
	 * 登录
	 *
	 * @param phoneNumber 电话号码
	 * @param validate    验证码
	 */
	@Override
	public void login(String phoneNumber, String validate) {
		Assert.hasText(phoneNumber, "电话号码不能为空！");
		Assert.hasText(validate, "验证码不能为空！");

		// 1、校验验证码
		this.checkVerifyCode(phoneNumber, validate, null);
	}

	/**
	 * 收回
	 *
	 * @param id 报价单id
	 */
	@Override
	public void withdraw(Long id) {
		Assert.notNull(id, "报价单id不能为空！");
		OtherQuoteEntity entity = super.selectById(id);
		if (entity == null) {
			throw new BusinessException("该报价单不存在！");
		}
		// 1、当询价单已定标时，记录不可收回，提示“该询价已结束，报价记录不可收回！”
		LambdaQueryWrapper<OtherInquiryEntity> wrapper = Wrappers.<OtherInquiryEntity>lambdaQuery()
				.eq(OtherInquiryEntity::getId, entity.getInquiryId())
				.eq(OtherInquiryEntity::getInquiryState, InquiryStateEnum.QUOTATION_STATE.getCode());
		OtherInquiryEntity materialInquiryEntity = otherInquiryService.getOne(wrapper);
		if (materialInquiryEntity != null) {
			entity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
			entity.setQuoteTime(null);
			super.updateById(entity);
		}else {
			throw new BusinessException("报价时间已截止，报价记录不可收回！");
		}
	}

	/**
	 * 修改单据状态
	 *
	 * @param id        id
	 * @param billState 单据状态
	 */
	@Override
	public void changeStatus(Long id, Integer billState) {
		Assert.notNull(id, "id不能为空！");
		Assert.notNull(billState, "单据状态不能为空！");

		OtherQuoteEntity entity = super.selectById(id);
		if (entity == null) {
			throw new BusinessException("该报价单不存在！");
		}
		LambdaQueryWrapper<OtherInquiryEntity> wrapper = Wrappers.<OtherInquiryEntity>lambdaQuery()
				.eq(OtherInquiryEntity::getId, entity.getInquiryId())
				.eq(OtherInquiryEntity::getInquiryState, InquiryStateEnum.QUOTATION_STATE.getCode());
		OtherInquiryEntity materialInquiryEntity = otherInquiryService.getOne(wrapper);
		if (materialInquiryEntity != null) {
			entity.setBillState(billState);
			if (BillStateEnum.COMMITED_STATE.getBillStateCode().equals(billState)) {
				entity.setQuoteTime(new Date());
			} else {
				entity.setQuoteTime(null);
			}
			super.updateById(entity);
		}else {
			throw new BusinessException("报价时间已截止，报价记录不可修改！");
		}
	}

	/**
	 * 最近一次的报价记录
	 *
	 * @param phoneNumber 报价手机号码（当前登录的手机号码）
	 *
	 * @return {@link OtherQuoteVO}
	 */
	@Override
	public OtherQuoteVO recentQuote(String phoneNumber) {
		Assert.hasText(phoneNumber, "电话号码不能为空！");

		LambdaQueryWrapper<OtherQuoteEntity> lambdaQueryWrapper = Wrappers.<OtherQuoteEntity>lambdaQuery()
				.eq(OtherQuoteEntity::getQuotePhone, phoneNumber)
				.orderByDesc(OtherQuoteEntity::getQuoteTime)
				.last("limit 1");
		List<OtherQuoteEntity> list = super.list(lambdaQueryWrapper);
		if (CollectionUtils.isEmpty(list)) {
			return null;
		}
		return BeanMapper.map(list.get(0), OtherQuoteVO.class);
	}
}
