package com.ejianc.business.prjfinance.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.prjfinance.bean.RiskMarginEntity;
import com.ejianc.business.prjfinance.service.IRiskMarginService;
import com.ejianc.business.prjfinance.vo.RiskMarginVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 风险保证金控制器
 *
 * @author CJ
 * @Description:
 * @date 2021/7/19 11:31
 */
@RestController
@RequestMapping(value = "/riskMargin/")
public class RiskMarginController {

    @Autowired
    private IRiskMarginService riskMarginService;

    @Autowired
    private IOrgApi orgApi;

    private final String BILL_CODE_RULE = "RISK_MARGIN_01";

    @Autowired
    private IBillCodeApi billCodeApi;

    /**
     * 分页查询本下的风险保证金
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<JSONObject> pageList(@RequestBody QueryParam queryParam) {
        JSONObject resp = new JSONObject();
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("applySubject");

        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
        }
        List<Long> childIds = new ArrayList<>();
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, childIds));

        IPage<RiskMarginEntity> pageData = riskMarginService.queryPage(queryParam, false);

        resp.put("total", pageData.getTotal());
        resp.put("current", pageData.getCurrent());
        resp.put("size", pageData.getSize());
        resp.put("pages", pageData.getPages());
        resp.put("records", pageData.getRecords());

        //查询保证金累计情况
        Map<String, BigDecimal> sumAmt = riskMarginService.sumCashDeposit(queryParam);
        resp.put("totalDeductCashDeposit", sumAmt.get("totalDeductCashDeposit"));
        resp.put("totalBackCashDeposit", sumAmt.get("totalBackCashDeposit"));

        return CommonResponse.success(resp);
    }

    /**
     * 风险保证金保存/更新
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<RiskMarginVO> saveOrUpdate(@RequestBody RiskMarginVO vo) {
        if(RiskMarginVO.BILL_TYPE_BACK_CASH_DEPOSIT.equals(vo.getBillType())) {
            //退保证金，检查申请金额是否小于等于剩余可退金额
            Map<String, BigDecimal> info = riskMarginService.getProjectCashDepositInfo(vo.getProjectId());
            if(null != vo.getId()) {
                if(info.get("remremainingRefundableAmt").compareTo(BigDecimal.ZERO) < 0) {
                    return CommonResponse.error("保存失败，申请金额大于剩余可退金额。");
                }
            } else if(vo.getApplyAmt().compareTo(info.get("remremainingRefundableAmt")) > 0) {
                return CommonResponse.error("保存失败，申请金额大于剩余可退金额。");
            }
        }
        RiskMarginEntity saveEntity = null;
        if(null == vo.getId()) {
            saveEntity = BeanMapper.map(vo, RiskMarginEntity.class);
            CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE_RULE, InvocationInfoProxy.getTenantid());
            if(!billCodeResp.isSuccess()) {
                return CommonResponse.error("保存失败，自动生成编码失败。");
            }
            saveEntity.setBillCode(billCodeResp.getData());
            saveEntity.setSourceType(RiskMarginVO.SOURCE_TYPE_MAUAL_ENERATION);
            saveEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
        } else {
            saveEntity = riskMarginService.selectById(vo.getId());
            saveEntity.setProjectId(vo.getProjectId());
            saveEntity.setProjectName(vo.getProjectName());
            saveEntity.setApplyAmt(vo.getApplyAmt());
            saveEntity.setApplyDate(vo.getApplyDate());
            saveEntity.setApplySubject(vo.getApplySubject());
            saveEntity.setOperatorId(vo.getOperatorId());
            saveEntity.setOperatorName(vo.getOperatorName());
            saveEntity.setOrgId(vo.getOrgId());
            saveEntity.setOrgName(vo.getOrgName());
        }

        riskMarginService.saveOrUpdate(saveEntity, false);

        RiskMarginVO resp = BeanMapper.map(saveEntity, RiskMarginVO.class);
        //累计扣保证金
        resp.setTotalDeductCashDeposit(vo.getTotalDeductCashDeposit());
        //累计退保证金
        resp.setTotalBackCashDeposit(vo.getTotalBackCashDeposit());
        //保证金结余
        resp.setSpareDepositAmt(vo.getSpareDepositAmt());
        //剩余可退金额
        resp.setRemremainingRefundableAmt(vo.getRemremainingRefundableAmt());
        return CommonResponse.success("保存成功！", resp);
    }

    /**
     * 查询指定的风险保证金详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<RiskMarginVO> queryDetail(@RequestParam(value = "id") Long id) {
        RiskMarginVO resp = null;
        RiskMarginEntity entity = riskMarginService.selectById(id);
        if(null != entity) {
            resp = BeanMapper.map(entity, RiskMarginVO.class);
            Map<String, BigDecimal> info = riskMarginService.getProjectCashDepositInfo(resp.getProjectId());

            //累计扣保证金
            resp.setTotalDeductCashDeposit(info.get("totalDeductCashDeposit"));
            //累计退保证金
            resp.setTotalBackCashDeposit(info.get("totalBackCashDeposit"));
            //保证金结余
            resp.setSpareDepositAmt(info.get("spareDepositAmt"));
            //剩余可退金额
            resp.setRemremainingRefundableAmt(info.get("remremainingRefundableAmt"));
        }
        //查询项目风险保证金情况
        return CommonResponse.success(resp);
    }

    /**
     * 风险保证金删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<RiskMarginVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            riskMarginService.removeByIds(vos.stream().map(RiskMarginVO::getId).collect(Collectors.toList()), false);
        }
        return CommonResponse.success("删除成功！");
    }

    /**
     * 根据项目Id查询生效的风险保证金列表
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "queryList")
    public CommonResponse<JSONObject> queryList(@RequestParam(value = "projectId") Long projectId,
                                                @RequestParam(value = "billType", required = false) String billType) {
        JSONObject resp = new JSONObject();
        List<RiskMarginVO> voList = new ArrayList<>();
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
        if(StringUtils.isNotBlank(billType)) {
            queryParam.getParams().put("billType", new Parameter(QueryParam.EQ, billType));
        }

        //查询生效的
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, Arrays.asList(new Integer[]{BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode()})));
        //按照申请日期倒序排列
        queryParam.getOrderMap().put("applyDate", "DESC");

        List<RiskMarginEntity> eList = riskMarginService.queryList(queryParam);
        if(CollectionUtils.isNotEmpty(eList)) {
            voList.addAll(BeanMapper.mapList(eList, RiskMarginVO.class));
        }
        Map<String, BigDecimal> info = riskMarginService.getProjectCashDepositInfo(projectId);

        resp.put("records", voList);
        //累计扣保证金
        resp.put("totalDeductCashDeposit", info.get("totalDeductCashDeposit"));
        //累计退保证金
        resp.put("totalBackCashDeposit", info.get("totalBackCashDeposit"));
        //保证金结余
        resp.put("spareDepositAmt", info.get("spareDepositAmt"));
        //剩余可退金额
        resp.put("remremainingRefundableAmt", info.get("remremainingRefundableAmt"));
        return CommonResponse.success(resp);
    }

    /**
     * 查询项目风险保证金信息
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "getProjectCashDepositInfo")
    public CommonResponse<Map<String, BigDecimal>> getProjectCashDepositInfo(@RequestParam(value = "projectId") Long projectId) {
        Map<String, BigDecimal> resp = riskMarginService.getProjectCashDepositInfo(projectId);

        return CommonResponse.success(resp);
    }

    /**
     * 风险保证金列表导出
     *
     * @param queryParam
     * @param response
     */
    @PostMapping(value = "excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("applySubject");
        queryParam.setPageIndex(1);
        queryParam.setPageSize(10000);

        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
        }

        List<Long> childIds = new ArrayList<>();
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, childIds));
        IPage<RiskMarginEntity> pageData = riskMarginService.queryPage(queryParam, false);
        List<RiskMarginVO> voList = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(pageData.getRecords())) {
            pageData.getRecords().forEach(risk -> {
                RiskMarginVO r = BeanMapper.map(risk, RiskMarginVO.class);
                r.setBillStateName(BillStateEnum.getEnumByStateCode(r.getBillState()).getDescription());
                r.setBillType(RiskMarginVO.BILL_TYPE_DEDUCT_CASH_DEPOSIT.equals(r.getBillType()) ? "扣保证金" : "退保证金");
                r.setSourceType(RiskMarginVO.SOURCE_TYPE_AUTOG_ENERATION.equals(r.getSourceType()) ? "自动生成" : "手动生成");
                voList.add(r);
            });
        }

        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", voList);
        ExcelExport.getInstance().export("riskMargin-export.xlsx", beans, response);
    }

}
