package com.ejianc.business.api;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.win.bean.WinEntity;
import com.ejianc.business.win.enums.WinEnum;
import com.ejianc.business.win.service.IWinService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IDefdocApi;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.io.Serializable;
import java.math.BigDecimal;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author baipengyan
 * @date 2022/8/8
 * @description 中标情况统计
 */
@RestController
@RequestMapping("winCase")
public class WinCaseController implements Serializable {
	private static final long serialVersionUID = -25982881142544835L;

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final IOrgApi iOrgApi;
	private final IDefdocApi defdocApi;
	private final SessionManager sessionManager;
	private final IWinService winService;
	private static final Long DEF_ID = 315181000557633607L;
	private static final String SPECIALITY = "speciality";
	private static final String VALUE = "value";
	private static final String SUM = "sum";

	public WinCaseController(IOrgApi iOrgApi, IDefdocApi defdocApi, SessionManager sessionManager, IWinService winService) {
		this.iOrgApi = iOrgApi;
		this.defdocApi = defdocApi;
		this.sessionManager = sessionManager;
		this.winService = winService;
	}


	/**
	 * 获取饼状图信息（组织本下）
	 *
	 * @param time 1：月度  2：季度  3：年度
	 *
	 * @return List<WinCaseVO>
	 */
	@GetMapping(value = "/fetchWinCase")
	public CommonResponse<List<WinCaseVO>> fetchWinCase(@RequestParam(value = "time") Integer time) {
		CommonResponse<List<DefdocDetailVO>> response = defdocApi.getDefDocByDefId(DEF_ID);
		if (!response.isSuccess() || CollectionUtils.isEmpty(response.getData())) {
			throw new BusinessException("根据档案id：" + DEF_ID + "，查询失败！");
		}
		List<DefdocDetailVO> defdocDetails = response.getData();
		Map<Long, DefdocDetailVO> defdocDetailMap = defdocDetails.stream().collect(Collectors.toMap(DefdocDetailVO::getId, Function.identity()));

		DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
		LocalDate nowDate = LocalDate.now();

		QueryWrapper<WinEntity> qw = Wrappers.query();
		qw.eq("tenant_id", InvocationInfoProxy.getTenantid());
		qw.eq("dr", 0);
		qw.in("bid_unit_id", iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
		qw.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		qw.eq("bid_win_flag", WinEnum.YES.getCode());
		if (time != null) {
			if (time == 1) {
				qw.between("handle_date", getMonthStartTime(dateTimeFormatter), getMonthEndTime(dateTimeFormatter));
			} else if (time == 2) {
				qw.between("handle_date", getFromThisQuarterStartTime(nowDate, dateTimeFormatter), getFromThisQuarterEndTime(nowDate, dateTimeFormatter));
			} else {
				qw.between("handle_date", getFromThisYearStartTime(nowDate, dateTimeFormatter), getFromThisYearEndTime(nowDate, dateTimeFormatter));
			}
		}
		qw.select("project_type_id as speciality, sum(bid_win_mny) as value, count(*) as sum");
		qw.groupBy("project_type_id");
		List<Map<String, Object>> map = winService.listMaps(qw);

		ArrayList<WinCaseVO> winCaseVOS = new ArrayList<>();
		for (Map<String, Object> mp : map) {
			WinCaseVO winCaseVO = new WinCaseVO();
			if (defdocDetailMap.containsKey(Long.valueOf(mp.get(SPECIALITY).toString()))) {
				DefdocDetailVO defdocDetailVO = defdocDetailMap.get(Long.valueOf(mp.get(SPECIALITY).toString()));
				winCaseVO.setSpeciality(Long.valueOf(mp.get(SPECIALITY).toString()));
				winCaseVO.setName(defdocDetailVO.getName());
				winCaseVO.setValue(new BigDecimal(mp.get(VALUE).toString()));
				winCaseVO.setSum(Integer.valueOf(mp.get(SUM).toString()));
				winCaseVOS.add(winCaseVO);
			}
		}

		return CommonResponse.success("查询成功！", winCaseVOS);
	}


	/**
	 * 本月开始时间
	 *
	 * @param dateTimeFormatter
	 *
	 * @return
	 */
	public static String getMonthStartTime(DateTimeFormatter dateTimeFormatter) {
		Long currentTime = System.currentTimeMillis();
		String timeZone = "GMT+8:00";
		Calendar cal = Calendar.getInstance();// 获取当前日期
		cal.setTimeZone(TimeZone.getTimeZone(timeZone));
		cal.setTimeInMillis(currentTime);
		cal.add(Calendar.YEAR, 0);
		cal.add(Calendar.MONTH, 0);
		cal.set(Calendar.DAY_OF_MONTH, 1);// 设置为1号,当前日期既为本月第一天
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		LocalDateTime monthStart = LocalDateTime.ofInstant(Instant.ofEpochMilli(cal.getTimeInMillis()), ZoneId.systemDefault());
		return monthStart.format(dateTimeFormatter);
	}

	/**
	 * 本月结束时间
	 *
	 * @return
	 */
	private static String getMonthEndTime(DateTimeFormatter dateTimeFormatter) {
		Long currentTime = System.currentTimeMillis();
		String timeZone = "GMT+8:00";
		Calendar cal = Calendar.getInstance();// 获取当前日期
		cal.setTimeZone(TimeZone.getTimeZone(timeZone));
		cal.setTimeInMillis(currentTime);
		cal.add(Calendar.YEAR, 0);
		cal.add(Calendar.MONTH, 0);
		cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));// 获取当前月最后一天
		cal.set(Calendar.HOUR_OF_DAY, 23);
		cal.set(Calendar.MINUTE, 59);
		cal.set(Calendar.SECOND, 59);
		cal.set(Calendar.MILLISECOND, 999);
		LocalDateTime monthEnd = LocalDateTime.ofInstant(Instant.ofEpochMilli(cal.getTimeInMillis()), ZoneId.systemDefault());
		return monthEnd.format(dateTimeFormatter);
	}


	//获取本季度开始时间
	public static String getFromThisQuarterStartTime(LocalDate nowDate, DateTimeFormatter dateTimeFormatter) {
		Month firstMonthOfQuarter = nowDate.getMonth().firstMonthOfQuarter();
		nowDate = LocalDate.of(nowDate.getYear(), firstMonthOfQuarter, 1);
		LocalDateTime fromThisQuarterStart = LocalDateTime.of(nowDate, LocalTime.MIN);
		return fromThisQuarterStart.format(dateTimeFormatter);

	}

	//获取本季度结束时间
	public static String getFromThisQuarterEndTime(LocalDate nowDate, DateTimeFormatter dateTimeFormatter) {
		Month firstMonthOfQuarter = nowDate.getMonth().firstMonthOfQuarter();
		Month endMonthOfQuarter = Month.of(firstMonthOfQuarter.getValue() + 2);
		nowDate = LocalDate.of(nowDate.getYear(), endMonthOfQuarter, endMonthOfQuarter.length(nowDate.isLeapYear()));
		LocalDateTime fromThisQuarterEnd = LocalDateTime.of(nowDate, LocalTime.MAX);
		return fromThisQuarterEnd.format(dateTimeFormatter);
	}

	/**
	 * 今年起开始时间
	 *
	 * @return
	 */
	public static String getFromThisYearStartTime(LocalDate nowDate, DateTimeFormatter dateTimeFormatter) {
		LocalDate firstDayOfYear = nowDate.with(TemporalAdjusters.firstDayOfYear());
		LocalDateTime fromThisYearStart = LocalDateTime.of(firstDayOfYear, LocalTime.MIN);
		return fromThisYearStart.format(dateTimeFormatter);
	}

	/**
	 * 今年起结束时间
	 *
	 * @return
	 */
	public static String getFromThisYearEndTime(LocalDate nowDate, DateTimeFormatter dateTimeFormatter) {
		LocalDate lastDayOfYear = nowDate.with(TemporalAdjusters.lastDayOfYear());
		LocalDateTime fromThisYearEnd = LocalDateTime.of(lastDayOfYear, LocalTime.MAX);
		return fromThisYearEnd.format(dateTimeFormatter);
	}

}
