package com.ejianc.business.open.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.bid.bean.ProjectEntity;
import com.ejianc.business.bid.enums.EffectFlagEnum;
import com.ejianc.business.bid.service.IProjectService;
import com.ejianc.business.bid.utils.MybatisPlusColumnResolver;
import com.ejianc.business.open.bean.OpenEntity;
import com.ejianc.business.open.service.IOpenService;
import com.ejianc.business.process.bean.ProcessEntity;
import com.ejianc.business.process.service.IProcessService;
import com.ejianc.business.win.bean.WinEntity;
import com.ejianc.business.win.service.IWinService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

@Service("open")
public class OpenBpmServiceImpl implements ICommonBusinessService {

	private final IOpenService service;
	private final IProjectService projectService;
	private final IProcessService processService;
	private final IWinService winService;
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final MybatisPlusColumnResolver mybatisPlusColumnResolver;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	public OpenBpmServiceImpl(IOpenService service, IProjectService projectService, IProcessService processService, IWinService winService, SessionManager sessionManager, IBillTypeApi billTypeApi, MybatisPlusColumnResolver mybatisPlusColumnResolver) {
		this.service = service;
		this.projectService = projectService;
		this.processService = processService;
		this.winService = winService;
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.mybatisPlusColumnResolver = mybatisPlusColumnResolver;
	}


	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		OpenEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();

		// 区分提交和审批
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 回写审批生效字段
		MybatisPlusColumnResolver.ColumnResolver<ProjectEntity> columnResolver = mybatisPlusColumnResolver.create();
		projectService.updateEffectFlag(entity.getProjectId(), columnResolver.columnToString(ProjectEntity::getOpenFlag), EffectFlagEnum.EFFECT.getCode());

		saveProcess(entity, "开标记录", "open/card");

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		/*
			单据撤回逻辑：
			1、单据操作包含：保存、返回列表、编辑、提交、删除，规则同工程云一致。
			2、单据撤回：当存在审批流程时且第一个人没有审批时，支持撤回；当不存在审批流程且没有下游业务时支持撤回。
			3、单据弃审：审批过程中，支持后台弃审；审批通过后，没有下游业务时支持弃审。
			下游业务包含：中标登记
		*/

		OpenEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 是否被中标登记引用
		LambdaQueryWrapper<WinEntity> wrapper = new LambdaQueryWrapper<>();
		wrapper.eq(WinEntity::getProjectId, entity.getProjectId());
		List<WinEntity> entityList = winService.list(wrapper);
		if (CollectionUtils.isNotEmpty(entityList)) {
			return CommonResponse.error("当前单据已被中标登记引用，不能撤回/弃审！");
		}

		// 回写审批生效字段
		MybatisPlusColumnResolver.ColumnResolver<ProjectEntity> columnResolver = mybatisPlusColumnResolver.create();
		projectService.updateEffectFlag(entity.getProjectId(), columnResolver.columnToString(ProjectEntity::getOpenFlag), EffectFlagEnum.INVALID.getCode());

		delProcess(entity.getId());

		return CommonResponse.success("单据撤回成功");
	}


	/**
	 * 新增投标流程记录
	 *
	 * @param entity      实体
	 * @param billName    单据名称
	 * @param frontendUrl 前端路由
	 */
	private void saveProcess(OpenEntity entity, String billName, String frontendUrl) {
		ProcessEntity processEntity = new ProcessEntity();
		processEntity.setBillId(entity.getId());
		processEntity.setBillName(billName);
		processEntity.setProjectId(entity.getProjectId());
		processEntity.setProjectCode(entity.getProjectCode());
		processEntity.setProjectName(entity.getProjectName());
		processEntity.setFrontendUrl(frontendUrl);
		processService.saveOrUpdate(processEntity);
	}

	/**
	 * 根据单据id删除对应投标流程记录
	 *
	 * @param billId 单据id
	 */
	private void delProcess(Long billId) {
		LambdaQueryWrapper<ProcessEntity> l = new LambdaQueryWrapper<>();
		l.eq(ProcessEntity::getBillId, billId);
		processService.remove(l);
	}

}
