package com.ejianc.business.api;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.bid.bean.BusinessEntity;
import com.ejianc.business.bid.bean.ProjectEntity;
import com.ejianc.business.bid.service.IBusinessService;
import com.ejianc.business.bid.service.IProjectService;
import com.ejianc.business.record.bean.RecordEntity;
import com.ejianc.business.record.service.IRecordService;
import com.ejianc.business.review.bean.ReviewEntity;
import com.ejianc.business.review.service.IReviewService;
import com.ejianc.business.utils.DateUtil;
import com.ejianc.business.win.bean.WinEntity;
import com.ejianc.business.win.enums.WinEnum;
import com.ejianc.business.win.service.IWinService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.io.Serializable;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author baipengyan
 * @date 2022/8/2
 * @description 统计接口
 */
@RestController
@RequestMapping("statistic")
public class StatisticController implements Serializable {
	private static final long serialVersionUID = 7896623284090719460L;

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final IOrgApi iOrgApi;
	private final SessionManager sessionManager;
	private final IBusinessService businessService;
	private final IProjectService projectService;
	private final IRecordService recordService;
	private final IReviewService reviewService;
	private final IWinService winService;

	public StatisticController(IOrgApi iOrgApi, SessionManager sessionManager, IBusinessService businessService, IProjectService projectService, IRecordService recordService, IReviewService reviewService, IWinService winService) {
		this.iOrgApi = iOrgApi;
		this.sessionManager = sessionManager;
		this.businessService = businessService;
		this.projectService = projectService;
		this.recordService = recordService;
		this.reviewService = reviewService;
		this.winService = winService;
	}


	/**
	 * 投标状态
	 *
	 * @param timeFlag thisYear | all
	 *
	 * @return List<StatisticVO>
	 */
	@GetMapping(value = "/getProBidState")
	public CommonResponse<List<StatisticVO>> getProBidState(@RequestParam(value = "orgId") Long orgId,
	                                                        @RequestParam(value = "timeFlag") String timeFlag) {
		List<StatisticVO> arr = new ArrayList<>();
		List<Long> orgIds = iOrgApi.findChildrenByParentId(orgId != null ? orgId : InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());

		// 商机登记
		QueryWrapper<BusinessEntity> business = new QueryWrapper<>();
		business.in("bid_unit_id", orgIds);
		business.eq("thisYear".equals(timeFlag), "YEAR(create_time)", DateUtil.getToYear());

		// 投标⽴项
		QueryWrapper<ProjectEntity> project = new QueryWrapper<>();
		project.in("bid_unit_id", orgIds);
		project.eq("thisYear".equals(timeFlag), "YEAR(create_time)", DateUtil.getToYear());

		// 投标备案
		QueryWrapper<RecordEntity> record = new QueryWrapper<>();
		record.in("bid_unit_id", orgIds);
		record.eq("thisYear".equals(timeFlag), "YEAR(create_time)", DateUtil.getToYear());

		// 投标评审
		QueryWrapper<ReviewEntity> review = new QueryWrapper<>();
		review.in("bid_unit_id", orgIds);
		review.eq("thisYear".equals(timeFlag), "YEAR(create_time)", DateUtil.getToYear());

		// 中标
		QueryWrapper<WinEntity> win = new QueryWrapper<>();
		win.in("bid_unit_id", orgIds);
		win.eq("thisYear".equals(timeFlag), "YEAR(create_time)", DateUtil.getToYear());
		win.eq("bid_win_flag", WinEnum.YES.getCode());

		arr.add(new StatisticVO("商机登记", businessService.count(business)));
		arr.add(new StatisticVO("投标立项", projectService.count(project)));
		arr.add(new StatisticVO("投标备案", recordService.count(record)));
		arr.add(new StatisticVO("投标评审", reviewService.count(review)));
		arr.add(new StatisticVO("中标", winService.count(win)));

		return CommonResponse.success("查询详情数据成功！", arr);
	}


	/**
	 * 预计开标数
	 *
	 * @param timeFlag thisYear
	 *
	 * @return List<StatisticVO>
	 */
	@GetMapping(value = "/getProBidSurvey")
	public CommonResponse<List<StatisticVO>> getProBidSurvey(@RequestParam(value = "timeFlag") String timeFlag,@RequestParam(value = "orgId") Long orgId) {
		List<StatisticVO> arr = new ArrayList<>();
		LocalDate localDate = LocalDate.now();
		LocalDate weekDate = localDate.plus(7, ChronoUnit.DAYS);
		LocalDate monthDate = localDate.plus(30, ChronoUnit.DAYS);
		CommonResponse<List<OrgVO>> childrenByParentId = iOrgApi.findChildrenByParentId(orgId==null?InvocationInfoProxy.getOrgId():orgId);
		if (!childrenByParentId.isSuccess()) {
			logger.error("分页查询失败，获取当前本下组织信息失败, {}", childrenByParentId.getMsg());
			return CommonResponse.error("查询失败，获取组织信息失败！");
		}
		//近30天预计开标数(投标立项)
		QueryWrapper queryMonth = new QueryWrapper();
		queryMonth.eq("tenant_id", InvocationInfoProxy.getTenantid());
		queryMonth.in("bid_unit_id", childrenByParentId.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
//		queryMonth.in("bill_state",Arrays.asList(
//				BillStateEnum.COMMITED_STATE.getBillStateCode(),
//				BillStateEnum.PASSED_STATE.getBillStateCode()
//		));
		queryMonth.ge("bid_open_date", localDate);
		queryMonth.le("bid_open_date", monthDate);

		int month = projectService.count(queryMonth);
		// 2.近7天预计开标数(投标立项)
		QueryWrapper queryWeek = new QueryWrapper();
		queryWeek.eq("tenant_id", InvocationInfoProxy.getTenantid());
		queryWeek.in("bid_unit_id", childrenByParentId.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
		queryWeek.ge("bid_open_date", localDate);
		queryWeek.le("bid_open_date", weekDate);
//		queryWeek.in("bill_state",Arrays.asList(
//				BillStateEnum.COMMITED_STATE.getBillStateCode(),
//				BillStateEnum.PASSED_STATE.getBillStateCode()
//		));
		logger.info("查询近30天预计开标数sql----{}", JSONObject.toJSONString(queryMonth));
		logger.info("查询近近7天预计开标数sql----{}", JSONObject.toJSONString(queryWeek));

		int week = projectService.count(queryWeek);
		arr.add(new StatisticVO("近30天预计开标数", month));
		arr.add(new StatisticVO("近7天预计开标数", week));
		return CommonResponse.success("查询详情数据成功！", arr);
	}

}
