package com.ejianc.business.back.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.back.bean.BackEntity;
import com.ejianc.business.back.service.IBackService;
import com.ejianc.business.pay.bean.PayBackRecordEntity;
import com.ejianc.business.pay.bean.PayEntity;
import com.ejianc.business.pay.service.IPayBackRecordService;
import com.ejianc.business.pay.service.IPayService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;

@Service("back")
public class BackBpmServiceImpl implements ICommonBusinessService {

	private final IBackService service;
	private final IPayService payService;
	private final IPayBackRecordService payBackRecordService;
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	public BackBpmServiceImpl(IBackService service, IPayService payService, IPayBackRecordService payBackRecordService, SessionManager sessionManager, IBillTypeApi billTypeApi) {
		this.service = service;
		this.payService = payService;
		this.payBackRecordService = payBackRecordService;
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
	}


	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		BackEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();

		// 区分提交和审批
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 风险金退还实体转换风险金缴纳的退还记录实体
		PayBackRecordEntity payBackRecord = this.convertToPayBackRecord(entity);

		// 根据风险保证金缴纳id查询，并重新计算风险金退还记录表汇总的退还金额，和剩余金额
		PayEntity pay = payService.selectById(entity.getSourceBondPayId());
		List<PayBackRecordEntity> payBackRecordList = pay.getPayBackRecordList();
		if (CollectionUtils.isNotEmpty(payBackRecordList)) {
			BigDecimal totalBackMny = payBackRecordList.stream().map(PayBackRecordEntity::getSourceBackMny)
					.reduce(payBackRecord.getSourceBackMny(), BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
			pay.setBackMny(totalBackMny);
		} else {
			pay.setBackMny(entity.getCurBackMny());
		}
		pay.setRemainMny(pay.getPayMny().subtract(pay.getBackMny()).setScale(2, RoundingMode.HALF_UP));
		payBackRecordList.add(payBackRecord);
		payService.saveOrUpdate(pay, false);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		/*
			单据撤回逻辑：
			1、单据操作包含：保存、返回列表、编辑、提交、删除，规则同工程云一致。
			2、单据撤回：当存在审批流程时且第一个人没有审批时，支持撤回；当不存在审批流程且没有下游业务时支持撤回。
			3、单据弃审：审批过程中，支持后台弃审；审批通过后，没有下游业务时支持弃审。
			下游业务包含：风险金缴纳
		*/

		BackEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 直接撤回，并删除对应风险金缴纳中的风险金退还记录
		QueryWrapper<PayBackRecordEntity> qw = new QueryWrapper<>();
		qw.eq("bond_pay_id", entity.getSourceBondPayId());
		qw.eq("source_bond_back_id", entity.getId());
		payBackRecordService.remove(qw);

		PayEntity pay = payService.selectById(entity.getSourceBondPayId());
		List<PayBackRecordEntity> payBackRecordList = pay.getPayBackRecordList();
		if (CollectionUtils.isNotEmpty(payBackRecordList)) {
			BigDecimal totalBackMny = payBackRecordList.stream().map(PayBackRecordEntity::getSourceBackMny)
					.reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
			pay.setBackMny(totalBackMny);
		} else {
			pay.setBackMny(BigDecimal.ZERO);
		}
		pay.setRemainMny(pay.getPayMny().subtract(pay.getBackMny()).setScale(2, RoundingMode.HALF_UP));
		payService.saveOrUpdate(pay, false);

		return CommonResponse.success("单据撤回成功");
	}

	/**
	 * 风险金退还信息插入到风险金缴纳的风险金退还记录表
	 *
	 * @param entity 风险金退还实体
	 *
	 * @return PayBackRecordEntity
	 */
	private PayBackRecordEntity convertToPayBackRecord(BackEntity entity) {
		PayBackRecordEntity pbr = new PayBackRecordEntity();
		pbr.setSourceEmployeeId(entity.getEmployeeId());
		pbr.setSourceEmployeeName(entity.getEmployeeName());
		pbr.setCommitDate(entity.getCommitDate());
		pbr.setCommitUserName(entity.getCommitUserName());
		pbr.setCommitUserCode(entity.getCommitUserCode());
		pbr.setEffectiveDate(new Date());
		pbr.setBillCode(entity.getBillCode());
		pbr.setBillName(entity.getBillName());
		pbr.setBillType(entity.getBillType());
		pbr.setBillState(entity.getBillState());
		pbr.setBillStateName(entity.getBillStateName());
		pbr.setSourceBondBackId(entity.getId());
		pbr.setSourceBillCode(entity.getBillCode());
		pbr.setSourceBillName(entity.getBillName());
		pbr.setSourceBackMny(entity.getCurBackMny());
		pbr.setSourceBackDate(entity.getBackDate());
		pbr.setSourceBackMemo(entity.getMemo());
		pbr.setBondPayId(entity.getSourceBondPayId());
		return pbr;
	}

}
