package com.ejianc.business.probuilddiary.ledger.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.probuilddiary.ledger.bean.LedgerDetailEntity;
import com.ejianc.business.probuilddiary.ledger.bean.LedgerEntity;
import com.ejianc.business.probuilddiary.ledger.service.ILedgerDetailService;
import com.ejianc.business.probuilddiary.ledger.service.ILedgerService;
import com.ejianc.business.probuilddiary.ledger.vo.LedgerVO;
import com.ejianc.business.probuilddiary.project.service.IProjectLogService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 日志台账
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("ledger")
public class LedgerController implements Serializable {
	private static final long serialVersionUID = 1889016669879547986L;
	private static final String RULE_CODE = "PRO-BUILD-DIARY-LEDGER";
	private static final String BILL_CODE = "EJCBT202209000020";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final ILedgerService service;
	private final ILedgerDetailService ledgerDetailService;
	private final IProjectLogService projectLogService;

	public LedgerController(SessionManager sessionManager, IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, ILedgerService service, ILedgerDetailService ledgerDetailService, IProjectLogService projectLogService) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
		this.ledgerDetailService = ledgerDetailService;
		this.projectLogService = projectLogService;
	}


	/**
	 * 保存修改
	 *
	 * @param saveOrUpdateVO vo
	 *
	 * @return CommonResponse<LedgerVO>
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<LedgerVO> saveOrUpdate(@RequestBody LedgerVO saveOrUpdateVO) {
		LedgerEntity entity = BeanMapper.map(saveOrUpdateVO, LedgerEntity.class);
		if (null == entity.getId() || 0 == entity.getId()) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}
		entity.setBillName("日志台账");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());
		service.saveOrUpdate(entity, false);
		return CommonResponse.success("保存或修改单据成功！", BeanMapper.map(service.selectById(entity.getId()), LedgerVO.class));
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return CommonResponse<LedgerVO>
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<LedgerVO> queryDetail(@RequestParam(value = "id", required = true) Long id) {
		LedgerEntity entity = service.selectById(id);
		LedgerVO vo = BeanMapper.map(entity, LedgerVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 删除
	 *
	 * @param vos 待删除的vos
	 *
	 * @return CommonResponse<String>
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<LedgerVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (LedgerVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(LedgerVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 请求参数
	 *
	 * @return CommonResponse<JSONObject>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("parentOrgName");
		fuzzyFields.add("projectName");
		fuzzyFields.add("memo");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
				return CommonResponse.error("查询失败，获取组织信息失败！");
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}

		IPage<LedgerEntity> page = service.queryPage(param, false);
		List<LedgerVO> ledgerVOS = BeanMapper.mapList(page.getRecords(), LedgerVO.class);
		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", ledgerVOS);
		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 响应头
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		param.setPageIndex(1);
		param.setPageSize(-1);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("projectName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}

		IPage<LedgerEntity> page = service.queryPage(param, false);
		List<LedgerVO> ledgerVOS = new ArrayList<>();
		page.getRecords().forEach(e -> {
			LedgerVO ledgerVO = BeanMapper.map(e, LedgerVO.class);
			ledgerVO.setBillStateName(BillStateEnum.getEnumByStateCode(ledgerVO.getBillState()).getDescription());
			ledgerVOS.add(ledgerVO);
		});
		Map<String, Object> beans = new HashMap<>();
		beans.put("records", ledgerVOS);
		ExcelExport.getInstance().export("ledger-export.xlsx", beans, response);
	}


	/**
	 * 查询验收单据数量
	 *
	 * @param jsonObject 参数
	 *
	 * @return CommonResponse<Map < String, Integer>>
	 */
	@PostMapping(value = "/countBillNum")
	public CommonResponse<Map<String, Integer>> countBillNum(@RequestBody JSONObject jsonObject) {
		return CommonResponse.success("查询成功！", projectLogService.countBillNum(jsonObject));
	}


	/**
	 * 查询其他子表数据
	 *
	 * @param id 主键
	 *
	 * @return LedgerSubVO
	 */
	@GetMapping(value = "/fetchOtherSubDetail")
	public CommonResponse<List<JSONObject>> fetchOtherSubDetail(@RequestParam(value = "id") Long id) {
		return CommonResponse.success("查询成功！", service.fetchOtherSubDetail(id));
	}

	/**
	 * 日志台账-施工内容（施工员）分页查询
	 *
	 * @param id        主键
	 * @param pageIndex 当前页
	 * @param pageSize  当前分页总页数
	 *
	 * @return CommonResponse<IPage < LedgerDetailVO>>
	 */
	@GetMapping(value = "/ledgerDetailQueryPage")
	public CommonResponse<JSONObject> ledgerDetailQueryPage(@RequestParam(value = "id", required = true) Integer id,
	                                                        @RequestParam(value = "pageIndex", defaultValue = "1") Integer pageIndex,
	                                                        @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize) {
		QueryParam param = new QueryParam();
		param.setPageIndex(pageIndex);
		param.setPageSize(pageSize);
		param.getParams().put("ledger_id", new Parameter(QueryParam.EQ, id));
		IPage<LedgerDetailEntity> page = ledgerDetailService.queryPage(param);
		List<LedgerVO> ledgerVOS = BeanMapper.mapList(page.getRecords(), LedgerVO.class);

		JSONObject resp = new JSONObject();
		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", ledgerVOS);
		return CommonResponse.success("查询分页成功！", resp);
	}

}
