package com.ejianc.business.proequipmentcorpout.contract.service.impl;

import cn.hutool.http.HttpRequest;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.ejianc.business.proequipmentcorpout.contract.bean.OutRentContractChangeEntity;
import com.ejianc.business.proequipmentcorpout.constants.OutRentConstants;
import com.ejianc.business.proequipmentcorpout.contract.service.IOutRentContractChangeAsyncService;
import com.ejianc.business.proequipmentcorpout.contract.service.IOutRentContractChangeService;
import com.ejianc.business.signaturemanage.api.ISignatureCommonApi;
import com.ejianc.business.signaturemanage.vo.WatermarkVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.ParamRegisterSetVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Recover;
import org.springframework.retry.annotation.Retryable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;

/**
 * @author baipengyan
 * @date 2022/5/20
 * @description 合同加水印异步http请求
 */
@Service("outContractChangeAsyncService")
public class OutRentContractChangeAsyncServiceImpl implements IOutRentContractChangeAsyncService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${common.env.base-host}")
    private String BaseHost;
    @Autowired
    private IOutRentContractChangeService service;
    @Autowired
    private IParamConfigApi paramConfigApi;
    @Autowired
    ISignatureCommonApi signatureCommonApi;

    private static final String WATERMARK_CHECK_PARAM_NAME = "P-00a9W886"; // 水印参数

    /**
     * 根据变更合同id给水印文件赋值
     *
     * @param id 变更合同id
     */
    @Override
    public void asyncWatermarkById(Long id) {
        OutRentContractChangeEntity changeEntity = service.selectById(id);
        logger.info("开始获取水印参数信息，变更合同id：{}--------", id);
        // 水印系统参数
        CommonResponse<ParamRegisterSetVO> response = paramConfigApi.getByCode(WATERMARK_CHECK_PARAM_NAME);
        logger.info("获取水印系统参数请求结果：{}", JSONObject.toJSONString(response));
        if (!response.isSuccess() || response.getData() == null) {
            throw new BusinessException("获取水印系统参数请求失败，失败原因：" + response.getMsg());
        }
        String valueData = response.getData().getValueData();
        Assert.hasText(valueData, "获取的水印系统参数不能为空!");

        // 是否限制： 0:不限制，1:限制
        if ("0".equals(valueData)) {
//            String billType = "EJCBT202210000035";
//            String sourceType = "equipmentOutRent";
            // 转换水印参数配置：默认规则
            WatermarkVO watermarkVO = signatureCommonApi.fetchWatermarkConfig(changeEntity.getChangeFileId(), changeEntity.getId(),
                    changeEntity.getCode(), OutRentConstants.BILL_TYPE_OUT_CONTRACT_CHANGE, OutRentConstants.FILE_SOURCE_TYPE_OUT_CONTRACT_WATER);
            Assert.notNull(watermarkVO, "获取水印系统参数失败!");
            // 获取上下文并异步调用添加水印
            HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
            HashMap<String, String> headers = new HashMap<>();
            headers.put("authority", request.getHeader("authority"));
            headers.put("ejc-token", request.getHeader("ejc-token"));
            this.fetchWatermarkAttachment(headers, watermarkVO);
        }
    }
    /**
     * 获取水印文件附件信息
     *
     * @param watermarkVO 水印配置参数
     * @return 水印文件信息
     */
    @Override
    @Async(value = "commonTask")
    @Retryable(maxAttempts = 3, backoff = @Backoff(delay = 2000, multiplier = 1.5), stateful = true)
    public void fetchWatermarkAttachment(HashMap<String, String> headers, WatermarkVO watermarkVO) {
        logger.info("异步获取水印文件附件信息开始，入参：{}", JSON.toJSONString(watermarkVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        Stopwatch start = Stopwatch.createStarted();
        String body = HttpRequest.post(BaseHost + "ejc-signaturemanage-web/common/fetchWatermarkAttachment")
                .addHeaders(headers)
                .body(JSON.toJSONString(watermarkVO))
                .timeout(60000)
                .execute().body();
        AttachmentVO attachmentVO = JSON.parseObject(body, AttachmentVO.class);
        logger.info("异步获取水印文件附件信息结束，请求签章中心耗时「{}」秒，文件中心的水印文件信息：{}", start.stop().elapsed(TimeUnit.SECONDS), JSON.toJSONString(attachmentVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        start.reset().start();
        // 更新合同水印文件id
        UpdateWrapper<OutRentContractChangeEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", watermarkVO.getBillId());
        updateWrapper.set("watermark_contract_file_id", attachmentVO.getId());
        boolean flag = service.update(updateWrapper);
        if (flag) {
            logger.info("获取水印文件附件信息并更新合同成功，写入数据库耗时「{}」秒，合同id：{}，原合同文件id：{}，水印合同文件id：{}", start.stop().elapsed(TimeUnit.SECONDS), watermarkVO.getBillId(), watermarkVO.getFileId(), attachmentVO.getId());
        }
    }

    @Recover
    public void recordWatermark(Exception e, HashMap<String, String> headers, WatermarkVO watermarkVO) {
        logger.error("获取水印文件附件信息并更新合同失败，请求参数：{}", JSON.toJSONString(watermarkVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
    }
}
