package com.ejianc.business.proequipmentcorpout.outLedger.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.proequipmentcorpout.appearance.bean.OutAppearanceEntity;
import com.ejianc.business.proequipmentcorpout.appearance.service.IOutAppearanceService;
import com.ejianc.business.proequipmentcorpout.constants.RentOutStatusEnum;
import com.ejianc.business.proequipmentcorpout.outLedger.bean.OutRentParameterDetailEntity;
import com.ejianc.business.proequipmentcorpout.outLedger.bean.OutRentParameterEntity;
import com.ejianc.business.proequipmentcorpout.outLedger.mapper.OutRentParameterMapper;
import com.ejianc.business.proequipmentcorpout.outLedger.service.IOutRentParameterDetailService;
import com.ejianc.business.proequipmentcorpout.outLedger.service.IOutRentParameterService;
import com.ejianc.business.proequipmentcorpout.outLedger.vo.OutRentParameterVO;
import com.ejianc.business.proequipmentcorpout.outrent.Enums.OutRentEquipmentStateEnum;
import com.ejianc.business.proequipmentcorpout.outrent.bean.OutRentEquipmentStartEntity;
import com.ejianc.business.proequipmentcorpout.outrent.bean.OutRentEquipmentStopEntity;
import com.ejianc.business.proequipmentcorpout.outrent.service.IOutRentEquipmentStartService;
import com.ejianc.business.proequipmentcorpout.outrent.service.IOutRentEquipmentStopService;
import com.ejianc.business.proequipmentcorpout.rental.bean.OutRentRentalEntity;
import com.ejianc.business.proequipmentcorpout.rental.service.IOutRentRentalService;
import com.ejianc.business.proequipmentcorprent.rent.enums.RentEquipmentStateEnum;
import com.ejianc.business.proequipmentcorprent.rent.enums.RentParameterTypeEnum;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 租出设备台账服务实现类
 *
 * @author CJ
 * @Description:
 * @date 2022/10/27 14:27
 */
@Service("outRentParameterService")
public class OutRentParameterServiceImpl extends BaseServiceImpl<OutRentParameterMapper, OutRentParameterEntity> implements IOutRentParameterService {

    private static final String BILL_CODE = "EQUIP_OUT_STORE_CORP";

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IOutRentParameterDetailService outRentParameterDetailService;

    /** 启用单据 */
    @Autowired
    private IOutRentEquipmentStartService startService;
    /** 停用单据 */
    @Autowired
    private IOutRentEquipmentStopService stopService;
    /** 设备退场 */
    @Autowired
    private IOutAppearanceService appearanceService;
    /** 设备租金计算 */
    @Autowired
    private IOutRentRentalService rentRentalService;



    @Override
    public void saveOrUpdateOutRentParams(OutRentParameterVO saveOrUpdateVO) {
        OutRentParameterEntity entity = generateEntity(saveOrUpdateVO);
        super.saveOrUpdate(entity, false);
    }

    private OutRentParameterEntity generateEntity(OutRentParameterVO saveOrUpdateVO) {
        OutRentParameterEntity entity = BeanMapper.map(saveOrUpdateVO, OutRentParameterEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            if(StringUtils.isBlank(entity.getCode())) {
                BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
                CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
                if(billCode.isSuccess()) {
                    entity.setCode(billCode.getData());
                }else{
                    throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
                }
            }
        }

        //编码检查
        QueryWrapper<OutRentParameterEntity> checkQuery = new QueryWrapper<>();
        checkQuery.eq("code", entity.getCode());
        if(null != entity.getId()) {
            checkQuery.ne("id", entity.getId());
        }
        OutRentParameterEntity codeEntity = super.getOne(checkQuery);
        if(null != codeEntity) {
            throw new BusinessException("操作失败，编码重复！");
        }
        return entity;
    }

    @Override
    public void saveOrUpdateOutRentParams(List<OutRentParameterVO> pushParamVOs) {
        logger.info("接收到待保存出库台帐数据：{}", JSONObject.toJSONString(pushParamVOs));
        List<String> codelist = new ArrayList<>(10);
        for(OutRentParameterVO pushParamVO : pushParamVOs) {
            if(StringUtils.isNotBlank(pushParamVO.getCode())) {
                if(codelist.contains(pushParamVO.getCode())) {
                    throw new BusinessException("保存失败，编码重复！");
                } else {
                    codelist.add(pushParamVO.getCode());
                }
            }
            super.saveOrUpdate(generateEntity(pushParamVO), false);
        }
    }

    @Override
    public void changeParameter(List<OutRentParameterEntity> parameterList, String sourceType) {
        logger.info("进入设备租出台账变更方法，变更类型：{}，变更参数：{}", RentParameterTypeEnum.getEnumByCode(sourceType).getDescription(),
                JSONObject.toJSONString(parameterList));
        if (CollectionUtils.isEmpty(parameterList)) {
            return;
        }
        // 对数据进行初始化处理
        Map<Long, OutRentParameterEntity> map = new HashMap<>();
        List<OutRentParameterDetailEntity> parameterDetailList = new ArrayList<>();
        for (OutRentParameterEntity entity : parameterList) {
            map.put(entity.getId(), entity);
            // 给子表赋值父表id
            for (OutRentParameterDetailEntity detailEntity : entity.getRentParameterDetailList()) {
                detailEntity.setParameterId(entity.getId());
            }
            parameterDetailList.addAll(entity.getRentParameterDetailList());
        }
        // 查询当前库中已有的数据
        QueryParam queryParam = new QueryParam();
        List<Long> parameterIdList = parameterList.stream().map(OutRentParameterEntity::getId)
                .collect(Collectors.toList());
        queryParam.getParams().put("id", new Parameter(QueryParam.IN, parameterIdList));
        List<OutRentParameterEntity> saveList = super.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(saveList)) {
            for (OutRentParameterEntity saveEntity : saveList) {
                // 变更台账记录
                OutRentParameterEntity entity = map.get(saveEntity.getId());
                saveEntity.setEquipmentState(entity.getEquipmentState());
                saveEntity.setEquipmentStateName(entity.getEquipmentStateName());
                if (RentParameterTypeEnum.启用单.getCode().equals(sourceType)) {
                    saveEntity.setStartDate(entity.getStartDate());
                    if(null!=entity.getMeterRentDate()){
                        saveEntity.setMeterRentDate(entity.getMeterRentDate());
                    }
                }
                else if (RentParameterTypeEnum.停用单.getCode().equals(sourceType)) {
                    saveEntity.setStopDate(entity.getStopDate());
                }
                else if (RentParameterTypeEnum.退场单.getCode().equals(sourceType)) {
                    saveEntity.setOutDate(entity.getOutDate());
                }
                else if(RentParameterTypeEnum.租金计算.getCode().equals(sourceType)){
                    if (RentOutStatusEnum.启用.getCode().equals(saveEntity.getEquipmentState())) {
                        saveEntity.setStartDate(entity.getOperationDate());
                    }
                    else if (RentOutStatusEnum.停用.getCode().equals(saveEntity.getEquipmentState())) {
                        saveEntity.setStopDate(entity.getOperationDate());
                    }
                    else if (RentOutStatusEnum.退场.getCode().equals(saveEntity.getEquipmentState())) {
                        saveEntity.setOutDate(entity.getOperationDate());
                    }
                }
                saveEntity.setOperationDate(new Date());
            }
            logger.info("插入数据------"+JSONObject.toJSONString(saveList));
            super.saveOrUpdateBatch(saveList);
        }
        if (CollectionUtils.isNotEmpty(parameterDetailList)) {
            // 插入操作记录
            outRentParameterDetailService.saveOrUpdateBatch(parameterDetailList);
        }
    }

    @Override
    public void cancelParameter(List<Long> parameterIdList, String sourceType, Long sourceId) {
        logger.info("进入设备台账撤回方法，撤销类型：{}，撤销单据id：{}，撤销参数：{}",
                RentParameterTypeEnum.getEnumByCode(sourceType).getDescription(), sourceId,
                JSONObject.toJSONString(parameterIdList));
        // 删除台账操作记录
        outRentParameterDetailService.delParameterDetail(sourceId, sourceType);
        // 查询删除后最后一条操作记录状态及时间
        QueryParam detailParam = new QueryParam();
        detailParam.getParams().put("parameterId", new Parameter(QueryParam.IN, parameterIdList));
        detailParam.getOrderMap().put("createTime", QueryParam.DESC);
        List<OutRentParameterDetailEntity> parameterDetailList = outRentParameterDetailService.queryList(detailParam, false);
        Map<Long, OutRentParameterDetailEntity> map = new LinkedHashMap<>();
        for (OutRentParameterDetailEntity detailEntity : parameterDetailList) {
            // 因为是按照时间倒序排序，因此每个操作记录第一次进入map时，都是最后一次操作记录
            if (!map.containsKey(detailEntity.getParameterId())) {
                map.put(detailEntity.getParameterId(), detailEntity);
            }
        }
        QueryParam saveParam = new QueryParam();
        saveParam.getParams().put("id", new Parameter(QueryParam.IN, parameterIdList));
        List<OutRentParameterEntity> saveList = super.queryList(saveParam, false);
        logger.info("查询到设备信息：{}", JSONObject.toJSONString(saveList));
        for (OutRentParameterEntity saveEntity : saveList) {
            OutRentParameterDetailEntity detailEntity = map.get(saveEntity.getId());
            if (OutRentEquipmentStateEnum.启用.getCode().equals(saveEntity.getEquipmentState())) {
                saveEntity.setStartDate(detailEntity.getOperationDate());
                saveEntity.setEquipmentState(OutRentEquipmentStateEnum.停用.getCode());
                saveEntity.setEquipmentStateName(OutRentEquipmentStateEnum.停用.getDescription());
                logger.info("设备状态：{}", OutRentEquipmentStateEnum.停用.getDescription());
            }
            else if (OutRentEquipmentStateEnum.停用.getCode().equals(saveEntity.getEquipmentState())) {
                saveEntity.setStopDate(detailEntity.getOperationDate());
                saveEntity.setEquipmentState(OutRentEquipmentStateEnum.启用.getCode());
                saveEntity.setEquipmentStateName(OutRentEquipmentStateEnum.启用.getDescription());
                logger.info("设备状态：{}", OutRentEquipmentStateEnum.启用.getDescription());
            }
            else if (OutRentEquipmentStateEnum.退场.getCode().equals(saveEntity.getEquipmentState())) {
                logger.info("进入退场状态撤回。saveEntity状态{}，detailEntity状态{}", saveEntity.getEquipmentState(), detailEntity.getEquipmentState());
                saveEntity.setEquipmentState(detailEntity.getEquipmentState());
                if (OutRentEquipmentStateEnum.停用.getCode().equals(saveEntity.getEquipmentState())){
                    saveEntity.setEquipmentState(OutRentEquipmentStateEnum.停用.getCode());
                    saveEntity.setEquipmentStateName(OutRentEquipmentStateEnum.停用.getDescription());
                }else {
                    saveEntity.setEquipmentState(OutRentEquipmentStateEnum.启用.getCode());
                    saveEntity.setEquipmentStateName(OutRentEquipmentStateEnum.启用.getDescription());
                }
                saveEntity.setOutDate(detailEntity.getOperationDate());
            }
            // 操作时间修改为撤销时间
            saveEntity.setOperationDate(new Date());
        }
        logger.info("修改后设备信息：{}", JSONObject.toJSONString(saveList));
        super.saveOrUpdateBatch(saveList);
        logger.info("修改保存后设备信息：{}", JSONObject.toJSONString(super.queryList(saveParam)));
    }

    @Override
    public String removeBySourceId(Long sourceId, String sourceBillType) {
        QueryWrapper<OutRentParameterEntity> delWrapper = new QueryWrapper<>();
        delWrapper.eq("source_id", sourceId);
        delWrapper.eq("source_bill_type", sourceBillType);
        List<OutRentParameterEntity> entitys = super.list(delWrapper);
        if(CollectionUtils.isNotEmpty(entitys)) {
         super.removeByIds(entitys.stream().map(OutRentParameterEntity::getId).collect(Collectors.toList()), false);
        }

        return null;
    }

    /**
     * 启用单、停用单、退场单保存前校验，校验不通过则抛出异常
     * <p>1.校验单据类型与当前设备状态，设备启用状态无法新增启用单，设备停用状态无法新增停用单</p>
     * <p>2.互斥性校验，启用单、停用单、退场单同一个合同下只能存在一个非生效状态的单据</p>
     * <p>3.日期校验，三个单据的日期状态是滚动向前的，每次新增单据时，需要查询当前单据的时间是否大于数据库中最新的时间（必须大于验收日期）</p>
     *
     * @param parameterIdList 需要校验的设备台账id列表
     * @param sourceType      单据类型
     * @param sourceId        单据id
     * @param contractId      合同id
     * @param myDate          校验的时间
     */
    @Override
    public void checkValidation(List<Long> parameterIdList, String sourceType, Long sourceId, Long contractId,
                                Date myDate) {
        logger.info("进行单据保存前校验，校验参数：parameterIdList：{}，sourceType：{}，sourceTypeName:{}，sourceId:{}，" +
                        "contractId:{}，myDate：{}", JSONObject.toJSONString(parameterIdList), sourceType,
                RentParameterTypeEnum.getEnumByCode(sourceType).getDescription(), sourceId, contractId, myDate);
        if (CollectionUtils.isEmpty(parameterIdList)) {
            return;
        }
        String sourceTypeName;
        if (RentParameterTypeEnum.启用单.getCode().equals(sourceType)) {
            sourceTypeName = "启用";
        }
        else if (RentParameterTypeEnum.停用单.getCode().equals(sourceType)) {
            sourceTypeName = "停用";
        }
        else if (RentParameterTypeEnum.退场单.getCode().equals(sourceType)) {
            sourceTypeName = "退场";
        }
        else if (RentParameterTypeEnum.租金计算.getCode().equals(sourceType)) {
            sourceTypeName = "租金计算";
        }
        else {
            sourceTypeName = "";
        }
        QueryParam parameterParam = new QueryParam();
        parameterParam.getParams().put("id", new Parameter(QueryParam.IN, parameterIdList));
        List<OutRentParameterEntity> parameterEntityList = super.queryList(parameterParam, false);
        Map<Long,OutRentParameterEntity> parameterEntityMap = new HashMap<>();
        StringBuilder sb = new StringBuilder();
        /*
         * 1.状态校验
         */
        for (OutRentParameterEntity entity : parameterEntityList) {
            parameterEntityMap.put(entity.getId(),entity);
            if ((RentParameterTypeEnum.启用单.getCode().equals(sourceType)
                    && RentOutStatusEnum.启用.getCode().equals(entity.getEquipmentState()))
                    || (RentParameterTypeEnum.停用单.getCode().equals(sourceType)
                    && RentOutStatusEnum.停用.getCode().equals(entity.getEquipmentState()))) {
                sb.append("设备[").append(entity.getEquipmentName()).append("]、");
            }
        }
        if (sb.length() > 0) {
            sb.delete(sb.length() - 1, sb.length());
            sb.append("已").append(sourceTypeName).append("，无法新增单据");
            throw new BusinessException(sb.toString());
        }
        // 未抛出异常，进行第二个校验
        /*
         *  2.互斥性校验
         */
        QueryParam checkQueryParam = new QueryParam();
        checkQueryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        checkQueryParam.getParams().put("billState", new Parameter(QueryParam.IN, Arrays.asList(
                BillStateEnum.UNCOMMITED_STATE.getBillStateCode(),
                BillStateEnum.APPROVING_HAS_STATE.getBillStateCode(),
                BillStateEnum.UNAPPROVED.getBillStateCode(),
                BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode())));
        // 修改的时候排除当前id
        if (sourceId != null) {
            checkQueryParam.getParams().put("id", new Parameter(QueryParam.NE, sourceId));
        }
        List<OutRentEquipmentStartEntity> startEntityList = startService.queryList(checkQueryParam, false);
        if (CollectionUtils.isNotEmpty(startEntityList)) {
            throw new BusinessException("存在未生效的启用单，无法新增单据！");
        }
        List<OutRentEquipmentStopEntity> stopEntityList = stopService.queryList(checkQueryParam, false);
        if (CollectionUtils.isNotEmpty(stopEntityList)) {
            throw new BusinessException("存在未生效的停用单，无法新增单据！");
        }
        List<OutAppearanceEntity> appearanceEntityList = appearanceService.queryList(checkQueryParam, false);
        if (CollectionUtils.isNotEmpty(appearanceEntityList)) {
            throw new BusinessException("存在未生效的退场单，无法新增单据！");
        }
        // 只校验是否存在自动租金计算
        checkQueryParam.getParams().put("rentalType",new Parameter(QueryParam.EQ,"1"));
        List<OutRentRentalEntity> rentalEntityList = rentRentalService.queryList(checkQueryParam, false);
        if (CollectionUtils.isNotEmpty(rentalEntityList)) {
            throw new BusinessException("存在未生效的租金计算单，无法新增单据！");
        }
        /*
         * 3.日期校验
         */
        QueryParam dateParam = new QueryParam();
        dateParam.getParams().put("parameterId", new Parameter(QueryParam.IN, parameterIdList));
        dateParam.getOrderMap().put("createTime", QueryParam.DESC);
        List<OutRentParameterDetailEntity> parameterDetailList = outRentParameterDetailService.queryList(dateParam, false);
        Map<Long, OutRentParameterDetailEntity> map = new LinkedHashMap<>();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        for (OutRentParameterDetailEntity detailEntity : parameterDetailList) {
            // 因为是按照时间倒序排序，因此每个操作记录第一次进入map时，都是最后一次操作记录
            if (!map.containsKey(detailEntity.getParameterId())) {
                map.put(detailEntity.getParameterId(), detailEntity);
                // 如果传入时间在操作时间之前，则抛出异常
                if (!myDate.after(detailEntity.getOperationDate())) {
                    OutRentParameterEntity entity = parameterEntityMap.get(detailEntity.getParameterId());
                    sb.append("设备[").append(entity.getEquipmentName()).append("]")
                            .append(sourceTypeName).append("日期不能小于")
                            .append(sdf.format(detailEntity.getOperationDate())).append(",");
                    // 租金计算直接抛出
                    if (RentParameterTypeEnum.租金计算.getCode().equals(sourceType)) {
                        throw new BusinessException(sourceTypeName + "日期不能小于" + sdf.format(detailEntity.getOperationDate()));
                    }
                }
            }
        }
        if (sb.length() > 0) {
            sb.delete(sb.length() - 1, sb.length());
            throw new BusinessException(sb.toString());
        }
    }

    /**
     * 根据合同id查询当前合同下所有的设备台账，如果有做过租金计算，则去上次租金计算日期到当次租金计算日期数据，
     * 如果未做过租金计算，则取入场日期到当次租金计算日期数据
     *
     * @param contractId 合同id
     * @param sourceId   来源id
     * @param endDate    截止日期
     * @return 合同下设备台账
     */
    @Override
    public List<OutRentParameterVO> getRentParameterByContractId(Long contractId, Long sourceId, Date endDate) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getParams().put("equipmentState", new Parameter(QueryParam.NE, OutRentEquipmentStateEnum.待启用.getCode()));//排除待启用
        List<OutRentParameterEntity> list = super.queryList(queryParam, false);
        if (CollectionUtils.isEmpty(list)) return null;
        List<Long> parameterIdList = list.stream().map(OutRentParameterEntity::getId).collect(Collectors.toList());
//        Date startDate = rentParameterDetailService.getMaxOperationDate(parameterIdList);
        // 进行日期校验
        this.checkValidation(parameterIdList, RentParameterTypeEnum.租金计算.getCode(), sourceId, contractId, endDate);
        // 查询字表，按照操作时间降序排序
        QueryParam detailParam = new QueryParam();
        detailParam.getParams().put("parameterId", new Parameter(QueryParam.IN, parameterIdList));
//        detailParam.getOrderMap().put("createTime", QueryParam.DESC);
//        if(startDate != null){
//            detailParam.getParams().put("operationDate", new Parameter(QueryParam.GE, startDate));
//        }
        detailParam.getParams().put("operationDate", new Parameter(QueryParam.LE, endDate));
        detailParam.getOrderMap().put("operationDate", QueryParam.DESC);
        detailParam.getOrderMap().put("createTime", QueryParam.DESC);
        List<OutRentParameterDetailEntity> parameterDetailList = outRentParameterDetailService.queryList(detailParam, false);
        // 组装查询结果
        Map<Long, List<OutRentParameterDetailEntity>> map = new HashMap<>();
        for (OutRentParameterDetailEntity detailEntity : parameterDetailList) {
            if (map.containsKey(detailEntity.getParameterId())) {
                map.get(detailEntity.getParameterId()).add(detailEntity);
            }
            else {
                List<OutRentParameterDetailEntity> packList = new ArrayList<>();
                packList.add(detailEntity);
                map.put(detailEntity.getParameterId(), packList);
            }
        }
        List<OutRentParameterDetailEntity> addDetailList;
        for (OutRentParameterEntity entity : list) {
            List<OutRentParameterDetailEntity> detailEntityList = map.get(entity.getId());
            // 对明细结果进行截取，取上次租金计算日期到当次租金计算日期数据
            // 因查询倒序排序，循环到类型为租金计算时，截取数据
            addDetailList = new ArrayList<>();
            for (OutRentParameterDetailEntity detailEntity : detailEntityList) {
                if (RentParameterTypeEnum.租金计算.getCode().equals(detailEntity.getSourceType())) {
                    // 添加上次租金计算日期，做起始日期
                    addDetailList.add(detailEntity);
                    break;
                }
                else {
                    addDetailList.add(detailEntity);
                }
            }
            // 对数据按操作日期做升序排序
            addDetailList.sort(Comparator.comparing(OutRentParameterDetailEntity::getOperationDate));
            entity.setRentParameterDetailList(addDetailList);
        }
        return BeanMapper.mapList(list, OutRentParameterVO.class);
    }
}
