package com.ejianc.business.profinance.person.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.profinance.person.bean.PersonLoanEntity;
import com.ejianc.business.profinance.person.service.IPersonLoanService;
import com.ejianc.business.profinance.person.vo.PersonLoanVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RList;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.Period;
import java.time.ZoneId;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 借款管理-个人借款  personLoan/fetchPersonLoanData
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("personLoan")
public class PersonLoanController implements Serializable {
	private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillTypeApi billTypeApi;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi iOrgApi;


    @Autowired
    private RedissonClient redissonClient;
    private static final String BILL_CODE = "PERS_L";//此处需要根据实际修改

    @Autowired
    private IPersonLoanService service;

    @Autowired
    private SessionManager sessionManager;

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PersonLoanVO> saveOrUpdate(@RequestBody PersonLoanVO saveOrUpdateVO) {
    	PersonLoanEntity entity = BeanMapper.map(saveOrUpdateVO, PersonLoanEntity.class);
    	if(entity.getId() == null){
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (!billCode.isSuccess()) {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
            entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            entity.setRefStatus("未引用");
        }
    	service.saveOrUpdate(entity, false);
    	PersonLoanVO vo = BeanMapper.map(entity, PersonLoanVO.class);
    	return CommonResponse.success("保存或修改单据成功！",vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PersonLoanVO> queryDetail(Long id) {
    	PersonLoanEntity entity = service.selectById(id);
    	PersonLoanVO vo = BeanMapper.map(entity, PersonLoanVO.class);
        return CommonResponse.success("查询详情数据成功！",vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<PersonLoanVO> vos) {
        if(ListUtil.isNotEmpty(vos)){
            for (PersonLoanVO vo : vos) {
                // 参数是单据类型编码字符串 根据需求是否打开下面代码
                /* CommonResponse<String> resp = billTypeApi.checkQuote("billTypeCode", vo.getId());
                if(!resp.isSuccess()){
                    return CommonResponse.error("删除失败！"+resp.getMsg());
                }*/
            }
        }
        service.removeByIds(vos.stream().map(PersonLoanVO::getId).collect(Collectors.toList()),true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
        JSONObject resp = new JSONObject();

        /** 模糊搜索配置字段示例 借款单号、借款人、借款人所在项目、项目所属组织、借款用途、经办人*/
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("borrowingProjectName");
        fuzzyFields.add("borrowingPersonName");
        fuzzyFields.add("parentOrgName");
        fuzzyFields.add("memo");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        Long orgId = InvocationInfoProxy.getOrgId();
        //若当前上下文为项目部，则根据项目部Id来进行查询
        if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
            param.getParams().put("orgId", new Parameter(QueryParam.EQ,orgId));
        } else {
            CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(orgId);
            if (!orgResp.isSuccess()) {
                logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                return CommonResponse.error("查询失败，获取组织信息失败！");
            }
            param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
                    orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        // 金额合计
        Map<String, BigDecimal> map = service.fetchTotalMny(param);
        param.getOrderMap().put("create_time",QueryParam.DESC);
        IPage<PersonLoanEntity> page = service.queryPage(param,false);
        List<PersonLoanVO> personLoanVOS = BeanMapper.mapList(page.getRecords(), PersonLoanVO.class);
        for (PersonLoanVO person:personLoanVOS){
            if (person.getExpectRepayDate()!=null){
                LocalDate localDate =  person.getExpectRepayDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
                Period period = LocalDate.now().until(localDate);
                person.setRepayDateDiff(period.getDays());
            }
        }
        resp.put("current", page.getCurrent());
        resp.put("size", page.getSize());
        resp.put("pages", page.getPages());
        resp.put("total", page.getTotal());
        resp.put("records", personLoanVOS);
        resp.put("totalLoanMny", map.get("totalLoanMny"));
        resp.put("totalRepaidLoanMny", map.get("totalRepaidLoanMny"));
        resp.put("totalLeftLoanMny", map.get("totalLeftLoanMny"));




        return CommonResponse.success("查询列表数据成功！", resp);
    }

    /**
     * 获取RPC数据
     * resp 返回值
     * isMustSuc 是否必须成功
     * errMsg 失败提示
     */
    private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
        if(isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }


    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        param.setPageIndex(1);
        param.setPageSize(-1);
        CommonResponse<JSONObject> jsonObjectCommonResponse = queryList(param);
        List<PersonLoanVO> list = new ArrayList<>();
        if (null != jsonObjectCommonResponse.getData()){
            list = (List<PersonLoanVO>) jsonObjectCommonResponse.getData().get("records");
            list.forEach(c->{
                c.setBillStateStr(BillStateEnum.getEnumByStateCode(c.getBillState()).getDescription());
                if (c.getExpectRepayDate()!=null){
                    LocalDate localDate =  c.getExpectRepayDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
                    Period period = LocalDate.now().until(localDate);
                    c.setRepayDateDiff(period.getDays());
                }
            });
        }
        //todo:字段翻译等等
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        if (param.getParams().containsKey("leftLoanMny")) {
            ExcelExport.getInstance().export("ToPayLoan-export.xlsx", beans, response);
        } else {
            ExcelExport.getInstance().export("PersonLoan-export.xlsx", beans, response);
        }
    }

    /**
     * @Description 参照
     * @Return void
     */
    @RequestMapping(value = "/refPersonLoanData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<PersonLoanVO>> refPersonLoanData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                                        String condition,
                                                                        String searchObject,
                                                                        String searchText) {
        QueryParam param = new QueryParam();
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        param.getParams().put("bill_state", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));
        param.getParams().put("left_loan_mny", new Parameter(QueryParam.GT, BigDecimal.ZERO));


        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        Long orgId = InvocationInfoProxy.getOrgId();
        //若当前上下文为项目部，则根据项目部Id来进行查询
        if(OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
            param.getParams().put("orgId", new Parameter(QueryParam.EQ, orgId));
        } else {
            CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(orgId);
            if(!orgResp.isSuccess()) {
                logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                return CommonResponse.error("查询失败，获取组织信息失败！");
            }
            param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
                    orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        if (StringUtils.isNotEmpty(condition)) {
            JSONObject _con = JSONObject.parseObject(condition);
            if (_con.containsKey("borrowingPersonId")) {
                param.getParams().put("borrowing_person_id", new Parameter(QueryParam.EQ, _con.get("borrowingPersonId")));
            }
            // 封装查询时排除的数据的结算池数据的ID
            String excludeIdStr = _con.getString("excludeIds");
            if (StringUtils.isNotBlank(excludeIdStr)) {
                param.getParams().put("id", new Parameter(QueryParam.NOT_IN, Lists.newArrayList(excludeIdStr.split(","))));
            }
        }
        IPage<PersonLoanEntity> page = service.queryPage(param,false);
        IPage<PersonLoanVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), PersonLoanVO.class));

        return CommonResponse.success("查询参照数据成功！",pageData);
     }


    @PostMapping(value = "/loanData")
    @ResponseBody
    public CommonResponse<Map<String, List<PersonLoanVO>>> loanData(@RequestBody JSONObject jsonObject) {
        Map<String, List<PersonLoanVO>> data = new HashMap<>();
        if (!jsonObject.containsKey("key")) {
            throw new BusinessException("请求参数不能为空！");
        }
        String key = jsonObject.getString("key");
        RList<PersonLoanVO> projectLoanList = redissonClient.getList(key);
        if (CollectionUtils.isEmpty(projectLoanList) && jsonObject.containsKey("ids")) {
            List<Long> ids = jsonObject.getJSONArray("ids").toJavaList(Long.class);
            Collection<PersonLoanEntity> pls = service.listByIds(ids);
            checkRefStatus(pls);
            projectLoanList.addAll(BeanMapper.mapList(pls, PersonLoanVO.class));
            projectLoanList.expire(Long.parseLong("5"), TimeUnit.MINUTES);
        } else {
            checkRefStatus(BeanMapper.mapList(projectLoanList, PersonLoanEntity.class));
        }
        data.put(key, BeanMapper.mapList(redissonClient.getList(key), PersonLoanVO.class));
        return CommonResponse.success("查询成功！", data);
    }

    private static void checkRefStatus(Collection<PersonLoanEntity> pls) {
        for (PersonLoanEntity pl : pls) {
            if ("引用".equals(pl.getRefStatus())) {
                throw new BusinessException("操作失败！借款单号[" + pl.getBillCode() + "]存在未生效的[个人还款单]，待生效后才允许继续还款");
            }
        }
    }
}
