package com.ejianc.business.profinance.person.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.profinance.person.bean.PersonLoanEntity;
import com.ejianc.business.profinance.person.bean.PersonRepayDetailEntity;
import com.ejianc.business.profinance.person.bean.PersonRepayEntity;
import com.ejianc.business.profinance.person.service.IPersonLoanService;
import com.ejianc.business.profinance.person.service.IPersonRepayService;
import com.ejianc.business.profinance.projectloan.bean.ProjectLoanEntity;
import com.ejianc.business.profinance.projectloan.bean.ProjectRepayDetailEntity;
import com.ejianc.business.profinance.projectloan.bean.ProjectRepayEntity;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service("personRepay")
public class PersonRepayBpmServiceImpl implements ICommonBusinessService {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final IOrgApi iOrgApi;
    private final SessionManager sessionManager;
    private final IPersonRepayService service;
    private final IBillTypeApi billTypeApi;
    private final IPersonLoanService personLoanService;

    public PersonRepayBpmServiceImpl(IOrgApi iOrgApi, SessionManager sessionManager, IPersonRepayService service, IBillTypeApi billTypeApi, IPersonLoanService personLoanService) {
        this.iOrgApi = iOrgApi;
        this.sessionManager = sessionManager;
        this.service = service;
        this.billTypeApi = billTypeApi;
        this.personLoanService = personLoanService;

    }

    /**
     * 提交前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        return CommonResponse.success("个人还款， 提交前回调成功");
    }

    ;

    /**
     * 提交完回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        PersonRepayEntity entity = service.selectById(billId);
        UserContext userContext = sessionManager.getUserContext();
        entity.setCommitDate(new Date());
        entity.setCommitUserCode(userContext.getUserCode());
        entity.setCommitUserName(userContext.getUserName());
        service.saveOrUpdate(entity, false);
        return CommonResponse.success("个人还款 审批回调处理成功");
    }

    /**
     * 有审批流的撤回前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return CommonResponse.success();
    }

    ;

    /**
     * 有审批流的撤回后回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }


    /**
     * 终审审核完回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        PersonRepayEntity entity = service.selectById(billId);
        if (entity != null) {
            entity.setEffectiveDate(new Date());
        }
        // 区分提交和审批
        if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
            //直审时添加单据提交人等信息
            UserContext userContext = sessionManager.getUserContext();
            entity.setCommitDate(new Date());
            entity.setCommitUserCode(userContext.getUserCode());
            entity.setCommitUserName(userContext.getUserName());
        }
        entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
        // 生效时间
        entity.setEffectiveDate(new Date());
        service.saveOrUpdate(entity, false);
        List<PersonRepayDetailEntity> detailList = entity.getDetailList();
        if (CollectionUtils.isNotEmpty(detailList)) {
            Map<Long, PersonRepayDetailEntity> collect = detailList.stream().collect(Collectors.toMap(PersonRepayDetailEntity::getSourceLoanId, Function.identity()));
            Set<Long> personLoanIds = collect.keySet();
            Collection<PersonLoanEntity> personLoanEntities = personLoanService.listByIds(personLoanIds);
            for (PersonLoanEntity person : personLoanEntities) {
                PersonRepayDetailEntity personRepayDetailEntity = collect.get(person.getId());
                person.setRepaidLoanMny(person.getRepaidLoanMny().add(personRepayDetailEntity.getCurRepayMny()).setScale(2, RoundingMode.HALF_UP));
                person.setLeftLoanMny(person.getLoanMny().subtract(person.getRepaidLoanMny()).setScale(2, RoundingMode.HALF_UP));
            }
            personLoanService.saveOrUpdateBatch(new ArrayList<>(personLoanEntities), 10, false);
            List<Long> sourceLoanIds = detailList.stream().map(PersonRepayDetailEntity::getSourceLoanId).collect(Collectors.toList());
            personLoanService.updateColumnValue(sourceLoanIds,"ref_status","未引用");
        }
        logger.info("终审审核完回调--end");
        return CommonResponse.success("个人还款 终审回调成功");
    }


    /**
     * 弃审后事件回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
        PersonRepayEntity entity = service.selectById(billId);
        //判断当前是否最新单据
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("repayPersonId",new Parameter(QueryParam.EQ,entity.getRepayPersonId()));
        LinkedHashMap<String, String> orderMap = new LinkedHashMap<>();
        orderMap.put("effective_date", QueryParam.DESC);
        queryParam.setOrderMap(orderMap);
        List<PersonRepayEntity> personRepayEntities = service.queryList(queryParam);
        if (!personRepayEntities.get(0).getId().equals(entity.getId())){
            return CommonResponse.error("当前单据不是最新单据，不能撤回或弃审！");

        }
        //合同是否被其他单据引用
        CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
        logger.info("平台返回查询被引用情况" + res.isSuccess() + "----" + res.getMsg());
        if (!res.isSuccess()) {
            return CommonResponse.error("当前单据已被下游业务引用，不能弃审！");
        }
        writeBackProjectLoan(entity);

        return CommonResponse.success("弃审/撤回前校验回调成功！");
    }

    /**
     * 回写对应借款单并重新计算借款单的借款余额
     *
     * @param entity 项目还款记录
     */
    private void writeBackProjectLoan(PersonRepayEntity entity) {
        List<PersonRepayDetailEntity> detailList = entity.getDetailList();
        if (CollectionUtils.isNotEmpty(detailList)) {
            Map<Long, PersonRepayDetailEntity> repayDetailEntityMap = detailList.stream().
                    collect(Collectors.toMap(PersonRepayDetailEntity::getSourceLoanId, Function.identity()));
            Set<Long> projectLoanIds = repayDetailEntityMap.keySet();
            Collection<PersonLoanEntity> personLoanEntities = personLoanService.listByIds(projectLoanIds);
            for (PersonLoanEntity person : personLoanEntities) {
                PersonRepayDetailEntity repayDetail = repayDetailEntityMap.get(person.getId());
                logger.info("修改还 子表实体——————:{}",JSONObject.toJSONString(repayDetail));
                logger.info("修改前实体——————:{}",JSONObject.toJSONString(person));
                //已还款金额 = 已还款金额- 还款金额
                person.setRepaidLoanMny(person.getRepaidLoanMny().subtract(repayDetail.getCurRepayMny()).setScale(2, RoundingMode.HALF_UP));
                logger.info("修改后实体——————:{}",JSONObject.toJSONString(person));
                person.setLeftLoanMny(person.getLoanMny().subtract(person.getRepaidLoanMny()).setScale(2, RoundingMode.HALF_UP));
            }

            personLoanService.saveOrUpdateBatch(new ArrayList<>(personLoanEntities), 10, false);
            List<Long> sourceLoanIds = detailList.stream().map(PersonRepayDetailEntity::getSourceLoanId).collect(Collectors.toList());
            personLoanService.updateColumnValue(sourceLoanIds,"ref_status","引用");
        }
    }
}
