package com.ejianc.business.pro.income.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.ejianc.business.contractbase.pool.settlepool.api.ISettlePoolApi;
import com.ejianc.business.contractbase.pool.settlepool.vo.SettlePoolVO;
import com.ejianc.business.contractpub.util.BeanConvertorUtil;
import com.ejianc.business.pro.income.bean.*;
import com.ejianc.business.pro.income.mapper.ProductionMapper;
import com.ejianc.business.pro.income.service.IContractRegisterService;
import com.ejianc.business.pro.income.service.IProductionDetailService;
import com.ejianc.business.pro.income.service.IProductionService;
import com.ejianc.business.pro.income.utils.*;
import com.ejianc.business.pro.income.vo.*;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 产值报量
 * 
 * @author generator
 * 
 */
@Service("productionService")
public class ProductionServiceImpl extends BaseServiceImpl<ProductionMapper, ProductionEntity> implements IProductionService{
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillCodeApi billCodeApi;

    private static final String BILL_CODE = "PRO_INCOME_PRODUCTION";//此处需要根据实际修改

    @Autowired
    private ValidateUtil validateUtil;

    @Autowired
    private DetailListUtil detailListUtil;

    @Autowired
    private IContractRegisterService registerService;
    
    @Autowired
    private IProductionDetailService detailService;

    @Autowired
    private ISettlePoolApi settlePoolApi;

    @Override
    public ProductionVO saveOrUpdate(ProductionVO saveorUpdateVO) {
        // 保存时校验合同version是否一致
        if(!validateUtil.validateUpStreamVersion(String.valueOf(saveorUpdateVO.getContractId()),
                BillTypeCodeEnum.施工合同.getCode(), saveorUpdateVO.getContractVersion())){
            throw new BusinessException("该合同已被更新，请刷新后重做！");
        }
        // 保存时校验项目预算书version是否一致
        List<ProductionDetailVO> detailVOList = saveorUpdateVO.getDetailList();
        if(CollectionUtils.isNotEmpty(detailVOList)){
            for(ProductionDetailVO detail : detailVOList){
                if("add".equals(detail.getRowState())){
                    if(!validateUtil.validateUpStreamVersion(String.valueOf(detail.getBudgetId()),
                            BillTypeCodeEnum.项目预算书.getCode(), detail.getBudgetVersion())){
                        throw new BusinessException("【" + detail.getBudgetCode() + "】项目预算书已被更新，请刷新后重做！");
                    }
                }
            };
        }
        //同一个合同只能存在一个自由态或审批中的单据
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, saveorUpdateVO.getContractId()));
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.NOT_IN, "1,3"));
        if(saveorUpdateVO.getId() != null){
            queryParam.getParams().put("id", new Parameter(QueryParam.NE, saveorUpdateVO.getId()));
        }
        List<ProductionEntity> list = super.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(list)) {
            throw new BusinessException("当前合同存在非审批通过态的产值报量，不允许保存!");
        }

        ProductionEntity entity = BeanMapper.map(saveorUpdateVO, ProductionEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), saveorUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        super.saveOrUpdate(entity);
        List<ProductionDetailEntity> detailList = entity.getDetailList();
        if(CollectionUtils.isNotEmpty(detailList)){
            for(ProductionDetailEntity detail : detailList){
                detail.setProductionId(entity.getId());
                if("del".equals(detail.getRowState())){
                    detailService.removeById(detail.getId(), false);
                } else {
                    detailService.saveOrUpdate(detail, false);
                    // 处理父子级关系
                    this.dealParentId(detail);
                    detailService.saveOrUpdate(detail, false);
                }
            };
        }
        return this.queryDetail(entity.getId());
    }

    /**
     * 处理父子级关系
     * @param detail
     */
    private void dealParentId(ProductionDetailEntity detail) {
        List<ProductionDetailCostEntity> costList = detail.getCostList();// 费用项清单
        if(CollectionUtils.isNotEmpty(costList)){
            Map<String, Long> idMap = new HashMap<>();
            costList.forEach(i->{
                idMap.put(i.getTid(), i.getId());
            });
            costList.forEach(i->{
                if(StringUtils.isNotEmpty(i.getTpid())){
                    i.setParentId(idMap.get(i.getTpid()));
                }
            });
        }
        List<ProductionDetailSubEntity> subList = detail.getSubList();// 分部分项清单
        if(CollectionUtils.isNotEmpty(subList)){
            Map<String, Long> idMap = new HashMap<>();
            subList.forEach(i->{
                idMap.put(i.getTid(), i.getId());
            });
            subList.forEach(i->{
                if(StringUtils.isNotEmpty(i.getTpid())){
                    i.setParentId(idMap.get(i.getTpid()));
                }
            });
        }
        List<ProductionDetailMeasureEntity> measureList = detail.getMeasureList();// 措施项清单
        if(CollectionUtils.isNotEmpty(measureList)){
            Map<String, Long> idMap = new HashMap<>();
            measureList.forEach(i->{
                idMap.put(i.getTid(), i.getId());
            });
            measureList.forEach(i->{
                if(StringUtils.isNotEmpty(i.getTpid())){
                    i.setParentId(idMap.get(i.getTpid()));
                }
            });
        }
        List<ProductionDetailOtherEntity> otherList = detail.getOtherList();// 其他项清单
        if(CollectionUtils.isNotEmpty(otherList)){
            Map<String, Long> idMap = new HashMap<>();
            otherList.forEach(i->{
                idMap.put(i.getTid(), i.getId());
            });
            otherList.forEach(i->{
                if(StringUtils.isNotEmpty(i.getTpid())){
                    i.setParentId(idMap.get(i.getTpid()));
                }
            });
        }
        List<ProductionDetailFeeEntity> feeList = detail.getFeeList();// 费税清单
        if(CollectionUtils.isNotEmpty(feeList)){
            Map<String, Long> idMap = new HashMap<>();
            feeList.forEach(i->{
                idMap.put(i.getTid(), i.getId());
            });
            feeList.forEach(i->{
                if(StringUtils.isNotEmpty(i.getTpid())){
                    i.setParentId(idMap.get(i.getTpid()));
                }
            });
        }
    }

    @Override
    public ProductionVO queryDetail(Long id) {
        ProductionEntity entity = super.selectById(id);
        detailListUtil.setDetailList(entity.getDetailList());// 解决queryList没有查询子表
        ProductionVO vo = BeanMapper.map(entity, ProductionVO.class);
        if (entity != null) {
            List<ProductionDetailVO> detailList = vo.getDetailList();
            // 构造子表树形
            if (CollectionUtils.isNotEmpty(detailList)) {
                for (ProductionDetailVO detail : detailList) {
                    List<ProductionDetailCostVO> costList = detail.getCostList();// 费用项清单
                    if(CollectionUtils.isNotEmpty(costList)){
                        costList.forEach(i->{
                            i.setTid(i.getId().toString());
                            i.setTpid(i.getParentId() != null && i.getParentId() > 0 ? i.getParentId().toString() : "");
                            i.setShadowId(i.getTid());
                            i.setRowState("edit");
                        });
                        new TreeSortUtil<ProductionDetailCostVO>().entryListToSort(costList);
                        detail.setCostList(TreeNodeBUtil.buildTree(costList));
                    }
                    List<ProductionDetailSubVO> subList = detail.getSubList();// 分部分项清单
                    if(CollectionUtils.isNotEmpty(subList)){
                        subList.forEach(i->{
                            i.setTid(i.getId().toString());
                            i.setTpid(i.getParentId() != null && i.getParentId() > 0 ? i.getParentId().toString() : "");
                            i.setShadowId(i.getTid());
                            i.setRowState("edit");
                        });
                        new TreeSortUtil<ProductionDetailSubVO>().entryListToSort(subList);
                        detail.setSubList(TreeNodeBUtil.buildTree(subList));
                    }
                    List<ProductionDetailMeasureVO> measureList = detail.getMeasureList();// 措施项清单
                    if(CollectionUtils.isNotEmpty(measureList)){
                        measureList.forEach(i->{
                            i.setTid(i.getId().toString());
                            i.setTpid(i.getParentId() != null && i.getParentId() > 0 ? i.getParentId().toString() : "");
                            i.setShadowId(i.getTid());
                            i.setRowState("edit");
                        });
                        new TreeSortUtil<ProductionDetailMeasureVO>().entryListToSort(measureList);
                        detail.setMeasureList(TreeNodeBUtil.buildTree(measureList));
                    }
                    List<ProductionDetailOtherVO> otherList = detail.getOtherList();// 其他项清单
                    if(CollectionUtils.isNotEmpty(otherList)){
                        otherList.forEach(i->{
                            i.setTid(i.getId().toString());
                            i.setTpid(i.getParentId() != null && i.getParentId() > 0 ? i.getParentId().toString() : "");
                            i.setShadowId(i.getTid());
                            i.setRowState("edit");
                        });
                        new TreeSortUtil<ProductionDetailOtherVO>().entryListToSort(otherList);
                        detail.setOtherList(TreeNodeBUtil.buildTree(otherList));
                    }
                    List<ProductionDetailFeeVO> feeList = detail.getFeeList();// 费税清单
                    if(CollectionUtils.isNotEmpty(feeList)){
                        feeList.forEach(i->{
                            i.setTid(i.getId().toString());
                            i.setTpid(i.getParentId() != null && i.getParentId() > 0 ? i.getParentId().toString() : "");
                            i.setShadowId(i.getTid());
                            i.setRowState("edit");
                        });
                        new TreeSortUtil<ProductionDetailFeeVO>().entryListToSort(feeList);
                        detail.setFeeList(TreeNodeBUtil.buildTree(feeList));
                    }
                }
            }
        }
        return vo;
    }

    @Override
    public String validateContract(Long id) {
        //同一个合同只能存在一个自由态或审批中的单据
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, id));
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.NOT_IN, "1,3"));
        List<ProductionEntity> list = super.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(list)) {
            throw new BusinessException("当前合同存在非审批通过态的产值报量，不允许新增!");
        }
        return "校验通过！";
    }

    @Override
    public ProductionHistoryVO queryProductionHistory(Long id) {
        ContractRegisterEntity entity = registerService.selectById(id);
        ProductionHistoryVO vo = new ProductionHistoryVO();
        vo.setContractId(id);
        vo.setChangeStatus(entity.getChangeStatus());
        vo.setIsFinish(entity.getIsFinish());
        vo.setIsRelieve(entity.getIsRelieve());
        vo.setIsSuspend(entity.getIsSuspend());

        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, id));
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        queryParam.getOrderMap().put("productionDate", QueryParam.DESC);
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);
        List<ProductionEntity> list = super.queryList(queryParam);
        
        if(CollectionUtils.isEmpty(list)){
            return vo;
        }
        BigDecimal sumProductionMny = list.stream().map(ProductionEntity::getProductionMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal sumProductionTaxMny = list.stream().map(ProductionEntity::getProductionTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        vo.setSumProductionMny(sumProductionMny);// 产值总金额(无税)
        vo.setSumProductionTaxMny(sumProductionTaxMny);// 产值总金额
        vo.setProductionRecord(BeanMapper.mapList(list, ProductionVO.class));
        return vo;
    }

    @Override
    public boolean pushSettleToPool(ProductionVO vo) {
        SettlePoolVO spv = new SettlePoolVO();
        boolean flag = false;
        try {
            logger.info("结算单对象 -> 结算池对象自动转换开始-----"+ JSONObject.toJSONString(vo));
            // 对象自动转换
            BeanConvertorUtil.convert(vo, spv);
            logger.info("结算单对象 -> 结算池对象自动转换结束，下面开始手动转换");

            // 个别字段需要手动封装
            convertSettleVOToSettlePoolVO(vo, spv);
            logger.info("推送参数----"+JSONObject.toJSONString(spv));
            CommonResponse<SettlePoolVO> res = settlePoolApi.saveOrUpdateSettle(spv);
            if(res.isSuccess()){
                flag = true;
                logger.info("结算单推送结算池成功---{}", res.getMsg());
            }else {
                logger.error("结算单推送结算池失败！结算单id-{}，{}",vo.getId(), res.getMsg());
            }
        } catch (Exception e) {
            logger.error("结算单推送结算池失败！结算单id-{}", vo.getId(), e);
            throw new BusinessException("结算单推送结算池异常!");
        }
        return flag;
    }

    @Override
    public boolean delSettleFromPool(Long id) {
        SettlePoolVO spv = new SettlePoolVO();
        boolean flag = false;
        spv.setSourceId(id);
        try {
            logger.info("结算单弃审推送结算池开始,结算单id-{}",id);
            CommonResponse<SettlePoolVO> res = settlePoolApi.deleteSettle(spv);
            if(res.isSuccess()){
                flag = true;
                logger.info("结算单弃审推送结算池成功---{}", res.getMsg());
            }else {
                logger.error("结算单推送结算池失败！结算单id-{}，{}",id, res.getMsg());
            }
        }catch (Exception e){
            logger.error("结算单弃审推送结算池失败！结算单id-{}", id, e);
            throw new BusinessException("结算单弃审推送结算池异常!");
        }
        return flag;
    }

    // 将SettleReportVO转换成settlePoolVO
    private void convertSettleVOToSettlePoolVO(ProductionVO vo, SettlePoolVO spv) {
        spv.setSourceType("product_value");//结算类型
        spv.setSourceId(vo.getId());//结算单id
        spv.setId(vo.getId());
        spv.setSettleProperty(1);//属性类别=收入
        spv.setUltimateFlag(0);//是否最终结算=否
        spv.setCreateUserCode(vo.getCreateUserCode());//结算创建者账号
        spv.setCreateTime(vo.getCreateTime());//结算创建时间
        spv.setUpdateUserCode(vo.getUpdateUserCode());//结算修改者账号
        spv.setUpdateTime(vo.getUpdateTime());//结算修改时间
        spv.setBillCodeUrl("/ejc-proincome-frontend/#/production/card?id=" + vo.getId());// 穿透地址
        if(vo.getContractId()!=null){//有合同需要推送的字段
            //查询合同
            ContractRegisterEntity contractEntity = registerService.selectById(vo.getContractId());
            spv.setContractType("contraction");
            spv.setContractFlag(1);//合同标识
            spv.setSupplementFlag(contractEntity.getSupplementFlag());//是否补充写协议
            spv.setMaiContractId(contractEntity.getMainContractId());//原合同id
            spv.setMaiContractName(contractEntity.getMainContractName());//原合同name
            spv.setMaiContractCode(contractEntity.getMainContractCode());//原合同code
            spv.setContractCode(contractEntity.getBillCode());//合同编码
            spv.setPartyaId(contractEntity.getCustomerId());//甲方id
            spv.setPartyaName(contractEntity.getCustomerName());//甲方name
            spv.setPartybId(contractEntity.getSupplierId());//乙方id
            spv.setPartybName(contractEntity.getSupplierName());//乙方name
            spv.setSignDate(contractEntity.getSignDate());//签订日期
            BigDecimal lastTaxMny = BigDecimal.ZERO; // 截止本期累计结算金额
            BigDecimal lastMny = BigDecimal.ZERO; // 截止本期累计结算金额（无税）
            BigDecimal lastTax = BigDecimal.ZERO; // 截止本期累计结算税额
            /*获取除本次累计结算金额*/
            LambdaQueryWrapper<ProductionEntity> wrapper = new LambdaQueryWrapper<>();
            wrapper.orderByDesc(ProductionEntity::getCreateTime);
            wrapper.in(ProductionEntity::getBillState, 1,3);
            wrapper.eq(ProductionEntity::getDr, 0);
            wrapper.eq(ProductionEntity::getContractId, vo.getContractId());
            wrapper.ne(ProductionEntity::getId,vo.getId());
            List<ProductionEntity> list = super.list(wrapper);
            if(CollectionUtils.isNotEmpty(list)){
                for (ProductionEntity settlementEntity : list) {
                    BigDecimal settTaxMny = settlementEntity.getProductionTaxMny()==null?BigDecimal.ZERO:settlementEntity.getProductionTaxMny();
                    BigDecimal settMny = settlementEntity.getProductionMny()==null?BigDecimal.ZERO:settlementEntity.getProductionMny();
                    BigDecimal settTax = settlementEntity.getTaxMny()==null?BigDecimal.ZERO:settlementEntity.getTaxMny();
                    lastTaxMny = lastTaxMny.add(settTaxMny);
                    lastMny = lastMny.add(settMny);
                    lastTax = lastTax.add(settTax);
                }
            }
            spv.setLastTaxMny(lastTaxMny);//截止本期已结算金额（含税，不含本期）
            spv.setLastMny(lastMny);//截止本期已结算金额(无税，不含本期)
            spv.setLastTax(lastTax);//截止本期税额（不含本期）
        }
        logger.info("结算单对象 -> 结算池对象手动转换完成");
    }
}
