package com.ejianc.business.pro.income.utils;

import com.ejianc.framework.core.kit.base.annotation.NotNull;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.regex.Pattern;

public class DateUtil {

    public final static String CURRENT_YEAR = getCurrentYear();

    /**
     * 两个日期相差的天数，正数+1，负数-1
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static Integer getSubDay(Date startDate, Date endDate) {
        if(startDate == null || endDate == null){
            return 0;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        Calendar cal = Calendar.getInstance();
        long betweenDays = 0L;
        try {
            cal.setTime(sdf.parse(sdf.format(startDate)));
            long time1 = cal.getTimeInMillis();
            cal.setTime(sdf.parse(sdf.format(endDate)));
            long time2 = cal.getTimeInMillis();
            betweenDays = (time1 - time2) / (1000 * 3600 * 24);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        Integer days = Integer.parseInt(String.valueOf(betweenDays));
        return days < 0 ? days - 1 : days + 1;
    }

    /***
     * @Author mrsir_wxp
     * @Date 2021/2/1 比较两个日期大小，
     * @Description compareDate
     * @Param [dt1, dt2]
     * @Return 1：dt1>dt2,-1：dt1<dt2, 0相等
     */
    public static int compareDate(Date dt1, Date dt2) {
        try {
            if (dt1.getTime() > dt2.getTime()) {
                return 1;
            } else if (dt1.getTime() < dt2.getTime()) {
                return -1;
            } else {
                return 0;
            }
        } catch (Exception exception) {
            exception.printStackTrace();
        }
        return 0;
    }

    /***
     * @Author mrsir_wxp
     * @Date 2021/2/1 比较两个日期大小，
     * @Description compareDate
     * @Param [dt1, dt2]
     * @Return 1：dt1>dt2,-1：dt1<dt2, 0相等
     */
    public static int compareDay(Date dt1, Date dt2) {
        dt1 = parseDate(formatDate(dt1));
        dt2 = parseDate(formatDate(dt2));
        return compareDate(dt1, dt2);
    }

    /**
     * 比较两个日期并取最大值或最小值
     * @param date1
     * @param date2
     * @param flag false-最小值，true-最大值
     * @return
     */
    public static Date maxDate(Date date1, Date date2, Boolean flag) {
        if(date1 == null) {
            return date2;
        }
        if(date2 == null){
            return date1;
        }
        if(flag){
            if(compareDate(date1, date2) < 0){
                return date2;
            }
            return date1;
        } else {
            if(compareDate(date1, date2) < 0){
                return date1;
            }
            return date2;
        }
    }

    /**
     * 比较日期大小,如2020-10-23格式
     *
     * @param t1
     * @param t2
     * @return
     */
    public static int compareDateStr(String t1, String t2) {
        int trr1 = Integer.parseInt(t1.replaceAll("-", ""));
        int trr2 = Integer.parseInt(t2.replaceAll("-", ""));
        if (trr1 > trr2) {
            return 1;
        } else if (trr1 < trr2) {
            return -1;
        }
        return 0;
    }

    /**
     * 获取日期月份字符串，如01，02...
     * @description 这里使用了String.format（）方法，作用就是格式化输出参数. “%02d”是指定输出格式，%作先导标记，0表示自动补0, 2的意思是最小长度为2（如果用4，则1输出0001），d表示整数
     * @param date
     * @return
     */
    public static String getMonthStr(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        int month = cal.get(Calendar.MONTH) + 1;
        return String.format("%02d",month);
    }

    /**
     * 获取日期年份字符串，如2021，2022...
     * @description 这里使用了String.format（）方法，作用就是格式化输出参数. “%02d”是指定输出格式，%作先导标记，0表示自动补0, 2的意思是最小长度为2（如果用4，则1输出0001），d表示整数
     * @param date
     * @return
     */
    public static String getYearStr(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        String year = String.valueOf(cal.get(Calendar.YEAR));// 当前年
        return year;
    }

    /**
     * 获取日期所在周
     * @param date
     * @return
     */
    public static Integer getWeek(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        Integer week = cal.get(Calendar.WEEK_OF_MONTH);
        // 周日应该为上一周
        if(cal.get(Calendar.DAY_OF_WEEK) == 1){
            week -= 1;
        }
        return week;
    }

    /**
     * 获取当前年
     * @return
     */
    public static String getCurrentYear() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        String year = String.valueOf(cal.get(Calendar.YEAR));// 当前年
        return year;
    }

    /**
     * 获取两个日期间所有月份
     * @return
     */
    public static List<String> getMonthBetweenDate(Date startDate, Date endDate) {
        Calendar cal = Calendar.getInstance();
        List<String> list = new ArrayList<>();
        for(int i = 0; startDate.getTime() <= endDate.getTime(); i++){
            list.add(getMonth(startDate));
            cal.setTime(startDate);
            cal.add(Calendar.MONTH, 1);
            startDate = cal.getTime();
        }
        return list;
    }

    /**
     * 获取月份
     * @return
     */
    public static String getMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        return sdf.format(cal.getTime());
    }

    /**
     * 获取当前月
     * @return
     */
    public static String getCurrentMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        return sdf.format(cal.getTime());
    }

    /**
     * 获取上一月
     * @return
     */
    public static String getLastMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(Calendar.MONTH, -1);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        return sdf.format(cal.getTime());
    }

    /**
     * 获取下一月
     * @return
     */
    public static String getNextMonth(String month) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        Calendar cal = Calendar.getInstance();
        try {
            cal.setTime(sdf.parse(month));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        cal.add(Calendar.MONTH, 1);
        return sdf.format(cal.getTime());
    }

    /**
     * 将中文环境下的1-周一，...，7-周日，改为西方习惯的2-周一，...，7-周六，1-周日
     * @param day
     * @return
     */
    public static Integer turnZnDayOfWeek(Integer day) {
        day = day + 1;
        return day > 7 ? day - 6 : day;
    }

    /**
     * 将中文环境下的1-28及最后一天，改为对应日期
     * @param day
     * @return
     */
    public static Integer turnZnDayOfMonth(Integer day) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        if(day > cal.getActualMaximum(Calendar.DAY_OF_MONTH)){// 月末加1天，2月29日或者31、32特殊处理
            cal.add(Calendar.DATE, 1);
            day = cal.get(Calendar.DAY_OF_MONTH );
            return day;
        }
        return day == 30 ? cal.getActualMaximum(Calendar.DAY_OF_MONTH) : day;
    }

    /**
     * 将任意日期格式字符串，重新翻译为标准日期格式
     * @param str
     * @return
     * @throws ParseException
     */
    public static String formatDateString(String str) throws ParseException {
        if(StringUtils.isEmpty(str)){
            return str;
        }
        Date date = new Date();
        try {
            Double par = Double.parseDouble(str);
            date = DoubleToDate(par);
        } catch (NumberFormatException e) {
            SimpleDateFormat bsdf = new SimpleDateFormat(DateUtil.getDateFormat(str));
            date = bsdf.parse(str);
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        return sdf.format(date);
    }

    public static Date DoubleToDate(Double dVal) {
        Date tDate = new Date();
        long localOffset = tDate.getTimezoneOffset() * 60000; //系统时区偏移 1900/1/1 到 1970/1/1 的 25569 天
        tDate.setTime((long) ((dVal - 25569) * 24 * 3600 * 1000 + localOffset));
        return tDate;
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static Date parseDate(String date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            return sdf.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return new Date();
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static Date parseMonth(String date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        try {
            return sdf.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return new Date();
    }

    /**
     * 常规自动日期格式识别
     * @param str 时间字符串
     * @return Date
     * @author dc
     */
    public static String getDateFormat(String str) {
        boolean year = false;
        Pattern pattern = Pattern.compile("^[-\\+]?[\\d]*$");
        if(pattern.matcher(str.substring(0, 4)).matches()) {
            year = true;
        }
        StringBuilder sb = new StringBuilder();
        int index = 0;
        if(!year) {
            if(str.contains("月") || str.contains("-") || str.contains("/")) {
                if(Character.isDigit(str.charAt(0))) {
                    index = 1;
                }
            }else {
                index = 3;
            }
        }
        for (int i = 0; i < str.length(); i++) {
            char chr = str.charAt(i);
            if(Character.isDigit(chr)) {
                if(index==0) {
                    sb.append("y");
                }
                if(index==1) {
                    sb.append("M");
                }
                if(index==2) {
                    sb.append("d");
                }
                if(index==3) {
                    sb.append("H");
                }
                if(index==4) {
                    sb.append("m");
                }
                if(index==5) {
                    sb.append("s");
                }
                if(index==6) {
                    sb.append("S");
                }
            }else {
                if(i>0) {
                    char lastChar = str.charAt(i-1);
                    if(Character.isDigit(lastChar)) {
                        index++;
                    }
                }
                sb.append(chr);
            }
        }
        return sb.toString();
    }

    public static Date addDays(@NotNull final Date date, final int amount) {
        return DateUtils.addDays(date, amount);
    }

    public static Date addMonths(@NotNull final Date date, int amount) {
        return DateUtils.addMonths(date, amount);
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static String formatDate(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        return sdf.format(date);
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static String formatSeconds(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return sdf.format(date);
    }

    /**
     * 2016-11-10 07:33:23, 则返回2016-11-10 00:00:00
     */
    public static Date beginOfDate(@NotNull final Date date) {
        return DateUtils.truncate(date, Calendar.DATE);
    }

    /**
     * 2017-1-23 07:33:23, 则返回2017-1-23 23:59:59.999
     */
    public static Date endOfDate(@NotNull final Date date) {
        return new Date(nextDate(date).getTime() - 1);
    }

    /**
     * 2016-11-10 07:33:23, 则返回2016-11-11 00:00:00
     */
    public static Date nextDate(@NotNull final Date date) {
        return DateUtils.ceiling(date, Calendar.DATE);
    }

    /**
     * 2016-11-10, 则返回2016-11-01
     */
    public static Date beginOfMonth(@NotNull final Date date) {
        return DateUtils.truncate(date, Calendar.MONTH);
    }

    /**
     * 2016-11-10, 则返回2016-01-01
     */
    public static Date beginOfYear(@NotNull final Date date) {
        return DateUtils.truncate(date, Calendar.YEAR);
    }

    /**
     * 比较是否重叠
     * @param T1 区间一开始时间
     * @param T2 区间一结束时间
     * @param T3 区间二开始时间
     * @param T4 区间二结束时间
     * @return
     */
    public static Boolean overlap(Date T1, Date T2, Date T3, Date T4){
        // T1-T2为第一个时间段     T3-T4为第二个时间段
        // T1 <= T3 <= T2
        if(T1.getTime() <= T3.getTime() && T3.getTime() <= T2.getTime()){
            return true;
        }
        // T1 <= T4 <= T2
        if(T1.getTime() <= T4.getTime() && T4.getTime() <= T2.getTime()){
            return true;
        }
        // T3 <= T1 <= T2 <= T4
        if(T3.getTime() <= T1.getTime() && T2.getTime() <= T4.getTime()){
            return true;
        }
        // T1 <= T3 <= T4 <= T2
        if(T1.getTime() <= T3.getTime() && T4.getTime() <= T2.getTime()){
            return true;
        }
        return false;
    }

    public static void main(String[] args) {
        String date1 = "2021-9.10";
        String date2 = "2021-12";
        System.out.println(getDateFormat(date1));
        try {
            System.out.println(formatDateString(date1));
        } catch (ParseException e) {
            e.printStackTrace();
        }
//        System.out.println(getSubDay(parseDate(date2), parseDate(date1)));
    }

}
