package com.ejianc.business.market.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.costplan.api.ICostPlanApi;
import com.ejianc.business.market.bean.CloseCostEntity;
import com.ejianc.business.enums.CapitalStatusEnum;
import com.ejianc.business.market.service.ICloseCostService;
import com.ejianc.business.warn.vo.WarningDTO;
import com.ejianc.foundation.share.api.IProjectPoolApi;
import com.ejianc.foundation.share.api.IProjectSetApi;
import com.ejianc.foundation.share.vo.ProjectPoolSetVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

@Service("closeCost")
public class CloseCostBpmServiceImpl implements ICommonBusinessService {
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IProjectSetApi projectSetApi;
	private final IWarnCenterApi warnCenterApi;
	private final ICloseCostService service;
	@Value(value = "${warning.code}")
	private String WARN_CODE;
	@Autowired
	private ICostPlanApi costPlanApi;

	@Autowired
	private IProjectPoolApi projectPoolApi;

	public CloseCostBpmServiceImpl(SessionManager sessionManager, IBillTypeApi billTypeApi, IProjectSetApi projectSetApi, IWarnCenterApi warnCenterApi, ICloseCostService service) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.projectSetApi = projectSetApi;
		this.warnCenterApi = warnCenterApi;
		this.service = service;
	}

	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		CloseCostEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 区分提交和审批
		UserContext userContext = sessionManager.getUserContext();
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 更新项目池资金管控状态为成本关门
		CommonResponse<String> res = projectSetApi.changeProjectStatus(entity.getProjectId(), null, null, null, null, CapitalStatusEnum.成本关门.getCode());
		if (!res.isSuccess()) {
			logger.info("成本关门【{}】更新【项目池】资金管控状态为【{}】失败，失败原因：{}", entity.getBillCode(), CapitalStatusEnum.成本关门, res.getMsg());
			throw new BusinessException("成本关门【" + entity.getBillCode() + "】更新【项目池】资金管控状态为【" + CapitalStatusEnum.成本关门 + "】失败，失败原因：" + res.getMsg());
		}

		// 发信息给相关角色（单据触发预警）
		sendMsg(entity);


		//修改成本关门计划的完成状态
		if (entity != null){
			costPlanApi.updateFinishStateCostClose(entity.getProjectId(),entity.getId(),1);
		}

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		return CommonResponse.success("单据撤回/弃审成功");
	}


	private void sendMsg(CloseCostEntity entity) {
		CommonResponse<WarningDTO> response = warnCenterApi.getWarnSet(WARN_CODE, entity.getOrgId());
		logger.info("查询预警中心设置：{}", JSONObject.toJSONString(response, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
		if (!response.isSuccess() || null == response.getData()) {
			logger.info("未查询到预警中心设置，不发预警！");
		} else {
			WarningDTO warningDTO = response.getData();
			EarlyWarnTransVO vo = new EarlyWarnTransVO();
			vo.setSourceId(String.valueOf(entity.getProjectId()));
			vo.setTenantId(entity.getTenantId());
			vo.setOrgId(entity.getOrgId());
			vo.setOrgName(entity.getOrgName());
			vo.setSendOrgId(String.valueOf(entity.getOrgId()));
			vo.setEarlywarnName(warningDTO.getName());
			vo.setBillName(entity.getBillName());
			vo.setEarlywarnContent(warningDTO.getWarningContentTemplate().replaceAll("#projectName#", entity.getProjectName()));
			vo.setWarnSetId(warningDTO.getId());
			vo.setWarnSetParamId(CollectionUtils.isNotEmpty(warningDTO.getWarningSetings()) ? warningDTO.getWarningSetings().get(0).getId() : null);
			vo.setPcUrl("/ejc-promarket-frontend/#/closeCost/card");
			vo.setPcTitle("成本关门");
			vo.setWarnLevel(CollectionUtils.isNotEmpty(warningDTO.getWarningSetings()) ? warningDTO.getWarningSetings().get(0).getWarningLevel() : "中");
			vo.setWarnType(String.valueOf(warningDTO.getWarnType()));
			List<EarlyWarnTransVO> warnTransVO = new ArrayList<>();
			warnTransVO.add(vo);
			CommonResponse<String> res = warnCenterApi.sendToWarnCenterByCode(warnTransVO, WARN_CODE);
			if (!res.isSuccess()) {
				logger.info("预警内容发送至预警中心失败，具体参数：{},预警参数编码：{}，失败原因：{}", JSONObject.toJSONString(warnTransVO), WARN_CODE, res.getMsg());
			}
			logger.info("预警中心响应：{}", JSONObject.toJSONString(res, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
		}
	}


	/**
	 * 弃审后事件回调
	 *
	 * @param billId
	 * @param state
	 * @return
	 */
	@Override
	public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		CloseCostEntity entity = service.selectById(billId);

		//撤回成本分析计划的完成状态
		costPlanApi.updateFinishStateCostClose(entity.getProjectId(),entity.getId(),0);
		return CommonResponse.success();
	}

	/**
	 * 审批--获取当前节点的审批人---从服务中选择
	 *
	 * @param billId 单据id
	 * @param billTypeCode 单据类型编码
	 * @param sign 节点标志
	 * @return
	 */
	@Override
	public CommonResponse<JSONArray> queryApproveUser(Long billId, String billTypeCode, String sign, Map<String, Object> other){
		JSONArray arr = new JSONArray();
		logger.info("sign:{}",sign);

		//根据标识判断是否（成本相关人员审批的）节点
		if ("choseCostUser".equals(sign)){
			//获取表单详情数据

			//根据项目id获取项目基本信息中的成本对接人信息
			CloseCostEntity entity = service.selectById(billId);
			Long projectId = entity.getProjectId();

			JSONObject jsonObject = new JSONObject();
			CommonResponse<JSONArray> commonResponse = projectPoolApi.queryProjectPoolById(projectId);


			if(!commonResponse.isSuccess()){
				throw new BusinessException("查询项目失败!");
			}
			JSONArray array = commonResponse.getData();
			List<ProjectPoolSetVO> vos = JSONObject.parseArray(array.toJSONString(), ProjectPoolSetVO.class);

			logger.info("vos:{}",JSONObject.toJSONString(vos));

			ProjectPoolSetVO vo = vos.get(0);
			Long costAdapter = vo.getCostAdapter();
			String costAdapterName = vo.getCostAdapterName();

			logger.info("userId:{}",costAdapter);
			logger.info("userName:{}",costAdapterName);


			jsonObject.put("userId",costAdapter);
			jsonObject.put("userName",costAdapterName);

			arr.add(jsonObject);

		}
		return CommonResponse.success(arr);
	}
}
