package com.ejianc.business.market.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.market.bean.AdjustApplicationEntity;
import com.ejianc.business.enums.ArchiveStatusEnum;
import com.ejianc.business.enums.BusinessStatusEnum;
import com.ejianc.business.enums.CapitalStatusEnum;
import com.ejianc.business.enums.SettleStatusEnum;
import com.ejianc.business.market.service.IAdjustApplicationService;
import com.ejianc.business.market.vo.AdjustApplicationVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 工程状态调整申请
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("adjustApplication")
public class AdjustApplicationController implements Serializable {
	private static final long serialVersionUID = -6785045006497587831L;
	private static final String RULE_CODE = "ADJUST-APPLICATION";
	private static final String BILL_CODE = "EJCBT202208000028";
	private static final String BUSINESS_STATUS = "业务状态";
	private static final String SETTLE_STATUS = "结算状态";
	private static final String ARCHIVE_STATUS = "资料归档状态";
	private static final String CAPITAL_STATUS = "资金管控状态";
	private static final String TEXT = "text";
	private static final String VALUE = "value";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IAdjustApplicationService service;

	public AdjustApplicationController(SessionManager sessionManager, IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, IAdjustApplicationService service) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
	}

	/**
	 * 保存修改
	 *
	 * @param saveOrUpdateVO vo
	 *
	 * @return CommonResponse<AdjustApplicationVO>
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<AdjustApplicationVO> saveOrUpdate(@RequestBody AdjustApplicationVO saveOrUpdateVO) {
		AdjustApplicationEntity entity = BeanMapper.map(saveOrUpdateVO, AdjustApplicationEntity.class);
		if (null == entity.getId() || 0 == entity.getId()) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}
		entity.setBillName("工程状态调整申请");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());
		service.saveOrUpdate(entity, false);
		return CommonResponse.success("保存或修改单据成功！", BeanMapper.map(service.selectById(entity.getId()), AdjustApplicationVO.class));
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return CommonResponse<AdjustApplicationVO>
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<AdjustApplicationVO> queryDetail(@RequestParam(value = "id", required = true) Long id) {
		AdjustApplicationEntity entity = service.selectById(id);
		AdjustApplicationVO vo = BeanMapper.map(entity, AdjustApplicationVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 删除
	 *
	 * @param vos 待删除的vos
	 *
	 * @return CommonResponse<String>
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<AdjustApplicationVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (AdjustApplicationVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(AdjustApplicationVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 请求参数
	 *
	 * @return CommonResponse<JSONObject>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("parentOrgName");
		fuzzyFields.add("projectName");
		fuzzyFields.add("memo");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
				return CommonResponse.error("查询失败，获取组织信息失败！");
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}
		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<AdjustApplicationEntity> page = service.queryPage(param, false);
		List<AdjustApplicationVO> adjust = BeanMapper.mapList(page.getRecords(), AdjustApplicationVO.class);
		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", adjust);
		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 响应头
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		param.setPageIndex(1);
		param.setPageSize(-1);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("parentOrgName");
		fuzzyFields.add("projectName");
		fuzzyFields.add("memo");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}
		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<AdjustApplicationEntity> page = service.queryPage(param, false);
		List<AdjustApplicationVO> adjusts = new ArrayList<>();
		page.getRecords().forEach(e -> {
			AdjustApplicationVO adjust = BeanMapper.map(e, AdjustApplicationVO.class);
			adjust.setBillStateName(BillStateEnum.getEnumByStateCode(adjust.getBillState()).getDescription());
			adjust.setStateAfterAdjustment(BusinessStatusEnum.getNameByCode(adjust.getStateAfterAdjustment()).toString());
			adjusts.add(adjust);
		});
		Map<String, Object> beans = new HashMap<>();
		beans.put("records", adjusts);
		ExcelExport.getInstance().export("AdjustApplication-export.xlsx", beans, response);
	}

	/**
	 * 校验该项目是否存在未生效的【工程状态调整申请】单据
	 *
	 * @param projectId 项目id
	 *
	 * @return CommonResponse<String>
	 */
	@GetMapping(value = "/checkOnly")
	public CommonResponse<String> checkOnly(@RequestParam(value = "id", required = false) Long id,
	                                        @RequestParam(value = "projectId", required = true) Long projectId) {
		LambdaQueryWrapper<AdjustApplicationEntity> lambdaQuery = Wrappers.lambdaQuery();
		if (null != id) {
			lambdaQuery.ne(AdjustApplicationEntity::getId, id);
		}
		lambdaQuery.eq(AdjustApplicationEntity::getProjectId, projectId);
		lambdaQuery.notIn(AdjustApplicationEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		int count = service.count(lambdaQuery);
		if (count > 0) {
			throw new BusinessException("编辑失败！该项目存在未生效的申请单，不能继续申请调整。");
		}
		return CommonResponse.success("校验成功！");
	}

	/**
	 * 根据调整状态类型查询调整后状态
	 *
	 * @param adjustStateType 调整状态类型
	 *
	 * @return CommonResponse<JSONObject>
	 */
	@GetMapping(value = "/fetchStateAfterAdjustment")
	public CommonResponse<JSONObject> fetchStateAfterAdjustment(@RequestParam(value = "adjustStateType", required = true) String adjustStateType) {
		JSONObject jsonObject = new JSONObject();
		List<JSONObject> data = new ArrayList<>();
		// 业务状态
		if (BUSINESS_STATUS.equals(adjustStateType)) {
			for (BusinessStatusEnum e : BusinessStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}
		}
		// 结算状态
		if (SETTLE_STATUS.equals(adjustStateType)) {
			for (SettleStatusEnum e : SettleStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}
		}
		// 资料归档状态
		if (ARCHIVE_STATUS.equals(adjustStateType)) {
			for (ArchiveStatusEnum e : ArchiveStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}
		}
		// 资金管控状态
		if (CAPITAL_STATUS.equals(adjustStateType)) {
			for (CapitalStatusEnum e : CapitalStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}
		}
		jsonObject.put("data", data);
		return CommonResponse.success("请求成功！", jsonObject);
	}


	/**
	 * 查询所有的调整后状态  0413 工程状态调整 需求改造
	 *
	 * @param
	 *
	 * @return CommonResponse<JSONObject>
	 */
	@GetMapping(value = "/fetchStateAfterAdjustmentAll")
	public CommonResponse<JSONObject> fetchStateAfterAdjustmentAll() {
		JSONObject jsonObject = new JSONObject();
		List<JSONObject> data = new ArrayList<>();
		// 业务状态
			for (BusinessStatusEnum e : BusinessStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}

		// 结算状态
			for (SettleStatusEnum e : SettleStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}

		// 资料归档状态
			for (ArchiveStatusEnum e : ArchiveStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}

		// 资金管控状态
			for (CapitalStatusEnum e : CapitalStatusEnum.values()) {
				JSONObject o = new JSONObject();
				o.put(TEXT, e.toString());
				o.put(VALUE, e.toString());
				data.add(o);
			}

		jsonObject.put("data", data);


		return CommonResponse.success("请求成功！", jsonObject);
	}
}
