package com.ejianc.business.promaterial.plan.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.promaterial.plan.bean.BatPlanDetailEntity;
import com.ejianc.business.promaterial.plan.service.IBatPlanDetailService;
import com.ejianc.business.promaterial.plan.vo.BatPlanDetailVO;
import com.ejianc.business.promaterial.plan.vo.BatPlanEnum;
import com.ejianc.business.promaterial.plan.vo.BatPlanVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IProjectPoolApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IDefdocApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.promaterial.plan.mapper.BatPlanMapper;
import com.ejianc.business.promaterial.plan.bean.BatPlanEntity;
import com.ejianc.business.promaterial.plan.service.IBatPlanService;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 材料批次采购计划主表
 *
 * @author generator
 */
@Service("batPlanService")
public class BatPlanServiceImpl extends BaseServiceImpl<BatPlanMapper, BatPlanEntity> implements IBatPlanService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /** 参照查询：经办人字段名 */
    private static final String CONDITION_TRUSTEES_ID = "employeeId";
    /** 参照查询：项目ID */
    private static final String CONDITION_PROJECT_ID = "projectId";
    /** 参照查询：组织ID */
    private static final String CONDITION_ORG_ID = "orgId";

    /** BILL_CODE */
    private static final String BILL_CODE = "PARCEL_PLAN_CODE";//此处需要根据实际修改

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private IDefdocApi defdocApi;

    @Autowired
    private IProjectPoolApi projectPoolApi;

    /** 子表services */
    @Autowired
    private IBatPlanDetailService batPlanDetailService;

    /**
     * 保存
     *
     * @param batPlanVO 保存数据
     * @return 保存结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRED)
    public BatPlanVO saveOrUpdateByVo(BatPlanVO batPlanVO) {
        // parentOrgCode如果是空的，则需要查询赋值
        if (StringUtils.isEmpty(batPlanVO.getParentOrgCode()) && batPlanVO.getParentOrgId() != null) {
            CommonResponse<OrgVO> orgResponse = iOrgApi.getOneById(batPlanVO.getParentOrgId());
            if (orgResponse.isSuccess()) {
                OrgVO orgVO = orgResponse.getData();
                batPlanVO.setParentOrgCode(orgVO.getCode());
            }
        }
        BatPlanEntity entity = BeanMapper.map(batPlanVO, BatPlanEntity.class);
        if (entity.getId() == null || entity.getId() == 0) {
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), batPlanVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }
            else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        // 采购状态默认待采购
        if (entity.getPlanState() == null) {
            entity.setPlanState(BatPlanEnum.PLAN_STATE_WAIT.getPlanState());
        }
        super.saveOrUpdate(entity, false);
        return this.copyVoByEntity(entity);
    }

    /**
     * 根据id查询详情
     *
     * @param id 单据id
     * @return 详情
     */
    @Override
    public BatPlanVO queryBatPlanDetailById(Long id) {
        BatPlanEntity entity = super.selectById(id);
        return this.copyVoByEntity(entity);
    }

    /**
     * 查询采购计划列表
     *
     * @param param 查询条件
     * @return 查询结果
     */
    @Override
    public IPage<BatPlanVO> queryBatPlanPage(QueryParam param) {
        this.setDefaultParam(param);
        return this.queryBatPlanVO(param);
    }

    /**
     * 导出
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public List<BatPlanVO> excelExportBatPlan(QueryParam param) {
        this.setDefaultParam(param);
        param.setPageIndex(1);
        // 默认10000条
        param.setPageSize(10000);
        List<BatPlanEntity> entityList = super.queryList(param);
        if (CollectionUtils.isNotEmpty(entityList)) {
            List<BatPlanVO> list = BeanMapper.mapList(entityList,BatPlanVO.class);
            for (BatPlanVO vo: list) {
                // 单据状态翻译
//                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
                vo.setBillStateName(this.getBillStateName(vo.getBillState()));
                // 采购状态翻译
                vo.setPlanStateName(BatPlanEnum.getDescriptionByStateCode(vo.getPlanState()));
            }
            return list;
        }
        return new ArrayList<>();
    }

    /**
     * 参照查询
     *
     * @param param        查询参数
     * @param condition    参数
     * @param searchObject 模糊查询匹配字段
     * @return 查询结果
     */
    @Override
    public IPage<BatPlanVO> refBatPlanData(QueryParam param, String condition, String searchObject) {
        this.setDefaultParam(param);
//        param.getFuzzyFields().add("billCode");
        // 只查询待采购
        param.getParams().put("planState", new Parameter(QueryParam.EQ, BatPlanEnum.PLAN_STATE_WAIT.getPlanState()));
        // 单据状态
        param.getParams().put("billState", new Parameter(QueryParam.IN,
                Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),
                        BillStateEnum.PASSED_STATE.getBillStateCode())));
        // searchObject做模糊查询处理
        if (StringUtils.isNotEmpty(searchObject)) {
            // 系统自带的searchObject实际上是做EQ处理的，在此置为null，手动做like处理
            param.setSearchObject(null);
            JSONObject searchJson = JSONObject.parseObject(searchObject);
            logger.info("参照查询searchObject：【{}】", searchJson);
            if (searchJson != null && !searchJson.isEmpty()) {
                for (Map.Entry<String, Object> entry : searchJson.entrySet()) {
                    param.getParams().put(entry.getKey(), new Parameter(QueryParam.LIKE, entry.getValue()));
                }
            }
        }
        if (StringUtils.isNotEmpty(condition)) {
            // 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
            logger.info("参照查询condition：【{}】", _con);
//            for (Map.Entry<String, Object> entry : _con.entrySet()) {
//                // 经办人字段名称不一致，特殊处理
//                if (CONDITION_TRUSTEES_ID.equals(entry.getKey())) {
//                    param.getParams().put("trusteesId", new Parameter(QueryParam.EQ, _con.get
//                    (CONDITION_TRUSTEES_ID)));
//                }
//                else {
//                    param.getParams().put(entry.getKey(), new Parameter(QueryParam.EQ, entry.getValue()));
//                }
//            }
            // 经办人
            if (_con.containsKey(CONDITION_TRUSTEES_ID)) {
                param.getParams().put("trusteesId", new Parameter(QueryParam.EQ, _con.get(CONDITION_TRUSTEES_ID)));
            }
            // 项目id
            if (_con.containsKey(CONDITION_PROJECT_ID)) {
                Long projectId = Long.parseLong(_con.get(CONDITION_PROJECT_ID).toString());
                CommonResponse<List<Long>> projectResponse = projectPoolApi.queryProjectIdsByParentProjectId(projectId);
                if (!projectResponse.isSuccess()) {
                    throw new BusinessException("查询项目信息失败！");
                }
                param.getParams().put("projectId", new Parameter(QueryParam.IN, projectResponse.getData()));
            }
            // 组织id
            if (_con.containsKey(CONDITION_ORG_ID)) {
                Long orgId = Long.valueOf(_con.get(CONDITION_ORG_ID).toString());
                CommonResponse<OrgVO> orgResp = iOrgApi.getOneById(orgId);
                OrgVO orgVO = orgResp.getData();
                /** 数据隔离，如果查询组织为项目部，查询orgId，否则查询parentOrgId本下 */
                if (OrgVO.ORG_TYPE_DEPARTMENT.equals(orgVO.getOrgType())) {
                    param.getParams().put("orgId", new Parameter(QueryParam.EQ, orgId));
                }
                else {
                    param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, iOrgApi
                            .findChildrenByParentIdWithoutProjectDept(orgId).getData().stream().map(OrgVO::getId)
                            .collect(Collectors.toList())));
                }
//                param.getParams().put("parentOrgId", new Parameter(QueryParam.EQ, ));
            }

        }
        IPage<BatPlanVO> pageData = this.queryBatPlanVO(param);
        // 查询采购清单
        if (CollectionUtils.isNotEmpty(pageData.getRecords())) {
            List<BatPlanVO> batPlanVOList = pageData.getRecords();
            List<String> idList = batPlanVOList.stream().map(BatPlanVO::getId).map(String::valueOf)
                    .collect(Collectors.toList());
            // 查询子表
            QueryParam detailParam = new QueryParam();
            detailParam.getParams().put("planId", new Parameter(QueryParam.IN, idList));
            List<BatPlanDetailEntity> detailEntityList = batPlanDetailService.queryList(detailParam);
            List<BatPlanDetailVO> detailVOList = BeanMapper.mapList(detailEntityList, BatPlanDetailVO.class);
            Map<Long, List<BatPlanDetailVO>> detailMap = new HashMap<>();
            List<BatPlanDetailVO> mapList;
            // 按照planID对子表结果分组
            for (BatPlanDetailVO vo : detailVOList) {
                if (detailMap.containsKey(vo.getPlanId())) {
                    detailMap.get(vo.getPlanId()).add(vo);
                }
                else {
                    mapList = new ArrayList<>();
                    mapList.add(vo);
                    detailMap.put(vo.getPlanId(), mapList);
                }
            }
            // 将采购计划清单添加到采购计划中
            batPlanVOList.forEach(batPlanVO -> {
                batPlanVO.setBatPlanDetailList(detailMap.get(batPlanVO.getId()));
            });
        }
        return pageData;
    }

    /**
     * 修改采购计划状态
     *
     * @param idList    id列表
     * @param planState 引用类型
     */
    @Override
    public void updateBatPlanByQuoteType(List<String> idList, Integer planState) {
        if (StringUtils.isEmpty(BatPlanEnum.getDescriptionByStateCode(planState))) {
            throw new BusinessException("计划状态错误");
        }
        // 批量更改
        if (CollectionUtils.isNotEmpty(idList)) {
            List<BatPlanEntity> entityList = (List<BatPlanEntity>) super.listByIds(idList);
            if (CollectionUtils.isNotEmpty(entityList)) {
                for (BatPlanEntity entity : entityList) {
                    // 采购完成不允许做其他操作
                    if (BatPlanEnum.PLAN_STATE_OVER.getPlanState().equals(entity.getPlanState())) {
                        logger.info("采购计划编码【{}】计划名称【{}】已采购完成！", entity.getBillCode(), entity.getPlanName());
                        throw new BusinessException("采购计划编码【" + entity.getBillCode() + "】计划名称【" +
                                entity.getPlanName() + "】已采购完成！");
                    }

                    entity.setPlanState(planState);
                }
//                entityList.forEach(entity -> {
//                    entity.setPlanState(planState);
//                });
                super.saveOrUpdateBatch(entityList);
            }
        }
    }

    /**
     * 查询采购类别
     *
     * @return 采购类别自定义档案
     */
    @Override
    public List<DefdocDetailVO> refPurchaseDate() {
        String wzDefCode = "pro-supply-material-supply-content";
        String zzcDefCode = "pro-supply-revolving-materials-supply-content";
        List<DefdocDetailVO> list = new ArrayList<>();
        CommonResponse<List<DefdocDetailVO>> wzRes = defdocApi.getDefDocByDefCode(wzDefCode);
        if (wzRes.isSuccess()) {
            list.addAll(wzRes.getData());
        }
        CommonResponse<List<DefdocDetailVO>> zzcRes = defdocApi.getDefDocByDefCode(zzcDefCode);
        if (zzcRes.isSuccess()) {
            list.addAll(zzcRes.getData());
        }
        return list;
    }

    /**
     * 查询分页列表并转换成VO
     *
     * @param param 查询参数
     * @return 查询结果
     */
    private IPage<BatPlanVO> queryBatPlanVO(QueryParam param) {
        IPage<BatPlanEntity> page = super.queryPage(param, false);
        IPage<BatPlanVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<BatPlanVO> batPlanVOList = BeanMapper.mapList(page.getRecords(), BatPlanVO.class);
        // 采购阶段转义
        for (BatPlanVO vo : batPlanVOList) {
            vo.setPlanStateName(BatPlanEnum.getDescriptionByStateCode(vo.getPlanState()));
        }
        pageData.setRecords(batPlanVOList);
        return pageData;
    }

    /**
     * 预制模糊查询字段
     *
     * @param param 查询参数
     */
    private void setDefaultParam(QueryParam param) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("planName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("unitName");
        fuzzyFields.add("trusteesName");
        fuzzyFields.add("purchaseName");
        // 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 数据隔离 本下 没有组织orgId的删除下面代码 */

        CommonResponse<List<OrgVO>> orgResponse = iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if (!orgResponse.isSuccess()) {
            throw new BusinessException("查询失败，获取组织信息失败！");
        }
        List<OrgVO> orgVOList = orgResponse.getData();
        List<Long> commonOrgIds = new ArrayList<>();
        //项目部 id
        List<Long> departmentIds = new ArrayList<>();
        orgVOList.forEach(org -> {
            if (5 == org.getOrgType()) {
                //项目部
                departmentIds.add(org.getId());
            }
            else {
                //普通组织
                commonOrgIds.add(org.getId());
            }
        });
        // 查询本下
        if (CollectionUtils.isNotEmpty(commonOrgIds)) {
            // 要求主表有orgId字段，保存单据所属组织
            param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, commonOrgIds));
        }
        // 查询项目部
        else if (CollectionUtils.isNotEmpty(departmentIds)) {
            // 要求主表有parentOrgId字段，保存单据所属项目部
            param.getParams().put("orgId", new Parameter(QueryParam.IN, departmentIds));
        }
    }

    /**
     * 根据实体copyVO
     *
     * @param entity 实体
     * @return vo
     */
    private BatPlanVO copyVoByEntity(BatPlanEntity entity) {
        BatPlanVO vo = BeanMapper.map(entity, BatPlanVO.class);
        vo.setPlanStateName(BatPlanEnum.getDescriptionByStateCode(vo.getPlanState()));
        return vo;
    }

    /**
     * 根据单据状态获取单据名称
     * @param billState 单据状态
     * @return 单据名称
     */
    private String getBillStateName(Integer billState){
        String billStateName = null;
        if (BillStateEnum.UNCOMMITED_STATE.getBillStateCode().equals(billState)) {
            billStateName = "自由态";
//            billStateName = BillStateEnum.getEnumByStateCode(billState).getDescription();
        }
        else if(BillStateEnum.COMMITED_STATE.getBillStateCode().equals(billState)){
            billStateName = "已提交";
//            billStateName = BillStateEnum.getEnumByStateCode(billState).getDescription();
        }
        else if(BillStateEnum.APPROVING_HAS_STATE.getBillStateCode().equals(billState)){
            billStateName = "审核中";
        }
        else if(BillStateEnum.PASSED_STATE.getBillStateCode().equals(billState)){
            billStateName = "审批通过";
        }
        else if(BillStateEnum.UNAPPROVED.getBillStateCode().equals(billState)){
            billStateName = "审批驳回";
//            billStateName = BillStateEnum.getEnumByStateCode(billState).getDescription();
        }
        else if(BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode().equals(billState)){
            billStateName = "审核中";
        }
        return billStateName;
    }
}
