package com.ejianc.business.proother.service.impl;

import com.ejianc.business.proother.bean.ContractEntity;
import com.ejianc.business.proother.enums.BillPushStatusEnum;
import com.ejianc.business.proother.enums.DraftTypeEnum;
import com.ejianc.business.proother.enums.PerformanceStatusEnum;
import com.ejianc.business.proother.enums.SignatureStatusEnum;
import com.ejianc.business.proother.service.IContractService;

import com.ejianc.business.proother.vo.ContractVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service("contract") 
public class ContractBpmServiceImpl implements ICommonBusinessService {

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	@Autowired
	private IContractService contractService;

	@Autowired
	private IBillTypeApi billTypeApi;

	@Autowired
	private SessionManager sessionManager;


	/**
	 * 有审批流提交完回调
	 *
	 * @param
	 * @return
	 */
	@Override
	public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode){
		ContractEntity e = contractService.selectById(billId);

		if(!(DraftTypeEnum.线下签订.getCode().toString()).equals(e.getDraftType()) && (null == e.getContractFileId())) {
			//确保线上起草方式时，用户已编辑合同文件
			return CommonResponse.error("请编辑合同文件并保存后再执行此操作！");
		}

		UserContext userContext = sessionManager.getUserContext();
		e.setCommitDate(new Date());
		e.setCommitUserCode(userContext.getUserCode());
		e.setCommitUserName(userContext.getUserName());
		contractService.saveOrUpdate(e,false);

		return CommonResponse.success("合同审批回调处理成功！");
	}

	/**
	 * 终审审核完回调
	 * 
	 * @param
	 * @return
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		/*
			起草方式为线下签订则不需要签章
			当不使用电子签章时：创建时默认为未签章，单据审批通过后自动更改为【已签章】
			当签章状态变为【已签章】时，合同状态变为【履约中】
		 */
		ContractEntity contractEntity = contractService.selectById(billId);
		if (contractEntity != null){

			if((DraftTypeEnum.线下签订.getCode().toString()).equals(contractEntity.getDraftType())) {
				//线下签订合同无需进行签章操作，则审批通过后直接置为已生效状态
				logger.info("合同id-{}为线下签订直接修改为签章状态并设置为履约中", billId);
				contractEntity.setSignatureStatus(SignatureStatusEnum.已签章.getCode());
				contractEntity.setPerformanceStatus(PerformanceStatusEnum.履约中.getCode());
				//合同已生效，添加生效时间
				contractEntity.setEffectiveDate(new Date());
			} else if(null == contractEntity.getContractFileId()) {
				//确保线上起草方式时，用户已编辑合同文件
				return CommonResponse.error("请编辑合同文件并保存后再执行此操作！");
			}
		}

		if(state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			//直审时添加单据提交人等信息
			contractEntity.setCommitDate(new Date());
			contractEntity.setCommitUserCode(sessionManager.getUserContext().getUserCode());
			contractEntity.setCommitUserName(sessionManager.getUserContext().getUserName());
		}

		//合同推送合同池
		boolean pushResult = contractService.pushContract(BeanMapper.map(contractEntity, ContractVO.class));
		if(pushResult) {
			contractEntity.setBillPushFlag(BillPushStatusEnum.推送成功.getStatus());
		} else {
			contractEntity.setBillPushFlag(BillPushStatusEnum.未成功推送.getStatus());
		}

		contractService.saveOrUpdate(contractEntity);
		return CommonResponse.success("合同审批回调成功！");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId
	 * @param state
	 * @return
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		/*
			弃审逻辑：
			1、审批过程中，支持后台弃审；
			2、当审批通过，且没有走签章流程且没有下游业务时，支持后台弃审；
		 */
		ContractEntity contractEntitiy = contractService.selectById(billId);

		if (!(SignatureStatusEnum.未签章.getCode().equals(contractEntitiy.getSignatureStatus()))){
			return CommonResponse.error("当前单据已有签章流程，不能撤回！");
		}

		//合同是否被其他单据引用
		CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode,billId);
		logger.info("平台返回查询被引用情况"+res.isSuccess()+"----"+res.getMsg());

		if(!res.isSuccess()) {
			return CommonResponse.error("当前单据已被下游业务引用，不能弃审！");
		}

		//从合同池删除数据
		boolean delResult = contractService.delContractFromPool(contractEntitiy.getId());
		if(delResult) {
			contractEntitiy.setBillPushFlag(BillPushStatusEnum.未成功推送.getStatus());
			contractService.saveOrUpdate(contractEntitiy, false);
		}

		return CommonResponse.success("弃审/撤回前校验回调成功！");
	}


}
