package com.ejianc.business.prosub.service.impl;

import cn.hutool.http.HttpRequest;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.ejianc.business.prosub.bean.ChangeEntity;
import com.ejianc.business.prosub.bean.ContractEntity;
import com.ejianc.business.prosub.service.IChangeService;
import com.ejianc.business.prosub.service.IContractAsyncService;
import com.ejianc.business.prosub.service.IContractService;
import com.ejianc.business.signaturemanage.vo.WatermarkVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Recover;
import org.springframework.retry.annotation.Retryable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.HashMap;
import java.util.concurrent.TimeUnit;

/**
 * @author baipengyan
 * @date 2022/5/20
 * @description 合同加水印异步http请求
 */
@Service("ContractAsyncService")
public class ContractAsyncServiceImpl implements IContractAsyncService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${common.env.base-host}")
    private String BaseHost;
    @Autowired
    private IContractService contractService;

    @Autowired
    private IChangeService changeService;

    @Autowired
    private IAttachmentApi attachmentApi;

    /**
     * 获取水印文件附件信息
     *
     * @param watermarkVO 水印配置参数
     * @param contractType 合同类型（主合同：contract 变更合同：changeContract）
     * @return 水印文件信息
     */
    @Override
    @Async(value = "commonTask")
    @Retryable(maxAttempts = 3, backoff = @Backoff(delay = 2000, multiplier = 1.5), stateful = true)
    public void fetchWatermarkAttachment(HashMap<String, String> headers, WatermarkVO watermarkVO, String contractType) {
        logger.info("异步获取水印文件附件信息开始，入参：{}", JSON.toJSONString(watermarkVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        Stopwatch start = Stopwatch.createStarted();
        String body = HttpRequest.post(BaseHost + "ejc-signaturemanage-web/common/fetchWatermarkAttachment")
                .addHeaders(headers)
                .body(JSON.toJSONString(watermarkVO))
                .timeout(60000)
                .execute().body();
        AttachmentVO attachmentVO = JSON.parseObject(body, AttachmentVO.class);
        logger.info("异步获取水印文件附件信息结束，请求签章中心耗时「{}」秒，文件中心的水印文件信息：{}", start.stop().elapsed(TimeUnit.SECONDS), JSON.toJSONString(attachmentVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        start.reset().start();
        // 更新合同水印文件id
        if ("contract".equals(contractType)){
            UpdateWrapper<ContractEntity> updateWrapper = new UpdateWrapper<>();
            updateWrapper.eq("id", watermarkVO.getBillId());
            updateWrapper.set("watermark_contract_file_id", attachmentVO.getId());
            boolean flag = contractService.update(updateWrapper);
            if (flag) {
                logger.info("获取水印文件附件信息并更新合同成功，写入数据库耗时「{}」秒，合同id：{}，原合同文件id：{}，水印合同文件id：{}", start.stop().elapsed(TimeUnit.SECONDS), watermarkVO.getBillId(), watermarkVO.getFileId(), attachmentVO.getId());
            }
        }else if ("changeContract".equals(contractType)){
            UpdateWrapper<ChangeEntity> updateWrapper = new UpdateWrapper<>();
            updateWrapper.eq("id", watermarkVO.getBillId());
            updateWrapper.set("change_watermark_contract_file_id", attachmentVO.getId());
            boolean flag = changeService.update(updateWrapper);
            if (flag) {
                logger.info("获取水印文件附件信息并更新变更合同成功，写入数据库耗时「{}」秒，合同id：{}，原变更合同文件id：{}，水印合同文件id：{}", start.stop().elapsed(TimeUnit.SECONDS), watermarkVO.getBillId(), watermarkVO.getFileId(), attachmentVO.getId());
            }
        }
    }

    @Recover
    public void recordWatermark(Exception e, HashMap<String, String> headers, WatermarkVO watermarkVO) {
        logger.error("获取水印文件附件信息并更新合同失败，请求参数：{}", JSON.toJSONString(watermarkVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
    }


    /**
     * 删除文件中心水印文件
     *
     * @param watermarkFileId 水印合同附件id
     * @return boolean 是否删除标识
     */
    @Override
    public boolean delWatermarkContractFile(Long watermarkFileId) {
        Assert.notNull(watermarkFileId, "水印合同附件不能为空！");
        CommonResponse<String> res = attachmentApi.delete(String.valueOf(watermarkFileId));
        if (!res.isSuccess()) {
            logger.error("删除文件中心水印文件失败，原因：{}，水印文件id：{}", res.getMsg(), watermarkFileId);
            throw new BusinessException("删除文件中心水印文件失败，原因：" + res.getMsg() + "，水印文件id：" + watermarkFileId);
        }
        return true;
    }
}
