package com.ejianc.business.process.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.ejianc.business.process.bean.DeductionEntity;
import com.ejianc.business.process.enums.BillPushStatusEnum;
import com.ejianc.business.process.enums.SupplierSignStatusEnum;
import com.ejianc.business.process.mapper.DeductionMapper;
import com.ejianc.business.process.service.IDeductionService;
import com.ejianc.business.process.vo.MeasureVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.foundation.share.api.IProSupplierApi;
import com.ejianc.foundation.share.utils.FileUtil;
import com.ejianc.foundation.share.vo.CooperateVO;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import javax.servlet.http.HttpServletRequest;
import java.io.InputStream;
import java.util.*;

/**
 * 扣奖罚单
 *
 * @author generator
 */
@Service("deductionService")
public class DeductionServiceImpl extends BaseServiceImpl<DeductionMapper, DeductionEntity> implements IDeductionService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private JedisPool jedisPool;

    private final String OPERATE = "DEDUCTION_BILL_SYNC";

    private final String PUSH_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/deduction/billSync";
    private final String DEL_SUP_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/deduction/billDel";

    @Autowired
    private ISystemDataPushService systemDataPushService;

    @Autowired
    private IAttachmentApi attachmentApi;

    @Autowired
    private IPushMessageApi pushMessageApi;

    @Autowired
    private IProSupplierApi proSupplierApi;

    /**
     * 扣奖罚单单据类型编码
     */
    private final String billType = "BT211230000000006";

    @Override
    public boolean pushBillToSupCenter(DeductionEntity deductionEntity, String billTypeCode, CooperateVO cooperateVO) {
        boolean locked = false, syncFlag = false;
        Jedis jedis = jedisPool.getResource();
        String key = billTypeCode + "::" + deductionEntity.getId().toString();

        //设置单据当前系统信息
        CommonResponse<String> ejcCloudSystemCode = proSupplierApi.getEjcCloudSystemCode();
        if(!ejcCloudSystemCode.isSuccess()) {
            this.executeUpdate(deductionEntity.getId(),false);
            logger.error("推送扣奖罚单据-{}失败，获取当前系统编码失败,{}", deductionEntity.getId(), ejcCloudSystemCode.getMsg());
            return false;
        }
        //设置当前系统ID
        deductionEntity.setSourceSystemId(ejcCloudSystemCode.getData());
        // 清空租户ID
        deductionEntity.setTenantId(null);

        try {
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);

            if(!locked) {
                this.executeUpdate(deductionEntity.getId(),false);
                logger.error("单据推送失败，单据锁获取失败！");
                releaseLock(jedis, false, key, OPERATE);
                return false;
            }

            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("transData", JSONObject.toJSONString(deductionEntity));

            //查询单据附件信息并下载
            CommonResponse<List<AttachmentVO>> fileResp = attachmentApi.queryListBySourceId(deductionEntity.getId(), null, null, null);
            if(fileResp.isSuccess()) {
                Map<String, Map<String, InputStream>> files = new HashMap<>();
                List<AttachmentVO> fileList = fileResp.getData();

                //Map<fileName, fileSourceType>
                Map<String, String> fileSourceTypeMap = new HashMap<>();
                List<Long> fileIds = new ArrayList<>();

                //从附件信息列表获取到： 1、附件名对应附件业务类型Map,2、获取到附件Id列表
                for(AttachmentVO attach : fileList) {
                    fileSourceTypeMap.put(attach.getFileName(), attach.getSourceType());
                    fileIds.add(attach.getId());
                }
                paramMap.put("nameSourceTypeMapping", JSONObject.toJSONString(fileSourceTypeMap));

                //当前单据携带有附件信息
                if(CollectionUtils.isNotEmpty(fileList)) {
                    Map<String, InputStream> fileMap = FileUtil.getInstance().batchDownFileFlow(fileIds, true);
                    fileMap.keySet().stream().forEach(fileKey -> {
                        Map<String, InputStream> file = new HashMap<>(1);
                        file.put(fileKey, fileMap.get(fileKey));
                        files.put(fileKey, file);
                    });
                }
                logger.info("向供应商-{}推送扣奖罚单据参数-{}", deductionEntity.getSupplierId(), JSONObject.toJSONString(paramMap));

                //推送单据到指定的供方
                CommonResponse<String> syncReqResp = systemDataPushService.exchangeDataAndFilesWithEachLinkSystem(PUSH_BILL_SERVER_URL,
                        paramMap,
                        deductionEntity.getSupplierId().toString(),
                        files);

                if(syncReqResp.isSuccess()) {
                    CommonResponse<String> supResp = JSONObject.parseObject(syncReqResp.getData(), CommonResponse.class);
                    if(supResp.isSuccess()) {
                        this.executeUpdate(deductionEntity.getId(),true);
                        syncFlag = true;
                    } else {
                        this.executeUpdate(deductionEntity.getId(),false);
                        logger.error("供方id-{}处理推送扣奖罚单据id-{}失败, {}", deductionEntity.getId(), deductionEntity.getSupplierId(), supResp.getMsg());
                    }
                } else {
                    this.executeUpdate(deductionEntity.getId(),false);
                    logger.error("发送请求推送扣奖罚单据id-{}给供方id-{}失败, {}", deductionEntity.getId(), deductionEntity.getSupplierId(), syncReqResp.getMsg());
                }
            } else {
                this.executeUpdate(deductionEntity.getId(),false);
                logger.error("获取扣奖罚单据id-{}对应附件信息失败, {}", deductionEntity.getId(), fileResp.getMsg());
            }

        } catch (Exception e) {
            this.executeUpdate(deductionEntity.getId(),false);
            logger.error("推送扣奖罚单据id-{}给供方id-{} 异常，", deductionEntity.getId(), deductionEntity.getSupplierId(), e);
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

        return syncFlag;
    }


    /**
     * 更新单据推送状态
     * @param id 单据id
     * @param flag 是否成功推送
     */
    private void executeUpdate(Long id,Boolean flag) {
        // 更新协同推送状态
        UpdateWrapper<DeductionEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", id);
        // 修改推送状态
        if (Boolean.TRUE.equals(flag)) {
            updateWrapper.set("bill_push_flag", BillPushStatusEnum.推送成功.getStatus());
        } else {
            updateWrapper.set("bill_push_flag", BillPushStatusEnum.未成功推送.getStatus());
        }
        super.update(updateWrapper);
    }


    @Override
    public boolean delPushBill(DeductionEntity deductionEntity, String billTypeCode) {
        boolean locked = false, delSuc = false;
        Jedis jedis = jedisPool.getResource();
        String key = billTypeCode + "::" + deductionEntity.getId().toString();

        //设置单据当前系统信息
        CommonResponse<String> ejcCloudSystemCode = proSupplierApi.getEjcCloudSystemCode();
        if (!ejcCloudSystemCode.isSuccess()) {
            logger.error("推送扣奖罚单据-{}失败，获取当前系统编码失败,{}", deductionEntity.getId(), ejcCloudSystemCode.getMsg());
        }
        //设置当前系统ID
        deductionEntity.setSourceSystemId(ejcCloudSystemCode.getData());

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);

            if(!locked) {
                logger.error("单据作废失败，单据锁获取失败！");
                releaseLock(jedis, false, key, OPERATE);
                return false;
            }

            Map<String, String> paramMap = new HashMap<>();
            paramMap.put("sourceId", deductionEntity.getId().toString());
            paramMap.put("sourceSystemId", deductionEntity.getSourceSystemId());

            logger.info("扣奖罚单据id-{}弃审，通知供方-{}单据作废!", deductionEntity.getSupplierId(), deductionEntity.getId());

            //推送单据到指定的供方
            CommonResponse<String> syncReqResp = systemDataPushService.exchangeDataWithEachLinkSystem(DEL_SUP_BILL_SERVER_URL,
                    RequestMethod.POST,
                    JSONObject.toJSONString(paramMap),
                    deductionEntity.getSupplierId().toString());

            if(syncReqResp.isSuccess()) {
                CommonResponse<String> supResp = JSONObject.parseObject(syncReqResp.getData(), CommonResponse.class);
                if(supResp.isSuccess()) {
                    delSuc = true;
                } else {
                    logger.error("供方-{}处理扣奖罚单据id-{}作废失败, {}", deductionEntity.getSupplierId(), deductionEntity.getId(), supResp.getMsg());
                }
            } else {
                logger.error("发送请求通知供方-{} 扣奖罚单据id-{}作废失败, {}", deductionEntity.getSupplierId(), deductionEntity.getId(), syncReqResp.getMsg());
            }
        } catch (Exception e) {
            logger.error("通知供方扣奖罚单据id-{}作废异常，", deductionEntity.getId(), e);
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

        return delSuc;
    }

    @Override
    public String updateBillSupSignSyncInfo(HttpServletRequest request) {
        String authority = request.getHeader("authority");
        String msg = null;

        Jedis jedis = null;
        boolean locked = false;

        String billId = request.getParameter("billId");
        String supOperatorName = request.getParameter("supOperatorName");
        String supOperatorPhone = request.getParameter("supOperatorPhone");
        String supOperatorUserCode = request.getParameter("supOperatorUserCode");
        Date supOperateTime = new Date(Long.valueOf(request.getParameter("supOperateTime")));
        String nameSourceTypeMapping = request.getParameter("nameSourceTypeMapping");
        Map<String, String> mp = JSONObject.parseObject(nameSourceTypeMapping, Map.class);

        DeductionEntity deductionEntity = super.selectById(billId);
        //设置供方签字信息
        deductionEntity.setSupOperateTime(supOperateTime);
        deductionEntity.setSupOperatorName(supOperatorName);
        deductionEntity.setSupOperatorPhone(supOperatorPhone);
        deductionEntity.setSupOperatorUserCode(supOperatorUserCode);
        deductionEntity.setSupplierSignStatus(SupplierSignStatusEnum.乙方已签字.getCode());

        String key = billType + "::" + deductionEntity.getId().toString();

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);

            if(!locked) {
                logger.error("单扣奖罚据id-{}签字信息回写加锁失败！", deductionEntity.getId());
                releaseLock(jedis, false, key, OPERATE);
                return "扣奖罚单据签字信息回写加锁失败";
            }

            //保存单据中附件并获取到上传后附件的Id
            Map<String, List<Long>> attachIdsMap = FileUtil.getInstance().handleReqFile((MultipartHttpServletRequest) request,
                    mp, billType, authority, deductionEntity.getId().toString());

            List<Long> attchIdsList = new ArrayList<>();
            for(List<Long> attachIds : attachIdsMap.values()) {
                if(CollectionUtils.isNotEmpty(attachIds)) {
                    attchIdsList.addAll(attachIds);
                }
            }
            //将附件关联在单据中
            deductionEntity.setAttachIds(attchIdsList);
            //将单据设置为乙方已签字状态
            deductionEntity.setSupplierSignStatus(MeasureVO.SUP_HAS_SIGNED_STATUS);

            //更新单据
            super.saveOrUpdate(deductionEntity, false);

            //向单据制单人和经办人推送该消息
            String msgSendResult = sendMsg(deductionEntity, "供方已签字提醒", "扣奖罚单据["+deductionEntity.getBillCode()+"]供方已签字完成");
            if(null != msgSendResult) {
                logger.error("向用户-{}发送扣奖罚单据id-{}签字提醒失败，原因：{}", StringUtils.join(deductionEntity.getCreateUserId(), deductionEntity.getEmployeeId()),
                        deductionEntity.getId(), msgSendResult);
            }
        } catch (Exception e) {
            logger.error("扣奖罚单据id-{}签字信息回写异常，", deductionEntity.getId(), e);
            msg = "扣奖罚单据签字信息回写失败！";
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

        return msg;
    }

    private String sendMsg(DeductionEntity deductionEntity, String subject, String content) {
        String[] msgRecUserIds = new String[] {deductionEntity.getCreateUserId().toString(), deductionEntity.getEmployeeId().toString()};
        logger.info("消息接收人: {}", StringUtils.join(msgRecUserIds, ","));
        PushMsgParameter msgParameter = new PushMsgParameter();
        //消息接收人
        msgParameter.setReceivers(msgRecUserIds);
        //消息内容
        msgParameter.setContent(content);
        //消息主题
        msgParameter.setSubject(subject);
        //消息类型
        msgParameter.setMsgType("notice");
        msgParameter.setTenantId(deductionEntity.getTenantId().toString());
        //消息保存
        msgParameter.setSaveFlag(true);
        //消息发送人
        msgParameter.setSendUserId(InvocationInfoProxy.getUserid());
        //消息发送渠道
        msgParameter.setChannel(new String[]{PushMsgParameter.CHANNEL_TYPE_SYS});

        CommonResponse<String> msgSendResp = pushMessageApi.pushMessage(msgParameter);
        if(!msgSendResp.isSuccess()) {
            return msgSendResp.getMsg();
        }
        return null;
    }

    public void releaseLock(Jedis jedis, boolean locked, String key, String OPERATE) {
        try {
            if(locked) {
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
    }
}
