package com.ejianc.business.purchase.utils;

import com.ejianc.framework.core.kit.base.annotation.NotNull;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.regex.Pattern;

public class DateUtil {

    public final static String CURRENT_YEAR = getCurrentYear();

    /**
     * 两个日期相差的天数，正数+1，负数-1
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static Integer getSubDay(Date startDate, Date endDate) {
        if(startDate == null || endDate == null){
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        Calendar cal = Calendar.getInstance();
        long betweenDays = 0L;
        try {
            cal.setTime(sdf.parse(sdf.format(startDate)));
            long time1 = cal.getTimeInMillis();
            cal.setTime(sdf.parse(sdf.format(endDate)));
            long time2 = cal.getTimeInMillis();
            betweenDays = (time1 - time2) / (1000 * 3600 * 24);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        Integer days = Integer.parseInt(String.valueOf(betweenDays));
        return days < 0 ? days - 1 : days + 1;
    }

    /***
     * @Author mrsir_wxp
     * @Date 2021/2/1 比较两个日期大小，
     * @Description compareDate
     * @Param [dt1, dt2]
     * @Return 1：dt1>dt2,-1：dt1<dt2, 0相等
     */
    public static int compareDate(Date dt1, Date dt2) {
        try {
            if (dt1.getTime() > dt2.getTime()) {
                return 1;
            } else if (dt1.getTime() < dt2.getTime()) {
                return -1;
            } else {
                return 0;
            }
        } catch (Exception exception) {
            exception.printStackTrace();
        }
        return 0;
    }

    /***
     * @Author mrsir_wxp
     * @Date 2021/2/1 比较两个日期大小，
     * @Description compareDate
     * @Param [dt1, dt2]
     * @Return 1：dt1>dt2,-1：dt1<dt2, 0相等
     */
    public static int compareDay(Date dt1, Date dt2) {
        dt1 = parseDate(formatDate(dt1));
        dt2 = parseDate(formatDate(dt2));
        return compareDate(dt1, dt2);
    }

    /**
     * 是否同一天
     * @param date1
     * @param date2
     * @return
     */
    public static boolean isSameDay(final Date date1, final Date date2) {
        if (date1 == null || date2 == null) {
            return false;
        }
        return formatDate(date1).equals(formatDate(date2));
    }

    /**
     * 是否同一周
     * @param date1
     * @param date2
     * @return
     */
    public static boolean isSameWeek(Date date1, Date date2){
        Calendar cal1 = Calendar.getInstance();
        cal1.setTime(date1);
        Calendar cal2 = Calendar.getInstance();
        cal2.setTime(date2);
        return cal1.get(Calendar.YEAR) == cal2.get(Calendar.YEAR) &&
                cal1.get(Calendar.WEEK_OF_YEAR) == cal2.get(Calendar.WEEK_OF_YEAR);
    }

    /**
     * 是否同一月
     * @param date1
     * @param date2
     * @return
     */
    public static boolean isSameMonth(final Date date1, final Date date2) {
        if (date1 == null || date2 == null) {
            return false;
        }
        return formatDate(date1, "yyyy-MM").equals(formatDate(date2, "yyyy-MM"));
    }

    public static boolean isSameMonth(final Date date1, final String date2) {
        if (date1 == null || date2 == null) {
            return false;
        }
        return formatDate(date1, "yyyy-MM").equals(date2);
    }
    /**
     * 当周第一天
     * @param date
     * @return
     */
    public static Date beginOfWeek(@NotNull final Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DAY_OF_WEEK, -1);
        cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        return cal.getTime();
    }

    /**
     * 当月第一天
     * @param date
     * @return
     */
    public static Date beginOfMonth(@NotNull final Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        return cal.getTime();
    }

    /**
     * 当年第一天
     * @param date
     * @return
     */
    public static Date beginOfYear(@NotNull final Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.MONTH, 0);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        return cal.getTime();
    }

    /**
     * 当周最后一天
     * @param date
     * @return
     */
    public static Date endOfWeek(@NotNull final Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DAY_OF_WEEK, -1);
        cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        cal.add(Calendar.DAY_OF_WEEK, 6);
        return endOfDate(cal.getTime());
    }

    /**
     * 当月最后一天
     * @param date
     * @return
     */
    public static Date endOfMonth(@NotNull final Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.MONTH, 1);
        cal.set(Calendar.DAY_OF_MONTH, 0);
        return endOfDate(cal.getTime());
    }

    /**
     * 上年最后一天
     * @param date
     * @return
     */
    public static Date endOfLastYear(@NotNull final Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_YEAR, 0);
        return endOfDate(cal.getTime());
    }

    /**
     * 2017-1-23 07:33:23, 则返回2017-1-23 23:59:59.999
     */
    public static Date endOfDate(@NotNull final Date date) {
        return new Date(nextDate(date).getTime() - 1);
    }

    /**
     * 2016-11-10 07:33:23, 则返回2016-11-11 00:00:00
     */
    public static Date nextDate(@NotNull final Date date) {
        return DateUtils.ceiling(date, Calendar.DATE);
    }

    public static int daysOfMonth(@NotNull final Date date){
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DATE, 1);
        cal.roll(Calendar.DATE, -1);
        return cal.get(Calendar.DATE);
    }

    /**
     * 比较两个日期并取最大值或最小值
     * @param date1
     * @param date2
     * @param flag false-最小值，true-最大值
     * @return
     */
    public static Date maxDate(Date date1, Date date2, Boolean flag) {
        if(date1 == null) {
            return date2;
        }
        if(date2 == null){
            return date1;
        }
        if(flag){
            if(compareDate(date1, date2) < 0){
                return date2;
            }
            return date1;
        } else {
            if(compareDate(date1, date2) < 0){
                return date1;
            }
            return date2;
        }
    }

    /**
     * 比较日期大小,如2020-10-23格式
     *
     * @param t1
     * @param t2
     * @return
     */
    public static int compareDateStr(String t1, String t2) {
        int trr1 = Integer.parseInt(t1.replaceAll("-", ""));
        int trr2 = Integer.parseInt(t2.replaceAll("-", ""));
        if (trr1 > trr2) {
            return 1;
        } else if (trr1 < trr2) {
            return -1;
        }
        return 0;
    }

    /**
     * 获取日期月份字符串，如01，02...
     * @description 这里使用了String.format（）方法，作用就是格式化输出参数. “%02d”是指定输出格式，%作先导标记，0表示自动补0, 2的意思是最小长度为2（如果用4，则1输出0001），d表示整数
     * @param date
     * @return
     */
    public static String getMonthStr(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        int month = cal.get(Calendar.MONTH) + 1;
        return String.format("%02d",month);
    }

    /**
     * 获取日期年份字符串，如2021，2022...
     * @description 这里使用了String.format（）方法，作用就是格式化输出参数. “%02d”是指定输出格式，%作先导标记，0表示自动补0, 2的意思是最小长度为2（如果用4，则1输出0001），d表示整数
     * @param date
     * @return
     */
    public static String getYearStr(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        String year = String.valueOf(cal.get(Calendar.YEAR));// 当前年
        return year;
    }

    /**
     * 获取当前年
     * @return
     */
    public static String getCurrentYear() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        String year = String.valueOf(cal.get(Calendar.YEAR));// 当前年
        return year;
    }

    /**
     * 获取上一年
     * @return
     */
    public static int getLastYear() {
        Calendar cal = Calendar.getInstance();
        int i = cal.get(Calendar.YEAR) - 1;
        return i;
    }

    /**
     * 获取当前月
     * @return
     */
    public static String getCurrentMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        return sdf.format(cal.getTime());
    }

    /**
     * 获取上一月
     * @return
     */
    public static String getLastMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(Calendar.MONTH, -1);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        return sdf.format(cal.getTime());
    }

    /**
     * 将中文环境下的1-周一，...，7-周日，改为西方习惯的2-周一，...，7-周六，1-周日
     * @param day
     * @return
     */
    public static Integer turnZnDayOfWeek(Integer day) {
        day = day + 1;
        return day > 7 ? day - 6 : day;
    }

    /**
     * 将中文环境下的1-28及最后一天，改为对应日期
     * @param day
     * @return
     */
    public static Integer turnZnDayOfMonth(Integer day) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        if(day > cal.getActualMaximum(Calendar.DAY_OF_MONTH)){// 月末加1天，2月29日或者31、32特殊处理
            cal.add(Calendar.DATE, 1);
            day = cal.get(Calendar.DAY_OF_MONTH );
            return day;
        }
        return day == 30 ? cal.getActualMaximum(Calendar.DAY_OF_MONTH) : day;
    }

    /**
     * 将任意日期格式字符串，重新翻译为标准日期格式
     * @param str
     * @return
     * @throws ParseException
     */
    public static String formatDateString(String str) {
        if(StringUtils.isEmpty(str)){
            return str;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        SimpleDateFormat bsdf = new SimpleDateFormat(getDateFormat(str));
        String dateStr = null;
        try {
            dateStr = sdf.format(bsdf.parse(str));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return dateStr;
    }

    /**
     * 常规自动日期格式识别
     * @param str 时间字符串
     * @return Date
     * @author dc
     */
    public static String getDateFormat(String str) {
        boolean year = false;
        Pattern pattern = Pattern.compile("^[-\\+]?[\\d]*$");
        if(pattern.matcher(str.substring(0, 4)).matches()) {
            year = true;
        }
        StringBuilder sb = new StringBuilder();
        int index = 0;
        if(!year) {
            if(str.contains("月") || str.contains("-") || str.contains("/")) {
                if(Character.isDigit(str.charAt(0))) {
                    index = 1;
                }
            }else {
                index = 3;
            }
        }
        for (int i = 0; i < str.length(); i++) {
            char chr = str.charAt(i);
            if(Character.isDigit(chr)) {
                if(index==0) {
                    sb.append("y");
                }
                if(index==1) {
                    sb.append("M");
                }
                if(index==2) {
                    sb.append("d");
                }
                if(index==3) {
                    sb.append("H");
                }
                if(index==4) {
                    sb.append("m");
                }
                if(index==5) {
                    sb.append("s");
                }
                if(index==6) {
                    sb.append("S");
                }
            }else {
                if(i>0) {
                    char lastChar = str.charAt(i-1);
                    if(Character.isDigit(lastChar)) {
                        index++;
                    }
                }
                sb.append(chr);
            }
        }
        return sb.toString();
    }

    public static Date addDays(@NotNull final Date date, final int amount) {
        return DateUtils.addDays(date, amount);
    }

    public static Date addMonths(@NotNull final Date date, int amount) {
        return DateUtils.addMonths(date, amount);
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static Date parseDate(String date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            return sdf.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return new Date();
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static String formatDate(Date date) {
        if(date == null){
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        return sdf.format(date);
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static String formatDate(Date date, String pattern) {
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        return sdf.format(date);
    }

    /**
     * 转换成标准日期格式
     * @param date
     * @return
     */
    public static String formatSeconds(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return sdf.format(date);
    }

    public static void main(String[] args) {
        String date1 = "2023-07-02 23:59:59";
        String date2 = "2025-03-01";
        System.out.println(beginOfYear(parseDate(date1)));
        System.out.println(formatDate(beginOfYear(parseDate(date1))));
    }

}
