package com.ejianc.business.quatity.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.service.additional.update.impl.LambdaUpdateChainWrapper;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.quatity.dao.ExperienceFeedbackIssuedDao;
import com.ejianc.business.quatity.dao.ExperienceFeedbackIssuedDetailDao;
import com.ejianc.business.quatity.entity.ExperienceFeedbackIssuedDetailEntity;
import com.ejianc.business.quatity.entity.ExperienceFeedbackIssuedEntity;
import com.ejianc.business.quatity.model.vo.ExpFeedBackIssuedBankAddVo;
import com.ejianc.business.quatity.model.vo.ExpFeedBackReportVo;
import com.ejianc.business.quatity.model.vo.ExperienceFeedbackIssuedDetailVo;
import com.ejianc.business.quatity.service.ExperienceFeedbackServer;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.sql.Date;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/5/15 上午10:10
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class ExperienceFeedbackServerImpl implements ExperienceFeedbackServer {
    private final ExperienceFeedbackIssuedDao dao;
    private final ExperienceFeedbackIssuedDetailDao detailDao;

    private final SessionManager sessionManager;

    @Override
    public IPage<ExperienceFeedbackIssuedDetailEntity> getTasksList(QueryParam param) {
        param.getFuzzyFields().addAll(Arrays.asList("attributionType", "specialized", "topic", "substance"));

        param.getOrderMap().put("createTime", QueryParam.DESC);
        //获取当前组织树的任务
        param.getParams().put("uploadDepartmentId", new Parameter(QueryParam.EQ, sessionManager.getUserContext().getOrgId()));

        return detailDao.queryPage(param, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void report(ExpFeedBackReportVo vo) {
        Long id = vo.getId();
        ExperienceFeedbackIssuedDetailEntity reportEntity = detailDao.getById(id);
        if (Objects.isNull(reportEntity)) {
            throw new BusinessException("上报数据查询失败");
        }
        Long issuedId = reportEntity.getIssuedId();

        reportEntity.setAnnex(vo.getAnnex());
        reportEntity.setPreparedUserCode(sessionManager.getUserContext().getUserCode());
        reportEntity.setPreparedUserName(sessionManager.getUserContext().getUserName());
        reportEntity.setUploadTime(new Date(System.currentTimeMillis()));
        reportEntity.setState("已上报");
        reportEntity.setOrgId(sessionManager.getUserContext().getOrgId());
        reportEntity.setOrgName(sessionManager.getUserContext().getOrgName());
        try {
            detailDao.saveOrUpdate(reportEntity);
        } catch (Exception e) {
            throw new BusinessException("上报失败，原因" + e.getMessage());
        } finally {
            List<ExperienceFeedbackIssuedDetailEntity> collect = detailDao.lambdaQuery()
                    .in(ExperienceFeedbackIssuedDetailEntity::getIssuedId, issuedId)
                    .list()
                    .stream()
                    .filter(e -> "未上报".equals(e.getState()))
                    .collect(Collectors.toList());
            if (collect.isEmpty()) {
                dao.lambdaUpdate().eq(BaseEntity::getId, issuedId).set(ExperienceFeedbackIssuedEntity::getCompletion, "已完成").update();
            } else {
                dao.lambdaUpdate().eq(BaseEntity::getId, issuedId).set(ExperienceFeedbackIssuedEntity::getCompletion, "部分完成").update();
            }
        }
    }

    @Override
    public IPage<ExperienceFeedbackIssuedDetailEntity> list(QueryParam param) {
        param.getFuzzyFields().addAll(Arrays.asList("attributionType", "specialized", "topic", "substance"));

        param.getOrderMap().put("createTime", QueryParam.DESC);

        return detailDao.queryPage(param, false);
    }

    @Override
    public void excelExport(List<Long> ids, HttpServletResponse response) {

        List<ExperienceFeedbackIssuedDetailEntity> list = detailDao.lambdaQuery().in(BaseEntity::getId, ids).list();
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        try {
            ExcelExport.getInstance().export("experienceFeedBackExport.xlsx", beans, response);
        } catch (Exception e) {
            throw new BusinessException("经验反馈导出失败：" + e.getMessage());
        }
    }

    @Override
    public ExperienceFeedbackIssuedDetailEntity issuedDetail(Long id) {
        try {
            return detailDao.selectById(id);
        } catch (Exception e) {
            throw new BusinessException("经验反馈查询：" + e.getMessage());
        }
    }

    @Override
    public void issuedDel(List<Long> ids) {
        try {
            detailDao.removeByIds(ids);
        } catch (Exception e) {
            throw new BusinessException("经验反馈删除：" + e.getMessage());
        }
    }

    @Override
    public ExperienceFeedbackIssuedDetailVo saveOrUpdate(ExperienceFeedbackIssuedDetailVo saveOrUpdateVO) {
        ExperienceFeedbackIssuedDetailEntity entity = BeanMapper.map(saveOrUpdateVO, ExperienceFeedbackIssuedDetailEntity.class);
        detailDao.saveOrUpdate(entity, false);
        return BeanMapper.map(entity, ExperienceFeedbackIssuedDetailVo.class);
    }

    @Override
    public void bankEnter(List<Long> ids) {
        try {
            detailDao.lambdaUpdate().in(BaseEntity::getId, ids).set(ExperienceFeedbackIssuedDetailEntity::getIsBank, true).update();
        } catch (Exception e) {
            throw new BusinessException("入库失败：" + e.getMessage());
        }

    }

    @Override
    public void bankDel(List<Long> ids) {
        try {
            List<ExperienceFeedbackIssuedDetailEntity> list = detailDao.lambdaQuery().in(BaseEntity::getId, ids).list();

            List<Long> objects1 = Lists.newArrayList();
            List<Long> objects2 = Lists.newArrayList();
            for (ExperienceFeedbackIssuedDetailEntity entity : list) {
                //经验反馈入库的数据
                if (Objects.nonNull(entity.getIssuedId())) {
                    objects1.add(entity.getId());
                } else {
                    //经验反馈库直接新增的数据
                    objects2.add(entity.getId());
                }
            }
            detailDao.lambdaUpdate().in(BaseEntity::getId, objects1).set(ExperienceFeedbackIssuedDetailEntity::getIsBank, false).update();
            detailDao.lambdaUpdate().in(BaseEntity::getId, objects2).remove();

        } catch (Exception e) {
            throw new BusinessException("经验反馈库删除失败：" + e.getMessage());
        }

    }

    @Override
    public ExperienceFeedbackIssuedDetailVo bankAdd(ExpFeedBackIssuedBankAddVo vo) {
        ExperienceFeedbackIssuedDetailEntity entity = BeanMapper.map(vo, ExperienceFeedbackIssuedDetailEntity.class);

        entity.setUploadDepartmentId(JSONObject.toJSONString(vo.getUploadDepartmentId()));
        entity.setUploadDepartmentName(JSONObject.toJSONString(vo.getUploadDepartmentName()));

        entity.setUploadTime(new Date(System.currentTimeMillis()));
        entity.setIsBank(true);
        detailDao.saveOrUpdate(entity, false);
        return BeanMapper.map(entity, ExperienceFeedbackIssuedDetailVo.class);

    }
}
