package com.ejianc.foundation.report.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.CustomColumnEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.ICustomColumnService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.CalculatorUtils;
import com.ejianc.foundation.report.util.PinYinUtil;
import com.ejianc.foundation.report.vo.ColumnVO;
import com.ejianc.foundation.report.vo.CustomColumnVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ResultAsTree;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.Requests;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;

@SuppressWarnings("deprecation")
@RestController
@RequestMapping("/col/")
public class ColumnController implements Serializable {

	private static final long serialVersionUID = 880340424115528835L;

	@Value("${oms.tenantid}")
	private String OMS_TENANT;
	
	@Autowired
	private IColumnService columnService;
	@Autowired
	private ICustomColumnService customColumnService;
	@Autowired
	private ITableService tableService;
	@Autowired
	private RestHighLevelClient client;
	
	/**
	 * 新增或者修改
	 * 
	 * @param Datas
	 * @return
	 * @throws BillCodeException 
	 */
	@RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> saveOrUpdate(@RequestBody ColumnVO columnVo) {
		TableEntity tableEntity = tableService.getById(columnVo.getTableId());

		if(StringUtils.isNotBlank(columnVo.getProperty())) {
			try {
				XContentBuilder mapping = null;
				switch (columnVo.getType()) {
					case "string":
						mapping = jsonBuilder().startObject().startObject(tableEntity.getIndexName()).startObject("properties")
								.startObject(columnVo.getProperty())
								.field("type", "text")
								.field("analyzer","ik_max_word")
								.field("search_analyzer","ik_smart")
								.endObject()
								.endObject()
								.endObject()
								.endObject();
						break;
					case "number":
						mapping = jsonBuilder().startObject().startObject(tableEntity.getIndexName()).startObject("properties")
								.startObject(columnVo.getProperty())
								.field("type", "long")
								.endObject()
								.endObject()
								.endObject()
								.endObject();
						break;
					case "time":
						mapping = jsonBuilder().startObject().startObject(tableEntity.getIndexName()).startObject("properties")
								.startObject(columnVo.getProperty())
								.field("type", "date")
								.field("format","yyyy-MM-dd HH:mm:ss")
								.endObject()
								.endObject()
								.endObject()
								.endObject();
						break;
				}

				PutMappingRequest mappingRequest = Requests.putMappingRequest(tableEntity.getIndexName()).type(tableEntity.getIndexName()).source(mapping);
				client.indices().putMapping(mappingRequest, RequestOptions.DEFAULT);
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		if(columnVo.getId() != null && columnVo.getId() > 0) {
			ColumnEntity uniqueBean = columnService.selectById(columnVo.getId());
			uniqueBean.setType(columnVo.getType());
			uniqueBean.setColumnName(columnVo.getColumnName());
			if(StringUtils.isNotBlank(columnVo.getFormula())) {
				List<ColumnEntity> columnList = columnService.queryFormulaList(columnVo.getTableId());
				Map<String, String> propertyMap = new HashMap<>();
				for(ColumnEntity columnEntity:columnList) {
					propertyMap.put(columnEntity.getProperty(), columnEntity.getProperty());
				}
				Boolean checked = CalculatorUtils.checkFormula(columnVo.getFormula(), propertyMap);
				if(!checked) {
					return CommonResponse.error("公式错误，不允许保存");
				}
				String property = PinYinUtil.getFullSpell(columnVo.getColumnName());
				uniqueBean.setProperty(property);
			}else{
				uniqueBean.setProperty(columnVo.getProperty());
			}
			uniqueBean.setFormula(columnVo.getFormula());
			uniqueBean.setSearchFlag(columnVo.getSearchFlag());
			uniqueBean.setVisible(columnVo.getVisible());
			uniqueBean.setSequence(columnVo.getSequence());
			uniqueBean.setFormatter(columnVo.getFormatter());
			//唯一性校验
			QueryWrapper<ColumnEntity> queryWrapper = new QueryWrapper<>();
			queryWrapper.eq("table_id", uniqueBean.getTableId());
			queryWrapper.eq("property", uniqueBean.getProperty());
			queryWrapper.ne("id", columnVo.getId());
			List<ColumnEntity> list = columnService.list(queryWrapper );
			if(list!=null&&list.size()>0){
				if(StringUtils.isNotBlank(uniqueBean.getFormula())){
					return CommonResponse.error("列名:"+uniqueBean.getColumnName()+"已存在");
				}else{
					return CommonResponse.error("属性:"+uniqueBean.getProperty()+"已存在");
				}
			}
			
			columnService.saveOrUpdate(uniqueBean, false);
			
			return CommonResponse.success("修改成功");
		}else{
			ColumnEntity saveBean = BeanMapper.map(columnVo, ColumnEntity.class);
			if(StringUtils.isNotBlank(columnVo.getFormula())) {
				List<ColumnEntity> columnList = columnService.queryFormulaList(columnVo.getTableId());
				Map<String, String> propertyMap = new HashMap<>();
				for(ColumnEntity columnEntity:columnList) {
					propertyMap.put(columnEntity.getProperty(), columnEntity.getProperty());
				}
				Boolean checked = CalculatorUtils.checkFormula(columnVo.getFormula(), propertyMap);
				if(!checked) {
					return CommonResponse.error("公式错误，不允许保存");
				}
				String property = PinYinUtil.getFullSpell(columnVo.getColumnName());
				saveBean.setProperty(property);
			}else{
				saveBean.setProperty(columnVo.getProperty());
			}
			if(columnVo.getSearchFlag() == null) {
				saveBean.setSearchFlag(0);
			}
			//唯一性校验
			QueryWrapper<ColumnEntity> queryWrapper = new QueryWrapper<>();
			queryWrapper.eq("table_id", saveBean.getTableId());
			queryWrapper.eq("property", saveBean.getProperty());
			List<ColumnEntity> list = columnService.list(queryWrapper);
			if(list!=null&&list.size()>0){
				if(StringUtils.isNotBlank(saveBean.getFormula())){
					return CommonResponse.error("列名:"+saveBean.getColumnName()+"已存在");
				}else{
					return CommonResponse.error("属性:"+saveBean.getProperty()+"已存在");
				}
			}
			
			columnService.saveOrUpdate(saveBean, false);
			
			return CommonResponse.success("保存成功");
		}
	}
	
	/**
	 * 查询待分页的列表
	 * 
	 * @param queryParam
	 * @return
	 */
	@RequestMapping(value = "/pageList", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<JSONObject> pageList(@RequestBody QueryParam queryParam) {
		queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//由于树形展示数据，则不进行分页查询
		queryParam.setPageIndex(1);
		queryParam.setPageSize(10000);
		IPage<ColumnEntity> pageData = columnService.queryPage(queryParam);
		List<Map> voList = BeanMapper.mapList(pageData.getRecords(), Map.class);

		JSONObject resp = new JSONObject();
		resp.put("current", pageData.getCurrent());
		resp.put("size", pageData.getSize());
		resp.put("total", pageData.getTotal());
		resp.put("pages", pageData.getPages());

		List<Map<String, Object>> records = ResultAsTree.createTreeData(voList);
		resp.put("records", records);

		JSONObject jsonObject = new JSONObject();
		jsonObject.put("data", resp);
		return CommonResponse.success(jsonObject);
	}
	
	/**
	 * 查询可计算的列
	 * 
	 * @param
	 * @return
	 */
	@RequestMapping(value = "/queryFormulaList", method = RequestMethod.GET)
	@ResponseBody
	public CommonResponse<JSONObject> queryFormulaList(@RequestParam Long tableId) {
		List<ColumnEntity> columnList = columnService.queryFormulaList(tableId);
		JSONObject jsonObject = new JSONObject();
		jsonObject.put("data", columnList);
		return CommonResponse.success(jsonObject);
	}
	
	/**
	 * 根据主键ID查询详情
	 * 
	 * @param id
	 * @return
	 */
	@RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
	@ResponseBody
	public CommonResponse<ColumnVO> queryDetail(@RequestParam Long id) {
		ColumnEntity columnEntity = columnService.selectById(id);
		if(columnEntity != null) {
			ColumnVO columnVo = BeanMapper.map(columnEntity, ColumnVO.class);
			return CommonResponse.success(columnVo);
		}
		return null;
	}
	
	/**
	 * 删除
	 * 
	 * @param ids
	 * @return
	 */
	@RequestMapping(value = "/delete", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> delete(@RequestBody List<Long> ids) {
		columnService.deleteByIds(ids);
		return CommonResponse.success("删除成功");
	}
	
	/**
	 * 列自定义
	 * 
	 * @param customColumnVo
	 * @return
	 */
	@RequestMapping(value = "/custom", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> customColumn(@RequestBody CustomColumnVO customColumnVo) {
		QueryWrapper<CustomColumnEntity> customColumnWrapper = new QueryWrapper<>();
		customColumnWrapper.eq("column_id", customColumnVo.getColumnId());
		CustomColumnEntity customColumnEntity = customColumnService.getOne(customColumnWrapper);
		if(customColumnEntity != null) {
			if(StringUtils.isNotBlank(customColumnVo.getColumnName())) {
				customColumnEntity.setColumnName(customColumnVo.getColumnName());
			}
			if(StringUtils.isNotBlank(customColumnVo.getVisible())) {
				customColumnEntity.setVisible(customColumnVo.getVisible());
			}
			customColumnService.saveOrUpdate(customColumnEntity, false);
		}else{
			CustomColumnEntity saveBean = new CustomColumnEntity();
			saveBean.setColumnId(customColumnVo.getColumnId());
			if(StringUtils.isNotBlank(customColumnVo.getColumnName())) {
				saveBean.setColumnName(customColumnVo.getColumnName());
			}
			if(StringUtils.isNotBlank(customColumnVo.getVisible())) {
				saveBean.setVisible(customColumnVo.getVisible());
			}
			customColumnService.saveOrUpdate(saveBean, false);
		}
		return CommonResponse.success("设置成功");
	}
	
	/**
	 * 恢复自定义列
	 * 
	 * @return
	 */
	@RequestMapping(value = "/resetCustomColumn", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> resetCustomColumn(@RequestParam Long tableId) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		customColumnService.deleteByTenantId(tableId, tenantId);
		return CommonResponse.success("恢复成功");
	}

}
