package com.ejianc.foundation.report.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.metadata.api.IMdProjectApi;
import com.ejianc.foundation.metadata.vo.MdProjectVO;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.DatasetEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.IDatasetService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.CalculatorUtils;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.util.JdkBase64Util;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/data")
@SuppressWarnings("unchecked")
public class DataExtractController implements Serializable {
	private static final Logger logger = LoggerFactory.getLogger(DataExtractController.class);
	private static final long serialVersionUID = 169770636614492806L;

	private volatile static String authority = null;
	private volatile static Long tenantId = null;
	private final static Integer QUERY_TIMEOUT = 60;
	private final static Integer BATCH_SIZE = 200;

	@Value("${common.env.base-host}")
	private String baseHost;
	@Autowired
	private IDatasetService datasetService;
	@Autowired
	private ITableService tableService;
	@Autowired
	private IMdProjectApi projectApi;
	@Autowired
	private IColumnService columnService;
	@Autowired
	private RestHighLevelClient client;

	@RequestMapping(value = "/extract", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> extractData(@RequestParam Long tableId, HttpServletRequest request) {
		authority = request.getHeader("authority");
		tenantId = InvocationInfoProxy.getTenantid();
		// 查询数据表
		TableEntity tableEntity = tableService.selectById(tableId);
		List<Long> datasetIds = new ArrayList<>(Arrays.asList(tableEntity.getDatasetIds().split(","))).stream().map(s -> Long.valueOf(s)).collect(Collectors.toList());
        List<JSONObject> data = new ArrayList<>();
        BulkRequest bulkRequest = new BulkRequest();
        int count = 1;

        try {
            for(Long datasetId : datasetIds) {
                // 查询主数据集
                DatasetEntity datasetEntity = datasetService.selectById(datasetId);
                // 通过元数据查询该数据集属于哪个项目
                CommonResponse<MdProjectVO> projectResponse = projectApi.queryDetail(datasetEntity.getMdProjectId());
                if (projectResponse.isSuccess()) {
                    MdProjectVO projectVO = projectResponse.getData();
                    String mainDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";
                    JSONObject mainParamJson = new JSONObject();
                    JSONObject mainQueryParam = new JSONObject();
                    mainQueryParam.put("tenantId", InvocationInfoProxy.getTenantid());
                    mainParamJson.put("sqlContent", JdkBase64Util.encode(datasetEntity.getSqlContent()));
                    mainParamJson.put("datasetType", "1");
                    mainParamJson.put("params", mainQueryParam);

                    String mainResponseStr = ReferHttpClientUtils.postByJson(mainDatasetUrl, JSON.toJSONString(mainParamJson));

                    CommonResponse<List<JSONObject>> mainResponse = JSON.parseObject(mainResponseStr, CommonResponse.class);
                    if (mainResponse.isSuccess()) {
                        List<JSONObject> mainDatasetList = mainResponse.getData();

                        // 查询子数据集
                        List<DatasetEntity> childDatasetList = datasetService
                                .queryChildrenByParentId(datasetEntity.getId());
                        if (mainDatasetList != null && mainDatasetList.size() > 0) {
                            // 先根据创建者空间查询当前空间的所有数据,然后删除之
                            Long resultSize = queryIndexSize(tableEntity.getIndexName(),
                                    InvocationInfoProxy.getTenantid());
                            Long deleteTimes = resultSize % BATCH_SIZE;
                            for (int i = 0; i < deleteTimes + 1; i++) {
                                DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(tableEntity.getIndexName());
                                deleteByQueryRequest.setQuery(new TermQueryBuilder("creator_space", InvocationInfoProxy.getTenantid()));
                                deleteByQueryRequest.setBatchSize(BATCH_SIZE);
                                client.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
                            }

                            // 查询数据列
                            Map<String, Object> paramMap = new HashMap<>();
                            paramMap.put("tableId", tableEntity.getId());
                            paramMap.put("tenantId", InvocationInfoProxy.getTenantid());
                            List<ColumnEntity> columnEntities = columnService.queryTenantColumnList(paramMap);
                            Map<String, String> columnMap = new HashMap<String, String>();
                            if (columnEntities != null && columnEntities.size() > 0) {
                                for (ColumnEntity columnEntity : columnEntities) {
                                    if(StringUtils.isNotBlank(columnEntity.getProperty())) {
                                        //属性为空的列认为是复核列
                                        if (StringUtils.isBlank(columnEntity.getFormula())) {
                                            columnMap.put(columnEntity.getProperty(), columnEntity.getType());
                                        } else {
                                            columnMap.put(columnEntity.getProperty(), columnEntity.getFormula());
                                        }
                                    }
                                }
                            }

                            int sequence = 1;
                            ExecutorService threadPool = null;
                            if(CollectionUtil.isNotEmpty(childDatasetList)) {
                                threadPool = Executors.newFixedThreadPool(childDatasetList.size());
                            }
                            for (JSONObject mainDataset : mainDatasetList) {
                                if (childDatasetList != null && childDatasetList.size() > 0) {
                                    List<Future<List<JSONObject>>> futureList = new ArrayList<>();
                                    for (DatasetEntity childDataset : childDatasetList) {
                                        Callable<List<JSONObject>> childCallable = new ChildDatasetCallable(childDataset, mainDataset, RequestContextHolder.getRequestAttributes());
                                        Future<List<JSONObject>> childFuture = threadPool.submit(childCallable);
                                        futureList.add(childFuture);
                                    }

                                    for (Future<List<JSONObject>> future : futureList) {
                                        List<JSONObject> childList = future.get();
                                        if (childList != null && childList.size() > 0) {
                                            JSONObject childObject = childList.get(0);
                                            for (Map.Entry<String, Object> entry : childObject.entrySet()) {
                                                mainDataset.put(entry.getKey(), entry.getValue());
                                            }
                                        }
                                    }
                                }

                                Map<String, Object> dataMap = new HashMap<>();
                                //排序
                                dataMap.put("creator_space", InvocationInfoProxy.getTenantid());
                                dataMap.put("data_sequence", count++);
                                for (Map.Entry<String, String> columnEntry : columnMap.entrySet()) {
                                    if ("time".equals(columnEntry.getValue())) {
                                        if(null != mainDataset.get(columnEntry.getKey())) {
                                            dataMap.put(columnEntry.getKey(), new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(columnEntry.getKey())));
                                            mainDataset.put(columnEntry.getKey(), new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(columnEntry.getKey())));
                                        }
                                    } else if ("string".equals(columnEntry.getValue()) || "number".equals(columnEntry.getValue())) {
                                        dataMap.put(columnEntry.getKey(), mainDataset.get(columnEntry.getKey()) + "");
                                        mainDataset.put(columnEntry.getKey(), mainDataset.get(columnEntry.getKey()));
                                    } else { // 计算公式
                                        dataMap.put(columnEntry.getKey(), CalculatorUtils.getResult(columnEntry.getValue(), mainDataset));
                                        mainDataset.put(columnEntry.getKey(), CalculatorUtils.getResult(columnEntry.getValue(), mainDataset));
                                    }
                                }

                                IndexRequest indexRequest = new IndexRequest(tableEntity.getIndexName());
                                indexRequest.source(dataMap, XContentType.JSON);
                                bulkRequest.add(indexRequest);
                            }
                        }

                        data.addAll(mainDatasetList);
                    }
                } else {
                    logger.error("元数据查询失败, 原因：", projectResponse.getMsg());
                }
            }
            //设置立即更新以解决数据查询不到的问题。
			bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
			client.bulk(bulkRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            logger.info("查询主数据集失败，", e);
            return CommonResponse.error("查询主数据集失败");
        }

        // 查询表头
//        List<GridHeader> gridHeaders = columnService.queryGridHeadList(tableId, tenantId);
//        JSONObject responseData = new JSONObject();
//        responseData.put("reportTitle", tableEntity.getTableName());
//        responseData.put("filterItems", tableEntity.getConditionJson());
//        responseData.put("data", data); //不再直接返回数据，而是需要页面重新加载数据
//        responseData.put("orderGroup", tableEntity.getOrderItemJson());

//        List<Map> headerList = BeanMapper.mapList(gridHeaders, Map.class);
//        responseData.put("gridheaders", ResultAsTree.createTreeData(headerList));
//        return CommonResponse.success("执行成功", responseData);
        return CommonResponse.success("执行成功");
	}

	/**
	 * 查询当前空间的所有数据
	 * 
	 * @param indexName
	 * @param creatorSpace
	 * @return
	 */
	private Long queryIndexSize(String indexName, Long creatorSpace) {
		Long resultSize = 0L;
		SearchRequest searchRequest = new SearchRequest(indexName);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();

		// 查询参数
		TermQueryBuilder termQueryBuilder = QueryBuilders.termQuery("creator_space", creatorSpace);

		BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
		boolQueryBuilder.must(termQueryBuilder);

		sourceBuilder.trackTotalHits(true);
		sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); // 设置超时时间
		searchRequest.source(sourceBuilder);
		try {
			resultSize = queryCreatorSize(searchRequest);
		} catch (IOException e) {
			try { // 重试一次
				resultSize = queryCreatorSize(searchRequest);
			} catch (IOException e1) {
				e1.printStackTrace();
				throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG：" + e1.getMessage());
			}
		}
		return resultSize;
	}

	private Long queryCreatorSize(SearchRequest searchRequest) throws IOException {
		SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
		SearchHits hits = response.getHits();
		return hits.getTotalHits().value;
	}

	/**
	 * 子线程
	 * 
	 * @author Administrator
	 *
	 */
	class ChildDatasetCallable implements Callable<List<JSONObject>> {

		private DatasetEntity childDataset;
		private JSONObject mainDataset;
		private RequestAttributes context;

		public ChildDatasetCallable() {
		}

		public ChildDatasetCallable(DatasetEntity childDataset, JSONObject mainDataset, RequestAttributes context) {
			this.childDataset = childDataset;
			this.mainDataset = mainDataset;
			this.context = context;
		}

		@Override
		public List<JSONObject> call() throws Exception {
			context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
			RequestContextHolder.setRequestAttributes(context);

			List<JSONObject> dataList = new ArrayList<JSONObject>();
			CommonResponse<MdProjectVO> projectResponse = projectApi.queryDetail(childDataset.getMdProjectId());
			if (projectResponse.isSuccess()) {
				MdProjectVO projectVO = projectResponse.getData();
				String childDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";

				JSONObject childParamJson = new JSONObject();
				JSONObject childQueryParam = new JSONObject();
				childQueryParam.put("tenantId", tenantId);
				if (StringUtils.isNotBlank(childDataset.getChildParam())) {
					String[] childParamArr = childDataset.getChildParam().split(",");
					for (String childParamStr : childParamArr) {
						childQueryParam.put(childParamStr, mainDataset.getString(childParamStr));
					}
				}
				childParamJson.put("sqlContent", JdkBase64Util.encode(childDataset.getSqlContent()));
				childParamJson.put("datasetType", "2");
				childParamJson.put("params", childQueryParam);

				String childResponseStr = ReferHttpClientUtils.postByJson(childDatasetUrl, JSON.toJSONString(childParamJson));

				CommonResponse<List<JSONObject>> childResponse = JSON.parseObject(childResponseStr, CommonResponse.class);
				if (childResponse.isSuccess()) {
					return childResponse.getData();
				}
			}
			return dataList;
		}

	}

}
