package com.ejianc.foundation.report.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.controller.param.GridHeader;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.ExcelExportUtil;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.core.util.ResultAsTree;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

@RestController
@RequestMapping("/data/report")
public class DataReportController implements Serializable {

	private static final long serialVersionUID = 6950770890208974786L;
	private final static Integer QUERY_TIMEOUT = 60;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
	@Autowired
	private ITableService tableService;
	@Autowired
	private IColumnService columnService;
	@Autowired
	private RestHighLevelClient client;

	@Value("${common.env.base-host}")
	private String BASE_HOST;

	/**
	 * 报表查询
	 * 
	 * @param tableId
	 * @param queryParam
	 * @return
	 */
	@RequestMapping(value = "/read/{tableId}", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<JSONObject> extractData(HttpServletRequest req,@PathVariable Long tableId,@RequestBody QueryParam queryParam) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		
		TableEntity tableEntity = tableService.selectById(tableId);
		//查询表头
		List<GridHeader> gridHeaders = columnService.queryGridHeadList(tableId, tenantId);
		//查询数据
		queryParam.getParams().put("creator_space", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//拼接查询排序条件
		if(StringUtils.isNotBlank(tableEntity.getOrderItemJson())) {
			List<JSONObject> orderGroups = JSONArray.parseArray(tableEntity.getOrderItemJson(), JSONObject.class);
			JSONObject order = null;
			for(JSONObject o : orderGroups) {
				if(null != o.get("isActive") && o.get("isActive").toString().equals("true")) {
					order = o;
					break;
				}
			}
			if(null == order) {
				order = orderGroups.get(0);
			}
			if(!queryParam.getOrderMap().containsKey(order.get("field").toString())) {
				queryParam.getOrderMap().put(order.get("field").toString(), order.get("sort").toString());
			}
		}
		String url  = tableEntity.getParamUrl();
		try {
			if(StringUtils.isNotBlank(url)) {
				if(url.indexOf("http") < 0) {
					url = BASE_HOST + url;
				}
				String respStr = HttpTookit.get(url, req);
				CommonResponse<JSONObject> params = JSONObject.parseObject(respStr, CommonResponse.class);
				if(!params.isSuccess()) {
					logger.error("请求服务-【{}】获取报表参数失败，", url, params.getMsg());
					return CommonResponse.success("数据查询失败，" + params.getMsg());
				}
				QueryParam q = JSONObject.parseObject(JSONObject.toJSONString(params.getData()), QueryParam.class);
				queryParam.getParams().putAll(q.getParams());
				queryParam.getComplexParams().addAll(q.getComplexParams());
			}
		} catch (Exception e) {
			logger.error("请求服务-【{}】获取报表参数失败，", url, e);
			return CommonResponse.error("数据查询失败，获取查询参数失败！");
		}
		List<String> heightFields = new ArrayList<>();
		for(GridHeader header : gridHeaders) {
            if(header.getVisible() && StringUtils.isNotBlank(header.getCode())) {
                heightFields.add(header.getCode());
            }
        }

		IPage<JSONObject> pageData = columnService.queryPageList(tableEntity.getIndexName(), queryParam, heightFields);

		List<Map> headerList = BeanMapper.mapList(gridHeaders, Map.class);
		JSONObject responseData = new JSONObject();
		responseData.put("reportTitle", tableEntity.getTableName());
		responseData.put("filterItems", tableEntity.getConditionJson());
		responseData.put("orderGroup", tableEntity.getOrderItemJson());
		responseData.put("data", pageData);
		responseData.put("gridheaders", ResultAsTree.createTreeData(headerList));
		return CommonResponse.success(responseData);
	}
	
	@RequestMapping(value = "/export", method = RequestMethod.POST)
	@ResponseBody
	public void exportData(@RequestBody QueryParam queryParam, HttpServletRequest request, HttpServletResponse response) throws IOException {
		Long tenantId = InvocationInfoProxy.getTenantid();
		String tableIdStr = request.getParameter("tableId");
		
		if(StringUtils.isNotBlank(tableIdStr)) {
			Long tableId = Long.parseLong(tableIdStr);
			
			TableEntity tableEntity = tableService.selectById(tableId);
			//查询表头
			List<GridHeader> gridHeaders = columnService.queryGridHeadList(tableId, tenantId);
			List<String> titleList = new ArrayList<String>();
			List<String> properties = new ArrayList<String>();
            List<String> heightFields = new ArrayList<>();
			for(int i=0;i<gridHeaders.size();i++) {
				GridHeader gridHeader = gridHeaders.get(i);
				if(gridHeader.getVisible()) {
					titleList.add(gridHeader.getName());
					properties.add(gridHeader.getCode());
					if(StringUtils.isNotBlank(gridHeader.getCode())) {
                        heightFields.add(gridHeader.getCode());
                    }
				}
			}
			ExcelExportUtil excelExport = new ExcelExportUtil();
			Long dataSize = queryIndexSize(tableEntity.getIndexName(), tenantId, queryParam);
			if(dataSize > 0) {
				queryParam.setPageSize(dataSize.intValue());
				queryParam.getParams().put("creator_space", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
				IPage<JSONObject> pageData = columnService.queryPageList(tableEntity.getIndexName(), queryParam, heightFields);
				excelExport.setData(pageData.getRecords());
			}
			if(properties.size() > 0) {
				excelExport.setHeardKey(properties.toArray(new String[properties.size()]));
				excelExport.setFontSize(12);
				excelExport.setSheetName(tableEntity.getTableName());
				excelExport.setTitle(tableEntity.getTableName());
				excelExport.setHeardList(titleList.toArray(new String[titleList.size()]));
				excelExport.exportExport(request, response);
			}
		}
	}
	
	/**
	 * 查询当前空间的所有数据
	 * 
	 * @param indexName
	 * @param creatorSpace
	 * @return
	 */
	private Long queryIndexSize(String indexName, Long creatorSpace,QueryParam queryParam) {
		Long resultSize = 0L;
		SearchRequest searchRequest = new SearchRequest(indexName);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
		boolQuery.must(QueryBuilders.termQuery("creator_space", creatorSpace));
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			if(param.getValue() == null || StringUtils.isBlank(param.getValue() + "")) {
				continue;
			}
			if(QueryParam.EQ.equals(param.getType())) {
				boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString()));
			} else if(QueryParam.LIKE.equals(param.getType())) {
				boolQuery.must(QueryBuilders.matchQuery(entry.getKey()+".keyword", param.getValue().toString()));
			} else if(QueryParam.IN.equals(param.getType())) {
				boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString().split(",")));
			} else if(QueryParam.BETWEEN.equals(param.getType())) {
				String[] dataArr = param.getValue().toString().split(",");
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true));
			} else if(QueryParam.GT.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue().toString()));
			} else if(QueryParam.GE.equals(param.getType())){
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue().toString()));
			} else if(QueryParam.LT.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue().toString()));
			} else if(QueryParam.LE.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue().toString()));
			}
		}
		sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        try {
        	resultSize = queryCreatorSize(searchRequest);
        } catch (IOException e) {
        	try { //重试一次
        		resultSize = queryCreatorSize(searchRequest);
			} catch (IOException e1) {
				e1.printStackTrace();
        		throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e1.getMessage());
			}
        }
        return resultSize;
	}
	
	private Long queryCreatorSize(SearchRequest searchRequest) throws IOException {
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHits hits = response.getHits();
        return hits.getTotalHits().value;
	}

}
