package com.ejianc.foundation.report.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.controller.param.GridHeader;
import com.ejianc.foundation.report.mapper.ColumnMapper;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.CalculatorUtils;
import com.ejianc.foundation.report.util.PinYinUtil;
import com.ejianc.foundation.report.vo.ColumnVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.ComplexParam;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseEntity;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.Requests;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.text.Text;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightField;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.TimeUnit;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;

@Service
public class ColumnServiceImpl extends BaseServiceImpl<ColumnMapper, ColumnEntity> implements IColumnService {

	private final static Integer QUERY_TIMEOUT = 60;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
	@Autowired
	private ColumnMapper columnMapper;
	@Autowired(required=false)
	private RestHighLevelClient client;
	@Autowired
	private ITableService tableService;
	@Value("${oms.sysUserCode}")
	private String sysUserCode;
	
	@Override
	public void deleteByIds(List<Long> ids) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		for(Long id:ids) {
			ColumnEntity columnEntity = columnMapper.selectById(id);
			if(tenantId.equals(columnEntity.getTenantId())) {
				columnMapper.deleteById(id);
			}else{
				throw new BusinessException("您没有权限删除此列");
			}
		}
	}

	@Override
	public IPage<ColumnEntity> queryPage(QueryParam queryParam) {
		Map<String, Parameter> paramMap = queryParam.getParams();
		
		Map<String, Object> condition = new HashMap<String, Object>();
		for(Map.Entry<String, Parameter> entry:paramMap.entrySet()) {
			condition.put(entry.getKey(), entry.getValue().getValue());
		}
		condition.put("pageIndex", (queryParam.getPageIndex()-1)*queryParam.getPageSize());
		condition.put("pageSize", queryParam.getPageSize());
		
		List<ColumnEntity> records = columnMapper.queryList(condition);
		Long count = columnMapper.queryCount(condition);
		
		IPage<ColumnEntity> page = new Page<>();
		page.setCurrent(queryParam.getPageIndex());
		page.setSize(queryParam.getPageSize());
		page.setTotal(count);
		page.setRecords(records);
		return page;
	}

	@Override
	public List<GridHeader> queryGridHeadList(Long tableId, Long tenantId) {
		List<GridHeader> gridHeadList = columnMapper.queryGridHeadList(tableId, tenantId);
		return gridHeadList;
	}

	@Override
	public IPage<JSONObject> queryPageList(String indexName, QueryParam queryParam, List<String> heightFields) {
		IPage<JSONObject> page = new Page<>();
		
		SearchRequest searchRequest = new SearchRequest(indexName);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		HighlightBuilder highlightBuilder = new HighlightBuilder();
        highlightBuilder.preTags("<span style=\"color:red\">");
        highlightBuilder.postTags("</span>");
        if(CollectionUtils.isNotEmpty(heightFields)) {
        	for(String field : heightFields) {
				highlightBuilder.field(field);
			}
        } else {
            highlightBuilder.field("*").requireFieldMatch(false);
        }
        sourceBuilder.highlighter(highlightBuilder);
		
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
		boolQuery = setParam(boolQuery, paramMap);

		//复杂查询
        if(CollectionUtils.isNotEmpty(queryParam.getComplexParams())) {
            boolQuery.must(parseComplexParams(queryParam.getComplexParams()));
        }

		sourceBuilder.query(boolQuery);
		Map<String, String> orderMap = queryParam.getOrderMap();
		if(orderMap.isEmpty()) {
			//强制指定sort类型，解决es创建索引后，随指定了_default_mapping但未初始化导致的查询时排序报错的问题。
			 sourceBuilder.sort(new FieldSortBuilder("data_sequence").unmappedType("long").order(SortOrder.ASC));
		} else {
			for(String key : queryParam.getOrderMap().keySet()) {
				sourceBuilder.sort(new FieldSortBuilder(key).order("asc".equals(orderMap.get(key)) ? SortOrder.ASC : SortOrder.DESC));
			}
		}
		sourceBuilder.from(queryParam.getPageIndex() <= 0 ? 0 : (queryParam.getPageIndex()-1)*queryParam.getPageSize());
        sourceBuilder.size(queryParam.getPageSize());
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        try {
        	page = queryPage(paramMap, searchRequest);
        } catch (IOException e) {
        	try { //重试一次
				page = queryPage(paramMap, searchRequest);
			} catch (IOException e1) {
				e1.printStackTrace();
        		throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e1.getMessage());
			}
        }
        return page;
	}

	private BoolQueryBuilder setParam(BoolQueryBuilder boolQuery, Map<String, Parameter> paramMap) {
        for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
            Parameter param = entry.getValue();
            if(param.getValue() == null || StringUtils.isBlank(param.getValue() + "")) {
                continue;
            }
            if(QueryParam.EQ.equals(param.getType())) {
                boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString()));
            } else if(QueryParam.LIKE.equals(param.getType())) {
                boolQuery.must(QueryBuilders.matchQuery(entry.getKey(), param.getValue().toString()).analyzer("ik_smart"));
            } else if(QueryParam.IN.equals(param.getType())) {
                Object inParam = param.getValue();
                if(inParam instanceof String) {
                    boolQuery.must(QueryBuilders.termsQuery(entry.getKey(), Arrays.asList(inParam.toString().split(","))));
                } else if(inParam instanceof JSONArray) {
                    boolQuery.must(QueryBuilders.termsQuery(entry.getKey(), JSONObject.parseArray(JSONObject.toJSONString(inParam), Long.class)));
                }
            } else if(QueryParam.BETWEEN.equals(param.getType())) {
                String[] dataArr = param.getValue().toString().split(",");
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true));
//                boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true));
            } else if(QueryParam.GT.equals(param.getType())) {
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue().toString()));
            } else if(QueryParam.GE.equals(param.getType())){
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue().toString()));
            } else if(QueryParam.LT.equals(param.getType())) {
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue().toString()));
            } else if(QueryParam.LE.equals(param.getType())) {
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue().toString()));
            }
        }
		return boolQuery;
    }

    private BoolQueryBuilder parseComplexParams(List<ComplexParam> complexParams) {
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
	    for(ComplexParam cp : complexParams) {
			if(ComplexParam.AND.equals(cp.getLogic())) {
				if(!cp.getParams().isEmpty()) {
					boolQuery.must(setParam(QueryBuilders.boolQuery(), cp.getParams()));
				}
				if(CollectionUtils.isNotEmpty(cp.getComplexParams())) {
					boolQuery.must(parseComplexParams(cp.getComplexParams()));
				}

			} else {
				if(!cp.getParams().isEmpty()) {
					boolQuery.should(setParam(QueryBuilders.boolQuery(), cp.getParams()));
				}
				if(CollectionUtils.isNotEmpty(cp.getComplexParams())) {
					boolQuery.should(parseComplexParams(cp.getComplexParams()));
				}
			}

        }

	    return boolQuery;
    }

    //查询es
	private IPage<JSONObject> queryPage(Map<String, Parameter> paramMap, SearchRequest searchRequest) throws IOException {
		IPage<JSONObject> page = new Page<>();
		List<JSONObject> list = new ArrayList<>();
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        for(SearchHit hit : response.getHits()){
            Map<String, Object> source = hit.getSourceAsMap();
            //处理高亮片段
            Map<String, HighlightField> highlightFields = hit.getHighlightFields();
            for(Map.Entry<String, HighlightField> highlightEntry:highlightFields.entrySet()) {
            	if(paramMap.containsKey(highlightEntry.getKey())) {
            		Parameter parameter = paramMap.get(highlightEntry.getKey());
            		if(StringUtils.isNotBlank(parameter.getValue() + "")) {
            			HighlightField nameField = highlightEntry.getValue();
            			if(nameField!=null){
            				Text[] fragments = nameField.fragments();
            				StringBuilder nameTmp = new StringBuilder();
            				for(Text text:fragments){
            					nameTmp.append(text);
            				}
            				//将高亮片段组装到结果中去
            				source.put(highlightEntry.getKey(), nameTmp.toString());
            			}
            		}
            	}
            }
        }
        SearchHits hits = response.getHits();
        for (SearchHit hit : hits) {
            list.add(new JSONObject(hit.getSourceAsMap()));
        }
        page.setRecords(list);
        page.setTotal(hits.getTotalHits().value);
        return page;
	}


	@Override
	public List<ColumnEntity> queryTenantColumnList(Map<String, Object> paramMap) {
		List<ColumnEntity> columnList = columnMapper.queryTenantColumnList(paramMap);
		return columnList;
	}

	@Override
	public List<ColumnEntity> queryFormulaList(Long tableId) {
		List<ColumnEntity> columnEntities = columnMapper.queryFormulaList(tableId,InvocationInfoProxy.getTenantid());
		return columnEntities;
	}


	private String saveOrUpdateColumnBatch(List<ColumnVO> columnVos, boolean resetBaseInfo) {
		TableEntity tableEntity = tableService.getById(columnVos.get(0).getTableId());
		List<ColumnEntity> saveList = new ArrayList<>();
		try {
			XContentBuilder mapping = null;
			for(ColumnVO columnVo : columnVos) {
				if(null == mapping) {
					mapping = jsonBuilder();
				}
				switch (columnVo.getType()) {
					case "string":
						mapping.startObject().startObject(tableEntity.getIndexName()).startObject("properties")
								.startObject(columnVo.getProperty())
								.field("type", "text")
								.field("analyzer","ik_max_word")
								.field("search_analyzer","ik_smart")
								.endObject()
								.endObject()
								.endObject()
								.endObject();
						break;
					case "number":
						mapping.startObject().startObject(tableEntity.getIndexName()).startObject("properties")
								.startObject(columnVo.getProperty())
								.field("type", "long")
								.endObject()
								.endObject()
								.endObject()
								.endObject();
						break;
					case "time":
						mapping.startObject().startObject(tableEntity.getIndexName()).startObject("properties")
								.startObject(columnVo.getProperty())
								.field("type", "date")
								.field("format","yyyy-MM-dd HH:mm:ss")
								.endObject()
								.endObject()
								.endObject()
								.endObject();
						break;
				}

				ColumnEntity uniqueBean = null;
				if(columnVo.getId() != null && columnVo.getId() > 0) {
					uniqueBean = columnMapper.selectById(columnVo.getId());
				}

				if(null != uniqueBean) {
					uniqueBean.setType(columnVo.getType());
					uniqueBean.setColumnName(columnVo.getColumnName());
					if(StringUtils.isNotBlank(columnVo.getFormula())) {
						List<ColumnEntity> columnList = this.queryFormulaList(columnVo.getTableId());
						Map<String, String> propertyMap = new HashMap<>();
						for(ColumnEntity columnEntity:columnList) {
							propertyMap.put(columnEntity.getProperty(), columnEntity.getProperty());
						}
						Boolean checked = CalculatorUtils.simpleMathFormuaCheck(columnVo.getFormula(), propertyMap);
						if(!checked) {
							return "公式错误，不允许保存";
						}
						String property = PinYinUtil.getFullSpell(columnVo.getColumnName());
						uniqueBean.setProperty(property);
					}else{
						uniqueBean.setProperty(columnVo.getProperty());
					}
					uniqueBean.setFormula(columnVo.getFormula());
					uniqueBean.setSearchFlag(columnVo.getSearchFlag());
					uniqueBean.setVisible(columnVo.getVisible());
					uniqueBean.setSequence(columnVo.getSequence());
					uniqueBean.setFormatter(columnVo.getFormatter());
					uniqueBean.setEnableSummarize(columnVo.getEnableSummarize());
					uniqueBean.setExportFormat(columnVo.getExportFormat());
					uniqueBean.setAlignType(columnVo.getAlignType());
					//唯一性校验
					QueryWrapper<ColumnEntity> queryWrapper = new QueryWrapper<>();
					queryWrapper.eq("table_id", uniqueBean.getTableId());
					queryWrapper.eq("property", uniqueBean.getProperty());
					queryWrapper.ne("id", columnVo.getId());
					List<ColumnEntity> list = this.list(queryWrapper );
					if(list!=null&&list.size()>0){
						if(StringUtils.isNotBlank(uniqueBean.getFormula())){
							return "列名:"+uniqueBean.getColumnName()+"已存在";
						}else{
							return "属性:"+uniqueBean.getProperty()+"已存在";
						}
					}

					saveList.add(uniqueBean);
				}else{
					ColumnEntity saveBean = BeanMapper.map(columnVo, ColumnEntity.class);
					if(resetBaseInfo) {
						this.resetBaseInfo(saveBean);
					}

					if(StringUtils.isNotBlank(columnVo.getFormula())) {
						List<ColumnEntity> columnList = this.queryFormulaList(columnVo.getTableId());
						Map<String, String> propertyMap = new HashMap<>();
						for(ColumnEntity columnEntity:columnList) {
							propertyMap.put(columnEntity.getProperty(), columnEntity.getProperty());
						}
						Boolean checked = CalculatorUtils.simpleMathFormuaCheck(columnVo.getFormula(), propertyMap);
						if(!checked) {
							return "公式错误，不允许保存";
						}
						String property = PinYinUtil.getFullSpell(columnVo.getColumnName());
						saveBean.setProperty(property);
					}else{
						saveBean.setProperty(columnVo.getProperty());
					}
					if(columnVo.getSearchFlag() == null) {
						saveBean.setSearchFlag(0);
					}
					//唯一性校验
					QueryWrapper<ColumnEntity> queryWrapper = new QueryWrapper<>();
					queryWrapper.eq("table_id", saveBean.getTableId());
					queryWrapper.eq("property", saveBean.getProperty());
					List<ColumnEntity> list = this.list(queryWrapper);
					if(list!=null&&list.size()>0){
						if(StringUtils.isNotBlank(saveBean.getFormula())){
							return "列名:"+saveBean.getColumnName()+"已存在";
						}else{
							return "属性:"+saveBean.getProperty()+"已存在";
						}
					}

					saveList.add(saveBean);
				}
			}

			PutMappingRequest mappingRequest = Requests.putMappingRequest(tableEntity.getIndexName()).type(tableEntity.getIndexName()).source(mapping);
			client.indices().putMapping(mappingRequest, RequestOptions.DEFAULT);

			this.saveOrUpdateBatch(saveList);
		} catch (Exception e) {
			logger.error("新增/更新ES字段异常,", e);
			throw new BusinessException("保存失败，字段新增/更新到ES失败！");
		}
		return null;
	}

	@Override
	@Transactional(rollbackFor = Exception.class)
	public String saveOrUpdateColumn(ColumnVO columnVo) {
		List<ColumnVO> cols = new ArrayList<>();
		cols.add(columnVo);
		return this.saveOrUpdateColumnBatch(cols, false);
	}

	@Override
	@Transactional(rollbackFor = Exception.class)
	public String savePubColsData(List<ColumnVO> cols) {
		return this.saveOrUpdateColumnBatch(cols, true);
	}

	private void resetBaseInfo(BaseEntity entity) {
		entity.setCreateUserCode(sysUserCode);
		entity.setUpdateTime(null);
		entity.setUpdateUserCode(sysUserCode);
		entity.setVersion(0);
	}

}
