package com.ejianc.foundation.report.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.controller.param.GridHeader;
import com.ejianc.foundation.report.mapper.ColumnMapper;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.CalculatorUtils;
import com.ejianc.foundation.report.util.PinYinUtil;
import com.ejianc.foundation.report.vo.ColumnVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.ComplexParam;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseEntity;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.Requests;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.text.Text;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightField;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;

@Service
public class ColumnServiceImpl extends BaseServiceImpl<ColumnMapper, ColumnEntity> implements IColumnService {

	private final static Integer QUERY_TIMEOUT = 60;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
	@Autowired
	private ColumnMapper columnMapper;
	@Autowired(required=false)
	private RestHighLevelClient client;
	@Autowired
	private ITableService tableService;
	@Value("${oms.sysUserCode}")
	private String sysUserCode;

	@Override
	public void deleteByIds(List<Long> ids) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		for(Long id:ids) {
			ColumnEntity columnEntity = columnMapper.selectById(id);
			if(tenantId.equals(columnEntity.getTenantId())) {
				columnMapper.deleteById(id);
			}else{
				throw new BusinessException("您没有权限删除此列");
			}
		}
	}

	@Override
	public IPage<ColumnEntity> queryPage(QueryParam queryParam) {
		Map<String, Parameter> paramMap = queryParam.getParams();
		
		Map<String, Object> condition = new HashMap<String, Object>();
		for(Map.Entry<String, Parameter> entry:paramMap.entrySet()) {
			condition.put(entry.getKey(), entry.getValue().getValue());
		}
		condition.put("pageIndex", (queryParam.getPageIndex()-1)*queryParam.getPageSize());
		condition.put("pageSize", queryParam.getPageSize());
		
		List<ColumnEntity> records = columnMapper.queryList(condition);
		Long count = columnMapper.queryCount(condition);
		
		IPage<ColumnEntity> page = new Page<>();
		page.setCurrent(queryParam.getPageIndex());
		page.setSize(queryParam.getPageSize());
		page.setTotal(count);
		page.setRecords(records);
		return page;
	}

	@Override
	public List<GridHeader> queryGridHeadList(Long tableId, Long tenantId) {
		List<GridHeader> gridHeadList = columnMapper.queryGridHeadList(tableId, tenantId);
		return gridHeadList;
	}

	@Override
	public IPage<JSONObject> queryPageList(String indexName, QueryParam queryParam, List<String> heightFields, Map<String, String> filedTypeMap)  {
		IPage<JSONObject> page = new Page<>();
		
		SearchRequest searchRequest = new SearchRequest(indexName);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		HighlightBuilder highlightBuilder = new HighlightBuilder();
        highlightBuilder.preTags("<span style=\"color:red\">");
        highlightBuilder.postTags("</span>");
        if(CollectionUtils.isNotEmpty(heightFields)) {
        	for(String field : heightFields) {
				highlightBuilder.field(field);
			}
        } else {
            highlightBuilder.field("*").requireFieldMatch(false);
        }
        sourceBuilder.highlighter(highlightBuilder);

		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        try {
			boolQuery = setParam(boolQuery, paramMap, filedTypeMap);

			//复杂查询
			if(CollectionUtils.isNotEmpty(queryParam.getComplexParams())) {
				boolQuery.must(parseComplexParams(queryParam.getComplexParams(), filedTypeMap));
			}

			sourceBuilder.query(boolQuery);
			Map<String, String> orderMap = queryParam.getOrderMap();
			if(orderMap.isEmpty()) {
				//强制指定sort类型，解决es创建索引后，随指定了_default_mapping但未初始化导致的查询时排序报错的问题。
				sourceBuilder.sort(new FieldSortBuilder("data_sequence").unmappedType("long").order(SortOrder.ASC));
			} else {
				for(String key : orderMap.keySet()) {
					sourceBuilder.sort(new FieldSortBuilder(key)
							.unmappedType("number".equals(filedTypeMap.get(key)) ? "long" : "time".equals(filedTypeMap.get(key)) ? "date" : "text").order("asc".equals(orderMap.get(key)) ? SortOrder.ASC : SortOrder.DESC));
				}
			}
			sourceBuilder.from(queryParam.getPageIndex() <= 0 ? 0 : (queryParam.getPageIndex()-1)*queryParam.getPageSize());
			sourceBuilder.size(queryParam.getPageSize());
			sourceBuilder.trackTotalHits(true);
			sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
			searchRequest.source(sourceBuilder);

        	page = queryPage(paramMap, searchRequest);
        } catch (Exception e) {
        	try { //重试一次
				page = queryPage(paramMap, searchRequest);
			} catch (Exception e1) {
				e1.printStackTrace();
        		throw new BusinessException("查询全部记录索引失败，MSG：", e1);
			}
        }
        return page;
	}

	private BoolQueryBuilder setParam(BoolQueryBuilder boolQuery, Map<String, Parameter> paramMap, Map<String, String> filedTypeMap) throws Exception {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
            Parameter param = entry.getValue();
            if(param.getValue() == null || StringUtils.isBlank(param.getValue() + "")) {
                continue;
            }
            if(QueryParam.EQ.equals(param.getType())) {
                boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString()));
            } else if(QueryParam.LIKE.equals(param.getType())) {
                boolQuery.must(QueryBuilders.matchQuery(entry.getKey(), param.getValue().toString()).analyzer("ik_smart"));
            } else if(QueryParam.IN.equals(param.getType())) {
                Object inParam = param.getValue();
                if(inParam instanceof String) {
                    boolQuery.must(QueryBuilders.termsQuery(entry.getKey(), Arrays.asList(inParam.toString().split(","))));
                } else if(inParam instanceof JSONArray) {
                    boolQuery.must(QueryBuilders.termsQuery(entry.getKey(), JSONObject.parseArray(JSONObject.toJSONString(inParam), Long.class)));
                }
            } else if(QueryParam.BETWEEN.equals(param.getType())) {
                String[] dataArr = param.getValue().toString().split(",");
                Object v1 = dataArr[0], v2 = dataArr[1];
                if("time".equals(filedTypeMap.get(entry.getKey()))) {

                	//日期格式转为yyyy-MM-dd HH:mm:ss
                	v1 = null != v1 && StringUtils.isNotBlank(v1.toString()) ? sdf.format(DateUtil.parse(v1.toString())) : v1;
                	v2 = null != v2 && StringUtils.isNotBlank(v2.toString()) ? sdf.format(DateUtil.parse(v2.toString())) : v2;
				}
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).from(v1).to(v2).includeLower(true).includeUpper(true));
//                boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true));
            } else if(QueryParam.GT.equals(param.getType())) {
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue().toString()));
            } else if(QueryParam.GE.equals(param.getType())){
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue().toString()));
            } else if(QueryParam.LT.equals(param.getType())) {
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue().toString()));
            } else if(QueryParam.LE.equals(param.getType())) {
                boolQuery.filter().add(QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue().toString()));
            }
        }
		return boolQuery;
    }

    private BoolQueryBuilder parseComplexParams(List<ComplexParam> complexParams, Map<String, String> filedTypeMap) throws Exception {
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
	    for(ComplexParam cp : complexParams) {
			if(ComplexParam.AND.equals(cp.getLogic())) {
				if(!cp.getParams().isEmpty()) {
					boolQuery.must(setParam(QueryBuilders.boolQuery(), cp.getParams(), filedTypeMap));
				}
				if(CollectionUtils.isNotEmpty(cp.getComplexParams())) {
					boolQuery.must(parseComplexParams(cp.getComplexParams(), filedTypeMap));
				}
			} else {
				if(!cp.getParams().isEmpty()) {
					boolQuery.should(setParam(QueryBuilders.boolQuery(), cp.getParams(), filedTypeMap));
				}
				if(CollectionUtils.isNotEmpty(cp.getComplexParams())) {
					boolQuery.should(parseComplexParams(cp.getComplexParams(), filedTypeMap));
				}
			}
        }

	    return boolQuery;
    }

    //查询es
	private IPage<JSONObject> queryPage(Map<String, Parameter> paramMap, SearchRequest searchRequest) throws IOException {
		IPage<JSONObject> page = new Page<>();
		List<JSONObject> list = new ArrayList<>();
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        for(SearchHit hit : response.getHits()){
            Map<String, Object> source = hit.getSourceAsMap();
            //处理高亮片段
            Map<String, HighlightField> highlightFields = hit.getHighlightFields();
            for(Map.Entry<String, HighlightField> highlightEntry:highlightFields.entrySet()) {
            	if(paramMap.containsKey(highlightEntry.getKey())) {
            		Parameter parameter = paramMap.get(highlightEntry.getKey());
            		if(StringUtils.isNotBlank(parameter.getValue() + "")) {
            			HighlightField nameField = highlightEntry.getValue();
            			if(nameField!=null){
            				Text[] fragments = nameField.fragments();
            				StringBuilder nameTmp = new StringBuilder();
            				for(Text text:fragments){
            					nameTmp.append(text);
            				}
            				//将高亮片段组装到结果中去
            				source.put(highlightEntry.getKey(), nameTmp.toString());
            			}
            		}
            	}
            }
        }
        SearchHits hits = response.getHits();
        for (SearchHit hit : hits) {
            list.add(new JSONObject(hit.getSourceAsMap()));
        }
        page.setRecords(list);
        page.setTotal(hits.getTotalHits().value);
        return page;
	}


	@Override
	public List<ColumnEntity> queryTenantColumnList(Map<String, Object> paramMap) {
		List<ColumnEntity> columnList = columnMapper.queryTenantColumnList(paramMap);
		return columnList;
	}

	@Override
	public List<ColumnEntity> queryFormulaList(Long tableId) {
		List<ColumnEntity> columnEntities = columnMapper.queryFormulaList(tableId,InvocationInfoProxy.getTenantid());
		return columnEntities;
	}


	private String saveOrUpdateColumnBatch(List<ColumnVO> columnVos, boolean resetBaseInfo) {
		TableEntity tableEntity = tableService.getById(columnVos.get(0).getTableId());
		List<ColumnEntity> saveList = new ArrayList<>();
		try {
			XContentBuilder mapping = null;
			for(ColumnVO columnVo : columnVos) {
				ColumnEntity uniqueBean = null;
				if(columnVo.getId() != null && columnVo.getId() > 0) {
					uniqueBean = columnMapper.selectById(columnVo.getId());
				}

				if(null != uniqueBean) {
					uniqueBean.setType(columnVo.getType());
					uniqueBean.setColumnName(columnVo.getColumnName());
					if(StringUtils.isNotBlank(columnVo.getFormula())) {
						List<ColumnEntity> columnList = this.queryFormulaList(columnVo.getTableId());
						Map<String, String> propertyMap = new HashMap<>();
						for(ColumnEntity columnEntity:columnList) {
							propertyMap.put(columnEntity.getProperty(), columnEntity.getProperty());
						}
						Boolean checked = CalculatorUtils.simpleMathFormuaCheck(columnVo.getFormula(), propertyMap);
						if(!checked) {
							return "公式错误，不允许保存";
						}
						String property = PinYinUtil.getFullSpell(columnVo.getColumnName());
						uniqueBean.setProperty(property);
						columnVo.setProperty(property);
					}else{
						uniqueBean.setProperty(columnVo.getProperty());
					}
					uniqueBean.setFormula(columnVo.getFormula());
					uniqueBean.setSearchFlag(columnVo.getSearchFlag());
					uniqueBean.setVisible(columnVo.getVisible());
					uniqueBean.setColFixed(columnVo.getColFixed());
					uniqueBean.setColFontBold(columnVo.getColFontBold());
					uniqueBean.setColBgColor(columnVo.getColBgColor());
					uniqueBean.setColFontColor(columnVo.getColFontColor());
					uniqueBean.setSequence(columnVo.getSequence());
					uniqueBean.setFormatter(columnVo.getFormatter());
					uniqueBean.setEnableSummarize(columnVo.getEnableSummarize());
					uniqueBean.setExportFormat(columnVo.getExportFormat());
					uniqueBean.setAlignType(columnVo.getAlignType());
					uniqueBean.setSuffixStr(columnVo.getSuffixStr());
					//唯一性校验
//					QueryWrapper<ColumnEntity> queryWrapper = new QueryWrapper<>();
//					queryWrapper.eq("table_id", uniqueBean.getTableId());
//					queryWrapper.eq("property", uniqueBean.getProperty());
//					queryWrapper.eq("tenant_id", InvocationInfoProxy.getTenantid());
//					queryWrapper.ne("id", columnVo.getId());
//					List<ColumnEntity> list = this.list(queryWrapper );
//					if(list!=null&&list.size()>0){
//						if(StringUtils.isNotBlank(uniqueBean.getFormula())){
//							return "列名:"+uniqueBean.getColumnName()+"已存在";
//						}else{
//							return "属性:"+uniqueBean.getProperty()+"已存在";
//						}
//					}
					//唯一性校验
					String checkResult = uniqueCheck(uniqueBean);
					if(StringUtils.isNotBlank(checkResult)) {
						return checkResult;
					}

					saveList.add(uniqueBean);
				}else{
					ColumnEntity saveBean = BeanMapper.map(columnVo, ColumnEntity.class);
					if(resetBaseInfo) {
						this.resetBaseInfo(saveBean);
					}

					if(StringUtils.isNotBlank(columnVo.getFormula())) {
						List<ColumnEntity> columnList = this.queryFormulaList(columnVo.getTableId());
						Map<String, String> propertyMap = new HashMap<>();
						for(ColumnEntity columnEntity:columnList) {
							propertyMap.put(columnEntity.getProperty(), columnEntity.getProperty());
						}
						Boolean checked = CalculatorUtils.simpleMathFormuaCheck(columnVo.getFormula(), propertyMap);
						if(!checked) {
							return "公式错误，不允许保存";
						}
						String property = PinYinUtil.getFullSpell(columnVo.getColumnName());
						saveBean.setProperty(property);
						columnVo.setProperty(property);
					}else{
						saveBean.setProperty(columnVo.getProperty());
					}

					if(columnVo.getSearchFlag() == null) {
						saveBean.setSearchFlag(0);
					}
					//唯一性校验
					String checkResult = uniqueCheck(saveBean);
					if(StringUtils.isNotBlank(checkResult)) {
						return checkResult;
					}
					saveList.add(saveBean);
				}

				if(null == mapping) {
					mapping = jsonBuilder().startObject().startObject(tableEntity.getIndexName()).startObject("properties");
				}
				configObject(columnVo.getType(), columnVo.getProperty(), mapping);
			}

            mapping.endObject().endObject().endObject();
			PutMappingRequest mappingRequest = Requests.putMappingRequest(tableEntity.getIndexName()).type(tableEntity.getIndexName()).source(mapping);
			client.indices().putMapping(mappingRequest, RequestOptions.DEFAULT);

			this.saveOrUpdateBatch(saveList);
		} catch (Exception e) {
			logger.error("新增/更新ES字段异常,", e);
			throw new BusinessException("保存失败，字段新增/更新到ES失败！");
		}
		return null;
	}

	public String uniqueCheck(ColumnEntity checkBean) {
		//唯一性校验
		QueryWrapper<ColumnEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.eq("table_id", checkBean.getTableId());
		queryWrapper.eq("property", checkBean.getProperty());
		queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
		if(null != checkBean.getId()) {
			queryWrapper.ne("id", checkBean.getId());
		}
		List<ColumnEntity> list = this.list(queryWrapper);

		if(CollectionUtils.isEmpty(list)) {
			return null;
		}
		//目前租户只能新增计算列，故普通列若属性重复则直接返回保存失败
		if(StringUtils.isBlank(checkBean.getFormula())) {
			return "属性:" + checkBean.getProperty()+"已存在";
		} else {
			//若为计算列时，则判断：若同一租户下存在属性相同的列则返回保存失败，否则修改列属性字段，使其可正常保存
			List<ColumnEntity> tenantCol = list.stream().filter(col -> col.getTenantId().equals(InvocationInfoProxy.getTenantid())).collect(Collectors.toList());
			if(CollectionUtils.isNotEmpty(tenantCol)) {
				return "列名:" + checkBean.getColumnName()+"已存在";
			}
		}

		return null;
	}

	@Override
	@Transactional(rollbackFor = Exception.class)
	public String saveOrUpdateColumn(ColumnVO columnVo) {
		List<ColumnVO> cols = new ArrayList<>();
		cols.add(columnVo);
		return this.saveOrUpdateColumnBatch(cols, false);
	}

	@Override
	@Transactional(rollbackFor = Exception.class)
	public String savePubColsData(List<ColumnVO> cols) {
		return this.saveOrUpdateColumnBatch(cols, true);
	}

	@Override
	public void syncColumnsBatch(List<ColumnEntity> columns) {
		TableEntity tableEntity = tableService.getById(columns.get(0).getTableId());
		XContentBuilder mapping = null;
		try {
			mapping = jsonBuilder().startObject().startObject(tableEntity.getIndexName()).startObject("properties");

			for(ColumnEntity column : columns) {
				configObject(column.getType(), column.getProperty(), mapping);
			}

			mapping.endObject().endObject().endObject();
			PutMappingRequest mappingRequest = Requests.putMappingRequest(tableEntity.getIndexName()).type(tableEntity.getIndexName()).source(mapping);
			client.indices().putMapping(mappingRequest, RequestOptions.DEFAULT);

		} catch (Exception e) {
			logger.error("更新ES字段异常,", e);
			throw new BusinessException("字段跟新失败，字段更新到ES失败！");
		}
	}

	private void configObject(String type, String propertity, XContentBuilder mapping) throws Exception {
		switch (type) {
			case "string":
				mapping.startObject(propertity)
						.field("type", "text")
						.field("analyzer","ik_max_word")
						.field("search_analyzer","ik_smart")
						.endObject();
				break;
			case "number":
				mapping.startObject(propertity)
						.field("type", "long")
						.endObject();
				break;
			case "time":
				mapping.startObject(propertity)
						.field("type", "date")
						.field("format","yyyy-MM-dd HH:mm:ss")
						.endObject();
				break;
			default:
				break;
		}
	}

	private void resetBaseInfo(BaseEntity entity) {
		entity.setCreateUserCode(sysUserCode);
		entity.setUpdateTime(null);
		entity.setUpdateUserCode(sysUserCode);
		entity.setVersion(0);
	}
}
