package com.ejianc.foundation.report.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.metadata.vo.MdProjectVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.DatasetEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.custom.vo.FinanceUseQueryVO;
import com.ejianc.foundation.report.custom.vo.FinanceUseVO;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.IDatasetService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.CalculatorUtils;
import com.ejianc.foundation.report.util.EJCDateUtil;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.util.JdkBase64Util;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedLongTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.ParsedSum;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 自定义报表查询
 * @date 2022/4/14
 */
@RestController
@RequestMapping("/customQuery")
public class CustomQueryController {

    private static final Logger logger = LoggerFactory.getLogger(CustomQueryController.class);

    @Value("${common.env.base-host}")
    private String baseHost;

    private final static Integer BATCH_SIZE = 200;


    /**
     * 公司领导门户 - 资金使用查询
     */
    private static final String INDEX_EJC_FINANCE_USE = "ejc_finance_use";

    private final static Integer QUERY_TIMEOUT = 60;

    @Autowired
    private IDatasetService datasetService;

    @Autowired
    private ITableService tableService;
    @Autowired
    private IColumnService columnService;
    @Autowired
    private RestHighLevelClient client;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private SessionManager sessionManager;


    /**
     * @param financeUseQueryVO
     * @description: 公司领导门户资金使用
     * * "production":产值统计1
     * * "partAReport":甲方报量1
     * * "yingshouweishou":应收未收 = 甲方报量 - 累计收款
     * * "backMoney":回款 = 累计收款1
     * * "openInvince":"开票
     * * "jiesuanjine":结算金额1
     * * "receiveInvince":收票金额
     * * "payMoney":"付款金额1
     * * "yingfuweifu":"应付未付 = 结算金额 - 付款金额
     * * "zhangmianjine":账面金额 = 回款 - 付款金额
     * * "monthData":['2021-09', '2021-10', '2021-11', '2021-12', '2022-01', '2022-02', '2022-03']
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.income.vo.FinanceUseVO>
     * @author songlx
     * @date: 2022/4/14
     */
    @RequestMapping(value = "/queryFinanceUse", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<FinanceUseVO> queryFinanceUse(@RequestBody FinanceUseQueryVO financeUseQueryVO) {
        SearchRequest searchRequest = new SearchRequest(INDEX_EJC_FINANCE_USE);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.from(0);
        sourceBuilder.size(10000);
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));

        if (InvocationInfoProxy.getOrgId() != null) {
            boolQuery.must(QueryBuilders.termsQuery("orgId", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            UserContext userContext = sessionManager.getUserContext();
            String authOrgIds = userContext.getAuthOrgIds();
            if (StringUtils.isNotEmpty(authOrgIds)) {
                String[] orgIds = authOrgIds.split(",");
                boolQuery.must(QueryBuilders.termsQuery("orgId", orgIds));
            }
        }


        if (financeUseQueryVO.isThisYear()) {
            boolQuery.must(QueryBuilders.termQuery("yyear", EJCDateUtil.getYear()));
        } else if (financeUseQueryVO.is3Month()) {
            boolQuery.must(QueryBuilders.termsQuery("yyearMonth", EJCDateUtil.getMonthIntBetween(EJCDateUtil.getMonth(-2), EJCDateUtil.getMonth(0))));
        } else if (financeUseQueryVO.is6Month()) {
            boolQuery.must(QueryBuilders.termsQuery("yyearMonth", EJCDateUtil.getMonthIntBetween(EJCDateUtil.getMonth(-5), EJCDateUtil.getMonth(0))));
        } else {
            boolQuery.must(QueryBuilders.termQuery("yyear", EJCDateUtil.getLatYear()));
        }
        // 以batchId为分组条件，terms为分组后的字段名称，field为将被分组的字段名称
        TermsAggregationBuilder aggregation = AggregationBuilders.terms("yyearMonth").field("yyearMonth")
                // 分组求和integral字段，并将求和后的字段名改为score
                // subAggregation为子聚合，即在batchId分组后的小组内聚合
                .subAggregation(AggregationBuilders.sum("payMoneyAgg").field("payMoney"))
                // 注意这里，下面介绍
                .subAggregation(AggregationBuilders.sum("subSettleTaxMoneyAgg").field("subSettleTaxMoney"));

        sourceBuilder.aggregation(aggregation);
        sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        SearchResponse search = null;
        try {
            search = client.search(searchRequest, RequestOptions.DEFAULT);
            // 和之前不同的是这里需要getAggregations获取聚合后的数据
            Aggregations aggregations = search.getAggregations();
// 从分组后的数据中拿到batchId的数据，这里以batchId分组，则分组后的数据都在batchId里
            ParsedLongTerms terms = aggregations.get("yyearMonth");
// 获取到分组后的所有bucket
            List<? extends Terms.Bucket> buckets = terms.getBuckets();

            for (Terms.Bucket bucket : buckets) {
                System.out.println(bucket.getKey());
                Aggregations bucketAggregations = bucket.getAggregations();
                ParsedSum payMoneyAgg = bucketAggregations.get("payMoneyAgg");
                System.out.println(payMoneyAgg.getValue());
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

       /* TermsAggregationBuilder field = AggregationBuilders.terms("yyearMonthAgg").field("yyearMonth.keyword");
        AggregationBuilder sumAge = AggregationBuilders.sum("total_sum").field("payMoney");
        field.subAggregation(sumAge);
        sourceBuilder.aggregation(field);
        sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        List<FinanceUseVO> list = new ArrayList<>();
        SearchResponse response = null;
        FinanceUseVO financeUseVO = new FinanceUseVO();
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, FinanceUseVO.class));
            }


        } catch (IOException e) {
            e.printStackTrace();
        }*/
        return CommonResponse.success("查询资金使用数据成功！");
    }


    /**
     * @description:  重新加载资金使用
     * @return:
     * @author songlx
     * @date: 2022/4/14
     */
    @RequestMapping(value = "/extractFinanceUse", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> extractFinanceUse(@RequestBody Map<String, String> param, HttpServletRequest request) {
        String tableIdStr = "569222288382103594";
        String tableCode = "ejc_finance_use";

        // 查询数据表
        TableEntity tableEntity = null;
        if(StringUtils.isNotBlank(tableIdStr)) {
            tableEntity = tableService.selectById(Long.valueOf(tableIdStr));
        } else if(StringUtils.isNotBlank(tableCode)) {
            tableEntity = tableService.getByCode(tableCode);
        }

        if(null == tableEntity) {
            return CommonResponse.error("执行失败，没有找到匹配的报表信息！");
        }
        return syncDbDataToES(tableEntity, param, request.getHeader("authority"), RequestContextHolder.getRequestAttributes());
    }

    private CommonResponse<String> syncDbDataToES(TableEntity tableEntity, Map<String, String> param, String authority, RequestAttributes context) {
        Long tenantId = null != param.get("tenantId") ? Long.valueOf(param.get("tenantId")) : InvocationInfoProxy.getTenantid();
        Long yyearMonth = null != param.get("yyearMonth") ? Long.valueOf(param.get("yyearMonth")) : Long.valueOf(EJCDateUtil.format(new Date(), EJCDateUtil.SHORT_MONTH));

        List<Long> datasetIds = new ArrayList<>(Arrays.asList(tableEntity.getDatasetIds().split(","))).stream().map(s -> Long.valueOf(s)).collect(Collectors.toList());
        BulkRequest bulkRequest = new BulkRequest();
        boolean hasReq = false;
        final AtomicInteger count = new AtomicInteger(0);
        ConcurrentHashMap<Long, MdProjectVO> projectCache = new ConcurrentHashMap<>();

        ExecutorService threadPool = null;
        try {
            if(CollectionUtil.isNotEmpty(datasetIds)) {
                threadPool = Executors.newFixedThreadPool(datasetIds.size());
            }
            List<Future> futureList = new ArrayList<>();
            for(Long datasetId : datasetIds) {
                Callable<List<IndexRequest>> mainDatasetCallable = new MainDatasetCallable(count, context, datasetId,
                        tableEntity, authority, tenantId, projectCache, yyearMonth);
                Future<List<IndexRequest>> childFuture = threadPool.submit(mainDatasetCallable);
                futureList.add(childFuture);
            }

            for (Future<List<IndexRequest>> future : futureList) {
                List<IndexRequest> list = future.get(30, TimeUnit.SECONDS);

                if(CollectionUtils.isNotEmpty(list)) {
                    hasReq = true;
                    for(IndexRequest ir : list) {
                        bulkRequest.add(ir);
                    }
                }
            }

            if(hasReq) {
                //设置立即更新以解决数据查询不到的问题。
                bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
                client.bulk(bulkRequest, RequestOptions.DEFAULT);
            }
        } catch (Exception e) {
            logger.info("查询主数据集失败，", e);
            return CommonResponse.error("查询主数据集失败");
        } finally {
            if(null != threadPool) {
                threadPool.shutdown();
            }
        }
        return CommonResponse.success("执行成功");
    }

    class MainDatasetCallable implements Callable<List<IndexRequest>> {
        private AtomicInteger count;
        private RequestAttributes context;
        private TableEntity tableEntity;
        private Long datasetId;
        private String authority;
        private Long tenantId;
        private Long yyearMonth;
        private ConcurrentHashMap<Long, MdProjectVO> projectCache;

        public MainDatasetCallable(AtomicInteger count, RequestAttributes context, Long datasetId, TableEntity tableEntity,
                                   String authority, Long tenantId, ConcurrentHashMap<Long, MdProjectVO> projectCache, Long yyearMonth) {
            this.count = count;
            this.context = context;
            this.tableEntity = tableEntity;
            this.datasetId = datasetId;
            this.authority = authority;
            this.tenantId = tenantId;
            this.projectCache = projectCache;
            this.yyearMonth = yyearMonth;
        }



        @Override
        public List<IndexRequest> call() throws Exception {
            InvocationInfoProxy.setTenantid(tenantId);
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);
            List<IndexRequest> irList = new ArrayList<>();

            Map<String, Object> resp = new HashMap<>();
            List<JSONObject> mainDatasetList = new ArrayList<>();

            // 查询主数据集
            DatasetEntity datasetEntity = datasetService.selectById(datasetId);

            // 通过元数据查询该数据集属于哪个项目
            MdProjectVO projectVO = projectCache.get(datasetEntity.getMdProjectId().longValue());
            if(null == projectVO) {
                String url = baseHost + "ejc-metadata-web/api/mdProjectApi/queryDetail";
                Map<String, String> param = new HashMap<>();
                Map<String, String> header = new HashMap<>();
                param.put("id", datasetEntity.getMdProjectId().toString());
                header.put("authority", authority);
                String projectInfoResp = HttpTookit.get(url, param, header);
                logger.info("查询【id-{}】元数据项目信息结果：{}", datasetEntity.getMdProjectId().toString(), projectInfoResp);
                CommonResponse<JSONObject> response = JSONObject.parseObject(projectInfoResp, CommonResponse.class);

//                CommonResponse<MdProjectVO> projectResponse = projectApi.queryDetail(datasetEntity.getMdProjectId());
                if (response.isSuccess()) {
                    projectVO = JSONObject.parseObject(JSONObject.toJSONString(response.getData()), MdProjectVO.class);
                    projectCache.put(projectVO.getId().longValue(), projectVO);
                } else {
                    logger.error("主数据集【id-{}】查询数据失败, 其对应元数据查询失败, 原因：{}", datasetId, response.getMsg());
                }
            }
            if(null != projectVO) {
                String mainDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";
                JSONObject mainParamJson = new JSONObject();
                JSONObject mainQueryParam = new JSONObject();
                mainQueryParam.put("tenantId", tenantId);
                String sql = "SELECT dsq.* FROM (" + datasetEntity.getSqlContent() + " ) dsq WHERE dsq.tenantId =#{tenantId}";

                mainParamJson.put("sqlContent", JdkBase64Util.encode(sql));
                mainParamJson.put("datasetType", "1");
                mainParamJson.put("params", mainQueryParam);

                String mainResponseStr = ReferHttpClientUtils.postByJson(mainDatasetUrl, JSON.toJSONString(mainParamJson));

                CommonResponse<List<JSONObject>> mainResponse = JSON.parseObject(mainResponseStr, CommonResponse.class);
                if (mainResponse.isSuccess()) {
                    mainDatasetList = mainResponse.getData();

                    // 查询子数据集
                    List<DatasetEntity> childDatasetList = datasetService
                            .queryChildrenByParentId(datasetEntity.getId());
                    if (mainDatasetList != null && mainDatasetList.size() > 0) {
                        // 先根据创建者空间查询当前空间的所有数据,然后删除之
                        Long resultSize = queryIndexSize(tableEntity.getIndexName(), tenantId);
                        Long deleteTimes = resultSize % BATCH_SIZE;
                        for (int i = 0; i < deleteTimes + 1; i++) {
                            DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(tableEntity.getIndexName());
                            BoolQueryBuilder boolQueryBuilder = new BoolQueryBuilder();
                            boolQueryBuilder.must(QueryBuilders.termQuery("tenantId", tenantId));
                            boolQueryBuilder.must(QueryBuilders.termQuery("yyearMonth", yyearMonth));
                            deleteByQueryRequest.setQuery(boolQueryBuilder);
                            deleteByQueryRequest.setBatchSize(BATCH_SIZE);
                            client.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
                        }

                        // 查询数据列
                        Map<String, Object> paramMap = new HashMap<>();
                        paramMap.put("tableId", tableEntity.getId());
                        paramMap.put("tenantId", tenantId);
                        List<ColumnEntity> columnEntities = columnService.queryTenantColumnList(paramMap);
                        Map<String, String> columnMap = new HashMap<String, String>();
                        List<String> columnProperties = new ArrayList<>();
                        if (columnEntities != null && columnEntities.size() > 0) {
                            for (ColumnEntity columnEntity : columnEntities) {
                                if(StringUtils.isNotBlank(columnEntity.getProperty())) {
                                    //属性为空的列认为是复核列
                                    if (StringUtils.isBlank(columnEntity.getFormula())) {
                                        columnMap.put(columnEntity.getProperty(), columnEntity.getType());
                                        columnProperties.add(columnEntity.getProperty());
                                    } else {
                                        columnProperties.add(columnEntity.getProperty());
                                        columnMap.put(columnEntity.getProperty(), columnEntity.getFormula());
                                    }
                                }
                            }
                        }

                        ExecutorService threadPool = null;
                        if(CollectionUtil.isNotEmpty(childDatasetList)) {
                            threadPool = Executors.newFixedThreadPool(childDatasetList.size());
                        }
                        try {
                            for (JSONObject mainDataset : mainDatasetList) {
                                if (childDatasetList != null && childDatasetList.size() > 0) {
                                    List<Future<List<JSONObject>>> futureList = new ArrayList<>();
                                    for (DatasetEntity childDataset : childDatasetList) {
                                        Callable<List<JSONObject>> childCallable = new ChildDatasetCallable(childDataset, mainDataset,
                                                context, tenantId, authority, projectCache);
                                        Future<List<JSONObject>> childFuture = threadPool.submit(childCallable);
                                        futureList.add(childFuture);
                                    }

                                    for (Future<List<JSONObject>> future : futureList) {
                                        List<JSONObject> childList = future.get();
                                        if (childList != null && childList.size() > 0) {
                                            JSONObject childObject = childList.get(0);
                                            if(null != childObject) {
                                                for (Map.Entry<String, Object> entry : childObject.entrySet()) {
                                                    mainDataset.put(entry.getKey(), entry.getValue());
                                                }
                                            }
                                        }
                                    }
                                }

                                Map<String, Object> dataMap = new HashMap<>();
                                //排序
                                dataMap.put("creator_space", tenantId);
                                dataMap.put("data_sequence", count.incrementAndGet());
                                String columnType = null;
                                for(String key : columnProperties) {
                                    columnType = columnMap.get(key);
                                    if ("time".equals(columnType)) {
                                        if(null != mainDataset.get(key)) {
                                            dataMap.put(key, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(key)));
                                            mainDataset.put(key, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(key)));
                                        }
                                    } else if ("string".equals(columnType) || "number".equals(columnType)) {
                                        dataMap.put(key, mainDataset.get(key));
                                        mainDataset.put(key, mainDataset.get(key));
                                    } else { // 计算公式
                                        dataMap.put(key, CalculatorUtils.getResult(columnType, mainDataset));
                                        mainDataset.put(key, CalculatorUtils.getResult(columnType, mainDataset));
                                    }
                                }

                                IndexRequest indexRequest = new IndexRequest(tableEntity.getIndexName());
                                indexRequest.source(dataMap, XContentType.JSON);
                                irList.add(indexRequest);
                            }
                        } finally {
                            if(null != threadPool) {
                                //线程池关闭
                                threadPool.shutdown();
                            }
                        }
                    }
                } else {
                    logger.error("******************* 执行SQL查询失败：{}  *******************", mainResponse.getMsg());
                }
            }

            return irList;
        }
    }

    /**
     * 子线程
     *
     * @author Administrator
     *
     */
    class ChildDatasetCallable implements Callable<List<JSONObject>> {

        private DatasetEntity childDataset;
        private JSONObject mainDataset;
        private RequestAttributes context;
        private Long tenantId;
        private String authority;
        private ConcurrentHashMap<Long, MdProjectVO> projectCache;

        public ChildDatasetCallable() {
        }

        public ChildDatasetCallable(DatasetEntity childDataset, JSONObject mainDataset,
                                    RequestAttributes context, Long tenantId, String authority, ConcurrentHashMap<Long, MdProjectVO> projectCache) {
            this.childDataset = childDataset;
            this.mainDataset = mainDataset;
            this.context = context;
            this.tenantId = tenantId;
            this.authority = authority;
            this.projectCache = projectCache;
        }

        @Override
        public List<JSONObject> call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            List<JSONObject> dataList = new ArrayList<JSONObject>();
//             通过元数据查询该数据集属于哪个项目
            MdProjectVO projectVO = projectCache.get(childDataset.getMdProjectId().longValue());
            if(null == projectVO) {
                String url = baseHost + "ejc-metadata-web/api/mdProjectApi/queryDetail";
                Map<String, String> param = new HashMap<>();
                Map<String, String> header = new HashMap<>();
                param.put("id", childDataset.getMdProjectId().toString());
                header.put("authority", authority);
                String projectInfoResp = HttpTookit.get(url, param, header);
                logger.info("查询【id-{}】元数据项目信息结果：{}", childDataset.getMdProjectId(), projectInfoResp);
                CommonResponse<JSONObject> response = JSONObject.parseObject(projectInfoResp, CommonResponse.class);
                if (response.isSuccess()) {
                    projectVO = JSONObject.parseObject(JSONObject.toJSONString(response.getData()), MdProjectVO.class);
                    projectCache.put(projectVO.getId().longValue(), projectVO);
                } else {
                    logger.error("数据子集【id-{}】查询数据失败, 其对应元数据查询失败, 原因：{}", childDataset.getId(), response.getMsg());
                }
            }

            if(null != projectVO) {
                String childDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";

                JSONObject childParamJson = new JSONObject();
                JSONObject childQueryParam = new JSONObject();
                childQueryParam.put("tenantId", tenantId);
                if (StringUtils.isNotBlank(childDataset.getChildParam())) {
                    String[] childParamArr = childDataset.getChildParam().split(",");
                    for (String childParamStr : childParamArr) {
                        childQueryParam.put(childParamStr, mainDataset.getString(childParamStr));
                    }
                }
                childParamJson.put("sqlContent", JdkBase64Util.encode(childDataset.getSqlContent()));
                childParamJson.put("datasetType", "2");
                childParamJson.put("params", childQueryParam);

                String childResponseStr = ReferHttpClientUtils.postByJson(childDatasetUrl, JSON.toJSONString(childParamJson));

                CommonResponse<List<JSONObject>> childResponse = JSON.parseObject(childResponseStr, CommonResponse.class);

                if (childResponse.isSuccess()) {
                    logger.info("执行sql-{}, 参数-{}, 结果-{}", childDataset.getSqlContent(), childQueryParam, JSONObject.toJSONString(childResponse.getData()));
                    return childResponse.getData();
                }
            }
            return dataList;
        }

    }

    /**
     * 查询当前空间的所有数据
     *
     * @param indexName
     * @param creatorSpace
     * @return
     */
    private Long queryIndexSize(String indexName, Long creatorSpace) {
        Long resultSize = 0L;
        SearchRequest searchRequest = new SearchRequest(indexName);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();

        // 查询参数
        TermQueryBuilder termQueryBuilder = QueryBuilders.termQuery("creator_space", creatorSpace);

        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.must(termQueryBuilder);

        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); // 设置超时时间
        searchRequest.source(sourceBuilder);
        try {
            resultSize = queryCreatorSize(searchRequest);
        } catch (IOException e) {
            try { // 重试一次
                resultSize = queryCreatorSize(searchRequest);
            } catch (IOException e1) {
                e1.printStackTrace();
                throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG：" + e1.getMessage());
            }
        }
        return resultSize;
    }

    private Long queryCreatorSize(SearchRequest searchRequest) throws IOException {
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHits hits = response.getHits();
        return hits.getTotalHits().value;
    }

}
