package com.ejianc.foundation.report.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.metadata.vo.MdProjectVO;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.DatasetEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.consts.FinanceUseConsts;
import com.ejianc.foundation.report.util.CalculatorUtils;
import com.ejianc.foundation.report.util.EJCDateUtil;
import com.ejianc.foundation.tenant.api.ITenantApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.util.JdkBase64Util;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.script.Invocable;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 自定义报表查询
 * @date 2022/4/15
 */
@Service
public class CustomReportService {

    private static final Logger logger = LoggerFactory.getLogger(CustomReportService.class);

    @Value("${common.env.base-host}")
    private String baseHost;

    private final static Integer BATCH_SIZE = 200;

    private final static Integer QUERY_TIMEOUT = 60;

    @Autowired
    private IDatasetService datasetService;
    @Autowired
    private IColumnService columnService;
    @Autowired
    private RestHighLevelClient client;

    @Autowired
    private ITableService tableService;

    @Autowired
    private ITenantApi tenantApi;


    public CommonResponse<String> customSyncDbDataToES(TableEntity tableEntity, Map<String, String> param, List<Integer> yyearMonthList, String authority, RequestAttributes context) {
        Long tenantId = null != param.get("tenantId") ? Long.valueOf(param.get("tenantId")) : InvocationInfoProxy.getTenantid();
        List<Long> datasetIds = new ArrayList<>(Arrays.asList(tableEntity.getDatasetIds().split(","))).stream().map(s -> Long.valueOf(s)).collect(Collectors.toList());
        BulkRequest bulkRequest = new BulkRequest();
        boolean hasReq = false;
        final AtomicInteger count = new AtomicInteger(0);
        ConcurrentHashMap<Long, MdProjectVO> projectCache = new ConcurrentHashMap<>();

        ExecutorService threadPool = null;
        try {
            if (CollectionUtil.isNotEmpty(datasetIds)) {
                threadPool = Executors.newFixedThreadPool(datasetIds.size());
            }
            List<Future> futureList = new ArrayList<>();
            for (Long datasetId : datasetIds) {
                Callable<List<IndexRequest>> mainDatasetCallable = new MainDatasetCallable(count, context, datasetId,
                        tableEntity, authority, tenantId, projectCache, yyearMonthList);
                Future<List<IndexRequest>> childFuture = threadPool.submit(mainDatasetCallable);
                futureList.add(childFuture);
            }

            for (Future<List<IndexRequest>> future : futureList) {
                List<IndexRequest> list = future.get(300, TimeUnit.SECONDS);

                if (CollectionUtils.isNotEmpty(list)) {
                    hasReq = true;
                    for (IndexRequest ir : list) {
                        bulkRequest.add(ir);
                    }
                }
            }

            if (hasReq) {
                //设置立即更新以解决数据查询不到的问题。
                bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
                client.bulk(bulkRequest, RequestOptions.DEFAULT);
            }
        } catch (Exception e) {
            logger.info("查询主数据集失败，", e);
            return CommonResponse.error("查询主数据集失败");
        } finally {
            if (null != threadPool) {
                threadPool.shutdown();
            }
        }
        return CommonResponse.success("执行成功");
    }

    class MainDatasetCallable implements Callable<List<IndexRequest>> {
        private AtomicInteger count;
        private RequestAttributes context;
        private TableEntity tableEntity;
        private Long datasetId;
        private String authority;
        private Long tenantId;
        private List<Integer> yyearMonthList;
        private ConcurrentHashMap<Long, MdProjectVO> projectCache;

        public MainDatasetCallable(AtomicInteger count, RequestAttributes context, Long datasetId, TableEntity tableEntity,
                                   String authority, Long tenantId, ConcurrentHashMap<Long, MdProjectVO> projectCache, List<Integer> yyearMonthList) {
            this.count = count;
            this.context = context;
            this.tableEntity = tableEntity;
            this.datasetId = datasetId;
            this.authority = authority;
            this.tenantId = tenantId;
            this.projectCache = projectCache;
            this.yyearMonthList = yyearMonthList;
        }


        @Override
        public List<IndexRequest> call() throws Exception {
            InvocationInfoProxy.setTenantid(tenantId);
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);
            List<IndexRequest> irList = new ArrayList<>();
            List<JSONObject> mainDatasetList = new ArrayList<>();

            // 查询主数据集
            DatasetEntity datasetEntity = datasetService.selectById(datasetId);

            // 通过元数据查询该数据集属于哪个项目
            MdProjectVO projectVO = projectCache.get(datasetEntity.getMdProjectId().longValue());
            if (null == projectVO) {
                String url = baseHost + "ejc-metadata-web/api/mdProjectApi/queryDetail";
                Map<String, String> param = new HashMap<>();
                Map<String, String> header = new HashMap<>();
                param.put("id", datasetEntity.getMdProjectId().toString());
                header.put("authority", authority);
                String projectInfoResp = HttpTookit.get(url, param, header);
                logger.info("查询【id-{}】元数据项目信息结果：{}", datasetEntity.getMdProjectId().toString(), projectInfoResp);
                CommonResponse<JSONObject> response = JSONObject.parseObject(projectInfoResp, CommonResponse.class);

                if (response.isSuccess()) {
                    projectVO = JSONObject.parseObject(JSONObject.toJSONString(response.getData()), MdProjectVO.class);
                    projectCache.put(projectVO.getId().longValue(), projectVO);
                } else {
                    logger.error("主数据集【id-{}】查询数据失败, 其对应元数据查询失败, 原因：{}", datasetId, response.getMsg());
                }
            }
            if (null != projectVO) {
                String mainDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";
                JSONObject mainParamJson = new JSONObject();
                JSONObject mainQueryParam = new JSONObject();
                mainQueryParam.put("tenantId", tenantId);
                String sql = "SELECT dsq.* FROM (" + datasetEntity.getSqlContent() + " ) dsq WHERE dsq.tenantId =#{tenantId}";

                for (Integer yyearMonth : yyearMonthList) {
                    mainParamJson.put("sqlContent", JdkBase64Util.encode(sql.replace("#CustomDynamicSQL#", String.valueOf(yyearMonth).substring(0, 4) + " yyear," + yyearMonth + " yyearMonth")));
                    mainParamJson.put("datasetType", "1");
                    mainParamJson.put("params", mainQueryParam);

                    String mainResponseStr = ReferHttpClientUtils.postByJson(mainDatasetUrl, JSON.toJSONString(mainParamJson));

                    CommonResponse<List<JSONObject>> mainResponse = JSON.parseObject(mainResponseStr, CommonResponse.class);
                    if (mainResponse.isSuccess()) {
                        mainDatasetList = mainResponse.getData();

                        // 查询子数据集
                        List<DatasetEntity> childDatasetList = datasetService
                                .queryChildrenByParentId(datasetEntity.getId());
                        if (mainDatasetList != null && mainDatasetList.size() > 0) {
                            // 先根据创建者空间查询当前空间的所有数据,然后删除之
                            Long resultSize = queryIndexSize(tableEntity.getIndexName(), tenantId, yyearMonth);
                            Long deleteTimes = resultSize % BATCH_SIZE;
                            for (int i = 0; i < deleteTimes + 1; i++) {
                                DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(tableEntity.getIndexName());
                                BoolQueryBuilder boolQueryBuilder = new BoolQueryBuilder();
                                boolQueryBuilder.must(QueryBuilders.termQuery("creator_space", tenantId));
                                boolQueryBuilder.must(QueryBuilders.termQuery("yyearMonth", yyearMonth));
                                deleteByQueryRequest.setQuery(boolQueryBuilder);
                                deleteByQueryRequest.setBatchSize(BATCH_SIZE);
                                client.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
                            }

                            // 查询数据列
                            Map<String, Object> paramMap = new HashMap<>();
                            paramMap.put("tableId", tableEntity.getId());
                            paramMap.put("tenantId", tenantId);
                            List<ColumnEntity> columnEntities = columnService.queryTenantColumnList(paramMap);
                            Map<String, String> columnMap = new HashMap<>();
                            Map<String, ColumnEntity> columnEntityMap = new HashMap<>();
                            List<String> columnProperties = new ArrayList<>();
                            if (columnEntities != null && columnEntities.size() > 0) {
                                for (ColumnEntity columnEntity : columnEntities) {
                                    if (StringUtils.isNotBlank(columnEntity.getProperty())) {
                                        //属性为空的列认为是复核列
                                        if (StringUtils.isBlank(columnEntity.getFormula())) {
                                            columnEntityMap.put(columnEntity.getProperty(), columnEntity);
                                            columnMap.put(columnEntity.getProperty(), columnEntity.getType());
                                            columnProperties.add(columnEntity.getProperty());
                                        } else {
                                            columnProperties.add(columnEntity.getProperty());
                                            columnMap.put(columnEntity.getProperty(), columnEntity.getFormula());
                                        }
                                    }
                                }
                            }

                            ExecutorService threadPool = null;
                            if (CollectionUtil.isNotEmpty(childDatasetList)) {
                                threadPool = Executors.newFixedThreadPool(childDatasetList.size());
                            }
                            try {
                                for (JSONObject mainDataset : mainDatasetList) {
                                    if (childDatasetList != null && childDatasetList.size() > 0) {
                                        List<Future<List<JSONObject>>> futureList = new ArrayList<>();
                                        for (DatasetEntity childDataset : childDatasetList) {
                                            Callable<List<JSONObject>> childCallable = new ChildDatasetCallable(childDataset, mainDataset,
                                                    context, tenantId, authority, projectCache);
                                            Future<List<JSONObject>> childFuture = threadPool.submit(childCallable);
                                            futureList.add(childFuture);
                                        }

                                        for (Future<List<JSONObject>> future : futureList) {
                                            List<JSONObject> childList = future.get();
                                            if (childList != null && childList.size() > 0) {
                                                JSONObject childObject = childList.get(0);
                                                if (null != childObject) {
                                                    for (Map.Entry<String, Object> entry : childObject.entrySet()) {
                                                        mainDataset.put(entry.getKey(), entry.getValue());
                                                    }
                                                }
                                            }
                                        }
                                    }

                                    Map<String, Object> dataMap = new HashMap<>();
                                    //排序
                                    dataMap.put("creator_space", tenantId);
                                    dataMap.put("data_sequence", count.incrementAndGet());
                                    String columnType = null;
                                    for (String key : columnProperties) {
                                        columnType = columnMap.get(key);
                                        if ("time".equals(columnType)) {
                                            if (null != mainDataset.get(key)) {
                                                dataMap.put(key, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(key)));
                                                mainDataset.put(key, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(key)));
                                            }
                                        } else if ("string".equals(columnType) || "number".equals(columnType)) {
                                            dataMap.put(key, mainDataset.get(key));
                                            mainDataset.put(key, mainDataset.get(key));
                                        } else if ("pic".equals(columnType)) {

                                        } else { // 计算公式
                                            dataMap.put(key, CalculatorUtils.getResult(columnType, mainDataset));
                                            mainDataset.put(key, CalculatorUtils.getResult(columnType, mainDataset));
                                        }
                                    }
                                    for (String key : columnProperties) {
                                        columnType = columnMap.get(key);
                                        if ("pic".equals(columnType)) {
                                            if(StringUtils.isNotEmpty(columnEntityMap.get(key).getFormatter())){
                                                ScriptEngineManager manager = new ScriptEngineManager();
                                                ScriptEngine engine = manager.getEngineByName("js");
                                                try {
                                                    engine.eval(columnEntityMap.get(key).getFormatter());
                                                    if(engine instanceof Invocable) {
                                                        Invocable invoke = (Invocable)engine;    // 调用merge方法，并传入两个参数
                                                        Integer r = (Integer)invoke.invokeFunction("getIconByRowData", mainDataset); //调用了js的aa方法
                                                        logger.info("调用js 方法 ---{}，参数----{}  ，结果 ---{}",columnEntityMap.get(key).getFormatter(),mainDataset,r);
                                                        dataMap.put(key, r);
                                                        mainDataset.put(key, r);
                                                    }else {
                                                        dataMap.put(key, 1);
                                                        mainDataset.put(key, 1);
                                                    }
                                                } catch (ScriptException | NoSuchMethodException e) {
                                                    e.printStackTrace();
                                                    logger.info("执行js脚本出错！！！",e);
                                                    dataMap.put(key, 1);
                                                    mainDataset.put(key, 1);
                                                }
                                            }else {
                                                dataMap.put(key, 1);
                                                mainDataset.put(key, 1);
                                            }
                                        }
                                    }

                                    IndexRequest indexRequest = new IndexRequest(tableEntity.getIndexName());
                                    indexRequest.source(dataMap, XContentType.JSON);
                                    irList.add(indexRequest);
                                }
                            } finally {
                                if (null != threadPool) {
                                    //线程池关闭
                                    threadPool.shutdown();
                                }
                            }
                        }
                    } else {
                        logger.error("******************* 执行SQL查询失败：{}  *******************", mainResponse.getMsg());
                    }
                }
            }
            return irList;
        }
    }

    /**
     * 子线程
     *
     * @author Administrator
     */
    class ChildDatasetCallable implements Callable<List<JSONObject>> {

        private DatasetEntity childDataset;
        private JSONObject mainDataset;
        private RequestAttributes context;
        private Long tenantId;
        private String authority;
        private ConcurrentHashMap<Long, MdProjectVO> projectCache;

        public ChildDatasetCallable() {
        }

        public ChildDatasetCallable(DatasetEntity childDataset, JSONObject mainDataset,
                                    RequestAttributes context, Long tenantId, String authority, ConcurrentHashMap<Long, MdProjectVO> projectCache) {
            this.childDataset = childDataset;
            this.mainDataset = mainDataset;
            this.context = context;
            this.tenantId = tenantId;
            this.authority = authority;
            this.projectCache = projectCache;
        }

        @Override
        public List<JSONObject> call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            List<JSONObject> dataList = new ArrayList<JSONObject>();
//             通过元数据查询该数据集属于哪个项目
            MdProjectVO projectVO = projectCache.get(childDataset.getMdProjectId().longValue());
            if (null == projectVO) {
                String url = baseHost + "ejc-metadata-web/api/mdProjectApi/queryDetail";
                Map<String, String> param = new HashMap<>();
                Map<String, String> header = new HashMap<>();
                param.put("id", childDataset.getMdProjectId().toString());
                header.put("authority", authority);
                String projectInfoResp = HttpTookit.get(url, param, header);
                logger.info("查询【id-{}】元数据项目信息结果：{}", childDataset.getMdProjectId(), projectInfoResp);
                CommonResponse<JSONObject> response = JSONObject.parseObject(projectInfoResp, CommonResponse.class);
                if (response.isSuccess()) {
                    projectVO = JSONObject.parseObject(JSONObject.toJSONString(response.getData()), MdProjectVO.class);
                    projectCache.put(projectVO.getId().longValue(), projectVO);
                } else {
                    logger.error("数据子集【id-{}】查询数据失败, 其对应元数据查询失败, 原因：{}", childDataset.getId(), response.getMsg());
                }
            }

            if (null != projectVO) {
                String childDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";

                JSONObject childParamJson = new JSONObject();
                JSONObject childQueryParam = new JSONObject();
                childQueryParam.put("tenantId", tenantId);
                if (StringUtils.isNotBlank(childDataset.getChildParam())) {
                    String[] childParamArr = childDataset.getChildParam().split(",");
                    for (String childParamStr : childParamArr) {
                        childQueryParam.put(childParamStr, mainDataset.getString(childParamStr));
                    }
                }
                childParamJson.put("sqlContent", JdkBase64Util.encode(childDataset.getSqlContent()));
                childParamJson.put("datasetType", "2");
                childParamJson.put("params", childQueryParam);

                String childResponseStr = ReferHttpClientUtils.postByJson(childDatasetUrl, JSON.toJSONString(childParamJson));

                CommonResponse<List<JSONObject>> childResponse = JSON.parseObject(childResponseStr, CommonResponse.class);

                if (childResponse.isSuccess()) {
                    logger.info("执行sql-{}, 参数-{}, 结果-{}", childDataset.getSqlContent(), childQueryParam, JSONObject.toJSONString(childResponse.getData()));
                    return childResponse.getData();
                }
            }
            return dataList;
        }

    }

    /**
     * 查询当前空间的所有数据
     *
     * @param indexName
     * @param creatorSpace
     * @param yyearMonth
     * @return
     */
    private Long queryIndexSize(String indexName, Long creatorSpace, Integer yyearMonth) {
        Long resultSize = 0L;
        SearchRequest searchRequest = new SearchRequest(indexName);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();

        // 查询参数
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.must(QueryBuilders.termQuery("creator_space", creatorSpace));
        boolQueryBuilder.must(QueryBuilders.termQuery("yyearMonth", yyearMonth));
        boolQueryBuilder.must(boolQueryBuilder);

        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); // 设置超时时间
        searchRequest.source(sourceBuilder);
        try {
            resultSize = queryCreatorSize(searchRequest);
        } catch (IOException e) {
            try { // 重试一次
                resultSize = queryCreatorSize(searchRequest);
            } catch (IOException e1) {
                e1.printStackTrace();
                throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG：" + e1.getMessage());
            }
        }
        return resultSize;
    }

    private Long queryCreatorSize(SearchRequest searchRequest) throws IOException {
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHits hits = response.getHits();
        return hits.getTotalHits().value;
    }


    public CommonResponse<String> refreshAllTenantData(HttpServletRequest request, Integer isUpdAll) {

        Long startTime = System.currentTimeMillis();
        //将待刷新数据的租户列表分为3份
        int sperateLen = 3;

        //1、获取所有有效的租户Id列表
        CommonResponse<List<Long>> tenantIdListResp = tenantApi.getAllValidTenantId();
        if (!tenantIdListResp.isSuccess()) {
            logger.error("获取所有有效租户Id列表失败，{}", tenantIdListResp.getMsg());
            return CommonResponse.error("刷新租户统计数据失败，获取所有有效租户Id列表失败");
        }
        List<Long> tenantIdList = tenantIdListResp.getData();
        if (CollectionUtils.isEmpty(tenantIdList)) {
            return CommonResponse.success("刷新租户统计数据完成，待处理数据的租户列表为空！");
        }

        //2、获取资金使用报表
        TableEntity table = tableService.selectById(FinanceUseConsts.ID_EJC_FINANCE_USE);
        if (table == null) {
            return CommonResponse.success("刷新租户统计数据完成，未发现资金使用报表信息！");
        }
        //3、多线程分别刷新每个租户的统计数据
        int len = tenantIdList.size() / sperateLen;
        List<Future> handlers = new ArrayList<>();
        ExecutorService threadPool = Executors.newFixedThreadPool(sperateLen);
        List<Long> tenantIds = null;
        for (int i = 0; i < sperateLen; i++) {
            tenantIds = tenantIdList.subList(i * len, (i + 1) * len < tenantIdList.size() ? (i + 1) * len : tenantIdList.size());
            TenantDataRefreshHandle handler = new TenantDataRefreshHandle(tenantIds,
                    request.getHeader("authority"), table, RequestContextHolder.getRequestAttributes(), isUpdAll);

            Future<String> childFuture = threadPool.submit(handler);
            handlers.add(childFuture);
        }

        for (Future<String> f : handlers) {
            try {
                String result = f.get();
                logger.info(result);
            } catch (Exception e) {
                logger.error("刷新租户-{}统计报表数据异常，", StringUtils.join(tenantIds, ","), e);
            }
        }

        Long endTime = System.currentTimeMillis();
        logger.info("刷新租户统计数据完成, 耗时（ms）：{}", endTime - startTime);

        return CommonResponse.success("刷新租户统计数据完成！");
    }

    class TenantDataRefreshHandle implements Callable<String> {

        private List<Long> tenantIds;
        private String authority;
        private TableEntity table;
        private RequestAttributes context;

        private Integer isUpdAll;

        private TenantDataRefreshHandle(List<Long> tenantIds, String authority, TableEntity table, RequestAttributes context, Integer isUpdAll) {
            this.table = table;
            this.tenantIds = tenantIds;
            this.authority = authority;
            this.context = context;
            this.isUpdAll = isUpdAll;
        }

        @Override
        public String call() {
            Map<String, String> param = new HashMap<>();
            CommonResponse<String> dataRefresh = null;
            for (Long tenantId : tenantIds) {
                param.put("tenantId", tenantId.toString());

                if(isUpdAll != null){
                    String endMonth = EJCDateUtil.getMonth(0);
                    String startMonth = isUpdAll >15 ? EJCDateUtil.getLastYear() + "-01" : EJCDateUtil.getMonth((0 - isUpdAll.intValue()));
                    if(isUpdAll == -1){
                        startMonth = EJCDateUtil.getLastYear() + "-01";
                        endMonth = EJCDateUtil.getLastYear() + "-12";
                    }

                    List<Integer> yyearMonthList = EJCDateUtil.getShortMonthIntBetween(startMonth, endMonth);
                    dataRefresh = customSyncDbDataToES(table, param, yyearMonthList, authority, context);
                    logger.info("刷新租户-{},报表-[名称:{}，编码：{}]数据结果：{}", tenantId, table.getTableName(), table.getCode(), dataRefresh.getMsg());

                } else {
                    List<Integer> monthIntBetween = EJCDateUtil.getShortMonthIntBetween(EJCDateUtil.getMonth(FinanceUseConsts.NEAR_MONTH_NUM), EJCDateUtil.getMonth(0));
                    dataRefresh = customSyncDbDataToES(table, param, monthIntBetween, authority, context);
                    logger.info("刷新租户-{},报表-[名称:{}，编码：{}]数据结果：{}", tenantId, table.getTableName(), table.getCode(), dataRefresh.getMsg());

                }

            }
            return "租户-[" + StringUtils.join(tenantIds, ",") + "]数据刷新任务执行完成！";
        }
    }

    public static void main(String[] args) {
        Integer isUpdAll =-1;
        String endMonth = EJCDateUtil.getMonth(0);
        String startMonth = isUpdAll >15 ? EJCDateUtil.getLastYear() + "-01" : EJCDateUtil.getMonth((0 - isUpdAll.intValue()));
        if(isUpdAll == -1){
            startMonth = EJCDateUtil.getLastYear() + "-01";
            endMonth = EJCDateUtil.getLastYear() + "-12";
        }
        System.out.println(startMonth);
        System.out.println(endMonth);
    }

}
