package com.ejianc.foundation.report.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.RefreshTimeEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.controller.param.GridHeader;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.IRefreshTimeService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.ExcelExportUtil;
import com.ejianc.framework.cache.redissonlock.RedissonLocker;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.EnvironmentTools;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.core.util.ResultAsTree;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

@RestController
@RequestMapping("/data/report")
public class DataReportController implements Serializable {

	private static final long serialVersionUID = 6950770890208974786L;
	private final static Integer QUERY_TIMEOUT = 60;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
	@Autowired
	private ITableService tableService;
	@Autowired
	private IColumnService columnService;
	@Autowired
	private RestHighLevelClient client;
	@Autowired
	private IRefreshTimeService refreshTimeService;
	@Autowired
	private EnvironmentTools environmentTools;
	@Autowired
	private RedissonLocker redissonLocker;
	
	@Value("${oms.tenantid}")
	private Long BASE_TENANTID;


	@PostMapping(value = "/readByCode/{tableCode}")
	public CommonResponse<JSONObject> getReport(HttpServletRequest req,@PathVariable String tableCode,@RequestBody QueryParam queryParam) {
		TableEntity tableEntity = tableService.getByCode(tableCode);

		if(null == tableEntity) {
			return CommonResponse.error("数据查询失败，没有匹配的报表信息！");
		}

		return getReport(req, tableEntity, queryParam, true);
	}

	/**
	 * 根据报表编码，查询指标字段数据
	 * @param req
	 * @param tableCode
	 * @param queryParam
	 * 借用 queryParam 中 pageSize 作为全部数量
	 * @return
	 */
	@PostMapping(value = "/readExByCode/{tableCode}")
	public CommonResponse<List<JSONObject>> readExByCode(HttpServletRequest req,@PathVariable String tableCode,@RequestBody QueryParam queryParam) {
		TableEntity tableEntity = tableService.getByCode(tableCode);
		Long tenantId = InvocationInfoProxy.getTenantid();

		String lockKey = tenantId+"_"+tableEntity.getId();
		boolean locked = redissonLocker.isLocked(lockKey);
		if(locked){
			throw new BusinessException("数据刷新中，请稍后再获取数据");
		}
		//查询表头
		List<GridHeader> gridHeaders = columnService.queryGridHeadList(tableEntity.getId(), tenantId);
		Map<String, String> filedTypeMap = new HashMap<>();
		for(GridHeader header : gridHeaders) {
			if(StringUtils.isNotBlank(header.getCode())) {
				filedTypeMap.put(header.getCode(), header.getType());
			}
		}
		String msg = configQueryParam(tableEntity, queryParam, req, true);
		if(StringUtils.isNotBlank(msg)) {
			return CommonResponse.error(msg);
		}

		List<ColumnEntity> keyList = new ArrayList<>();
		Map<String, Object> paramMap = new HashMap<>();
		paramMap.put("tableId", tableEntity.getId());
		paramMap.put("tenantId", tenantId);
		List<ColumnEntity> columnEntities = columnService.queryTenantColumnList(paramMap);
		for (ColumnEntity col : columnEntities) {
			if("number".equals(col.getType()) && col.getDataTotal()!=null && col.getDataTotal()){
				keyList.add(col);
			}
		}

		List<JSONObject> res = new ArrayList<>();
		logger.info("查询指标，参数-{}",JSONObject.toJSONString(queryParam));
		IPage<JSONObject> pageData = columnService.queryPageList(tableEntity.getIndexName(), queryParam, null, filedTypeMap);
		Map<String,JSONObject> map = new HashMap<>();
		for (ColumnEntity key : keyList) {
			JSONObject object = new JSONObject();
			object.put("property",key.getProperty());
			object.put("name",key.getColumnName());
			object.put("value",new BigDecimal("0.00"));
			map.put(key.getProperty(),object);
		}
		if(pageData.getTotal()>0){
			for (JSONObject record : pageData.getRecords()) {
				for (ColumnEntity key : keyList) {
					JSONObject object = map.get(key.getProperty());
					BigDecimal value = object.getBigDecimal("value");
					value = value.add(record.getBigDecimal(key.getProperty())!=null?record.getBigDecimal(key.getProperty()).setScale(2, RoundingMode.HALF_UP):new BigDecimal("0.00"));
					object.put("value",value);
					map.put(key.getProperty(),object);
				}
			}
			for (ColumnEntity key : keyList) {
				JSONObject object = map.get(key.getProperty());
				res.add(object);
			}
		}
		return CommonResponse.success(res);
	}

    /**
     * 不查询报表中配置的后端参数服务获取查询参数
     *
     * @param req
     * @param tableCode
     * @param queryParam
     * @return
     */
	@PostMapping(value = "/readByCode/uncustom/{tableCode}")
	public CommonResponse<JSONObject> getReportWithSpeParams(HttpServletRequest req,@PathVariable String tableCode,@RequestBody QueryParam queryParam) {
		TableEntity tableEntity = tableService.getByCode(tableCode);

		if(null == tableEntity) {
			return CommonResponse.error("数据查询失败，没有匹配的报表信息！");
		}

		return getReport(req, tableEntity, queryParam, false);
	}

	private CommonResponse<JSONObject> getReport(HttpServletRequest req, TableEntity tableEntity, QueryParam queryParam, boolean getCommonQueryParam) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		//查询表头
		List<GridHeader> gridHeaders = columnService.queryGridHeadList(tableEntity.getId(), tenantId);
		//查询数据
		queryParam.getParams().put("creator_space", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		String msg = configQueryParam(tableEntity, queryParam, req, getCommonQueryParam);
		if(StringUtils.isNotBlank(msg)) {
			return CommonResponse.error(msg);
		}

		List<String> heightFields = new ArrayList<>();
		Map<String, String> filedTypeMap = new HashMap<>();
		for(GridHeader header : gridHeaders) {
			if(StringUtils.isNotBlank(header.getCode())) {
				filedTypeMap.put(header.getCode(), header.getType());
				if(header.getVisible()) {
					heightFields.add(header.getCode());
				}
			}
		}
		queryParam.getParams().remove("_treeSelectId");
//		logger.info("查询sql queryParam："+JSONObject.toJSONString(queryParam));
		List<Map> headerList = BeanMapper.mapList(gridHeaders, Map.class);
		IPage<JSONObject> pageData = columnService.queryPageList(tableEntity.getIndexName(), queryParam, heightFields, filedTypeMap);

		JSONObject responseData = new JSONObject();
		responseData.put("reportTitle", tableEntity.getTableName());
		responseData.put("reportType", tableEntity.getReportType());
		responseData.put("leftTreeUrl", tableEntity.getLeftTreeUrl());
		if(StringUtils.isNotBlank(tableEntity.getRequestMode())) {
			responseData.put("requestMode", tableEntity.getRequestMode());
		}
		if(StringUtils.isNotBlank(tableEntity.getTreeKey())) {
			responseData.put("treeKey", tableEntity.getTreeKey());
		}
		if(StringUtils.isNotBlank(tableEntity.getTreeName())) {
			responseData.put("treeName", tableEntity.getTreeName());
		}
		responseData.put("filterItems", tableEntity.getConditionJson());
		responseData.put("orderGroup", tableEntity.getOrderItemJson());
		responseData.put("data", pageData);
		responseData.put("tableId", tableEntity.getId());
		responseData.put("gridheaders", ResultAsTree.createTreeData(headerList));
		
		QueryWrapper<RefreshTimeEntity> refreshTimeWrapper = new QueryWrapper<>();
		refreshTimeWrapper.eq("tenant_id", tenantId);
		refreshTimeWrapper.eq("table_id", tableEntity.getId());
		RefreshTimeEntity refreshTimeEntity = refreshTimeService.getOne(refreshTimeWrapper);
		if(refreshTimeEntity != null) {
			responseData.put("refreshTime", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(refreshTimeEntity.getLastRefreshTime()));
		}
		/** 是否支持图表 */
		responseData.put("supportImage",tableEntity.getSupportImage());
		/** 全部数量 */
		responseData.put("totalTableData",pageData.getTotal());
		/** 全部数量 */
		responseData.put("tableCode",tableEntity.getCode());
		/** 第一页进行查询，其他页不再查询 */
		if(queryParam.getPageIndex() == 1){
			if(tableEntity.getPid()==null){
				/** 查询报表子表编码*/
				List<String> tableCodes = new ArrayList<>();
				QueryParam param = new QueryParam();
				param.getParams().put("pid",new Parameter(QueryParam.EQ,tableEntity.getId()));
				List<TableEntity> list = tableService.queryList(param,false);
				if(ListUtil.isNotEmpty(list)){
					tableCodes.add(tableEntity.getCode()+","+tableEntity.getTableName());
					for (TableEntity entity : list) {
						tableCodes.add(entity.getCode()+","+entity.getTableName());
					}
					responseData.put("tableCodes",tableCodes);
				}
			}
			if(tableEntity.getSupportImage()!=null && tableEntity.getSupportImage()){
				responseData.put("dataX",getImgCols(tableEntity.getId(),"dataX"));
				responseData.put("dataY",getImgCols(tableEntity.getId(),"dataY"));
			}
		}
		return CommonResponse.success(responseData);

	}

	private JSONObject getImgCols(Long tbId,String col){
		QueryParam param = new QueryParam();
		param.getParams().put("tableId",new Parameter(QueryParam.EQ,tbId));
		param.getParams().put(col,new Parameter(QueryParam.EQ,true));
		List<ColumnEntity> list = columnService.queryList(param,false);
		JSONObject res = null;
		if(ListUtil.isNotEmpty(list)){
			res = new JSONObject();
			for (ColumnEntity columnEntity : list) {
				res.put(columnEntity.getProperty(),columnEntity.getColumnName());
			}
		}
		return res;
	}

	/**
	 * 报表查询
	 *  不查询报表中配置的后端参数服务获取查询参数
	 * @param tableId
	 * @param queryParam
	 * @return
	 */
	@RequestMapping(value = "/read/uncustom/{tableId}", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<JSONObject> extractDataWithSpeParams(HttpServletRequest req,@PathVariable Long tableId,@RequestBody QueryParam queryParam) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		
		TableEntity tableEntity = tableService.selectById(tableId);
		if(null == tableEntity) {
			return CommonResponse.error("数据查询失败，没有匹配的报表信息！");
		}
		return getReport(req, tableEntity, queryParam, false);
	}

	/**
	 * 报表查询
	 *
	 * @param tableId
	 * @param queryParam
	 * @return
	 */
	@RequestMapping(value = "/read/{tableId}", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<JSONObject> extractData(HttpServletRequest req,@PathVariable Long tableId,@RequestBody QueryParam queryParam) {
		Long tenantId = InvocationInfoProxy.getTenantid();

		TableEntity tableEntity = tableService.selectById(tableId);
		if(null == tableEntity) {
			return CommonResponse.error("数据查询失败，没有匹配的报表信息！");
		}
		return getReport(req, tableEntity, queryParam, true);
	}

	private String configQueryParam(TableEntity tableEntity, QueryParam queryParam,HttpServletRequest req, boolean getCommonQueryParam) {
		//拼接查询排序条件
		if(StringUtils.isNotBlank(tableEntity.getOrderItemJson())) {
			List<JSONObject> orderGroups = JSONArray.parseArray(tableEntity.getOrderItemJson(), JSONObject.class);
			JSONObject order = null;
			for(JSONObject o : orderGroups) {
				if(null != o.get("isActive") && o.get("isActive").toString().equals("true")) {
					order = o;
					break;
				}
			}
			if(null == order && CollectionUtils.isNotEmpty(orderGroups)) {
				order = orderGroups.get(0);
			}
			if(null != order && !queryParam.getOrderMap().containsKey(order.get("field").toString())) {
				queryParam.getOrderMap().put(order.get("field").toString(), order.get("sort").toString());
			}
		}
		Parameter parameter = queryParam.getParams().get("_treeSelectId");
		String leftTreeId = null;
		if(parameter != null) {
			leftTreeId = (String) parameter.getValue();
		}
		String url  = tableEntity.getParamUrl();
		try {
			if(StringUtils.isNotBlank(url) && getCommonQueryParam) {
				if(StringUtils.isNotBlank(leftTreeId)) {
					url += url.indexOf("?") == -1 ? "?":"&";
					url = url+"leftTreeId="+leftTreeId;
				}
				if(!url.contains("http")) {
					if(url.startsWith("/")){
						url = url.substring(1);
					}
					url = environmentTools.getBaseHost() + url;
				}
				Map<String, String> header = new HashMap<>();
				header.put("authority", req.getHeader("authority"));
				header.put("content-type", "application/json;charset=UTF-8");
				logger.info("请求地址-{}，header-{}", url, JSONObject.toJSONString(header));
				String respStr = HttpTookit.get(url, new HashMap<>(), header);
				CommonResponse<JSONObject> params = JSONObject.parseObject(respStr, CommonResponse.class);
				if(!params.isSuccess()) {
					logger.error("请求服务-【{}】获取报表参数失败，{}", url, params.getMsg());
					return "数据查询失败，" + params.getMsg();
				}
				QueryParam q = JSONObject.parseObject(JSONObject.toJSONString(params.getData()), QueryParam.class);
				queryParam.getParams().putAll(q.getParams());
				queryParam.getComplexParams().addAll(q.getComplexParams());
			}
		} catch (Exception e) {
			logger.error("请求服务-【{}】获取报表参数失败，", url, e);
			return "数据查询失败，获取查询参数失败！";
		}
		return null;
	}
	
	@RequestMapping(value = "/export", method = RequestMethod.POST)
	@ResponseBody
	public void exportData(@RequestBody QueryParam queryParam, HttpServletRequest request, HttpServletResponse response) throws IOException {
		Long tenantId = InvocationInfoProxy.getTenantid();
		String tableIdStr = request.getParameter("tableId");
		String tableCode = request.getParameter("tableCode");
		
		TableEntity tableEntity = null;
		if(StringUtils.isNotBlank(tableIdStr)) {
			Long tableId = Long.parseLong(tableIdStr);
			tableEntity = tableService.selectById(tableId);
		} else if(StringUtils.isNotBlank(tableCode)) {
			tableEntity = tableService.getByCode(tableCode);
		} else {
			throw new BusinessException("报表导出失败，导出参数[tableId, tableCode]不能都为空！");
		}
		//查询表头
		List<GridHeader> gridHeaders = columnService.queryGridHeadList(tableEntity.getId(), tenantId);
		List<String> titleList = new ArrayList<String>();
		List<String> properties = new ArrayList<String>();
		List<String> heightFields = new ArrayList<>();
		List<String> fieldFormat = new ArrayList<>();
		Map<String, String> filedTypeMap = new HashMap<>();
		Map<String, String> headerMap = new HashMap<>();
		for(int i=0;i<gridHeaders.size();i++) {
			GridHeader gridHeader = gridHeaders.get(i);
			if(gridHeader.getVisible() && StringUtils.isNotEmpty(gridHeader.getCode())) {
				titleList.add(gridHeader.getName());
				properties.add(gridHeader.getCode());
				fieldFormat.add(gridHeader.getExportFormat());
				if(StringUtils.isNotBlank(gridHeader.getCode())) {
					heightFields.add(gridHeader.getCode());
				}
			}
			filedTypeMap.put(gridHeader.getCode(), gridHeader.getType());
			headerMap.put(gridHeader.getCode()+"AlignType", gridHeader.getAlignType());
		}
		ExcelExportUtil excelExport = new ExcelExportUtil();
		String msg = configQueryParam(tableEntity, queryParam, request, true);
		if(StringUtils.isNotBlank(msg)) {
			throw new BusinessException(msg);
		}
		queryParam.setPageSize(10000);
		queryParam.setPageIndex(1);
		queryParam.getParams().put("creator_space", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		IPage<JSONObject> pageData = columnService.queryPageList(tableEntity.getIndexName(), queryParam, heightFields, filedTypeMap, false);
		excelExport.setData(pageData.getRecords());
		excelExport.setHeaderMap(headerMap);
		if(properties.size() > 0) {
			excelExport.setHeardKey(properties.toArray(new String[properties.size()]));
			excelExport.setFontSize(12);
			excelExport.setFieldFormat(fieldFormat);
			excelExport.setSheetName(tableEntity.getTableName());
			excelExport.setTitle(tableEntity.getTableName());
			excelExport.setHeardList(titleList.toArray(new String[titleList.size()]));
			excelExport.exportExport(request, response);
		}
	}
	
	/**
	 * 查询当前空间的所有数据
	 * 
	 * @param indexName
	 * @param creatorSpace
	 * @return
	 */
	private Long queryIndexSize(String indexName, Long creatorSpace,QueryParam queryParam) {
		Long resultSize = 0L;
		SearchRequest searchRequest = new SearchRequest(indexName);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
		boolQuery.must(QueryBuilders.termQuery("creator_space", creatorSpace));
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			if(param.getValue() == null || StringUtils.isBlank(param.getValue() + "")) {
				continue;
			}
			if(QueryParam.EQ.equals(param.getType())) {
				boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString()));
			} else if(QueryParam.LIKE.equals(param.getType())) {
				boolQuery.must(QueryBuilders.matchQuery(entry.getKey()+".keyword", param.getValue().toString()));
			} else if(QueryParam.IN.equals(param.getType())) {
				boolQuery.must(QueryBuilders.termQuery(entry.getKey(), param.getValue().toString().split(",")));
			} else if(QueryParam.BETWEEN.equals(param.getType())) {
				String[] dataArr = param.getValue().toString().split(",");
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true));
			} else if(QueryParam.GT.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue().toString()));
			} else if(QueryParam.GE.equals(param.getType())){
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue().toString()));
			} else if(QueryParam.LT.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue().toString()));
			} else if(QueryParam.LE.equals(param.getType())) {
				boolQuery.must(QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue().toString()));
			}
		}
		sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        try {
        	resultSize = queryCreatorSize(searchRequest);
        } catch (IOException e) {
        	try { //重试一次
        		resultSize = queryCreatorSize(searchRequest);
			} catch (IOException e1) {
				e1.printStackTrace();
        		throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e1.getMessage());
			}
        }
        return resultSize;
	}
	
	private Long queryCreatorSize(SearchRequest searchRequest) throws IOException {
        SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHits hits = response.getHits();
        return hits.getTotalHits().value;
	}

	/**
	 * 根据Id查询报表数据（供datav项目使用）
	 *
	 * @param tableId
	 * @return
	 */
	@PostMapping(value = "/getDataById/{tableId}")
	public CommonResponse<JSONObject> getDataById(@PathVariable Long tableId,@RequestBody QueryParam queryParam, HttpServletRequest req) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		JSONObject resp = new JSONObject();
		JSONObject table = new JSONObject();

		TableEntity tableEntity = tableService.selectById(tableId);
		if(null == tableEntity) {
			return CommonResponse.error("数据查询失败，没有匹配的报表信息！");
		}
		JSONObject reportData = getReport(req, tableEntity, queryParam, true).getData();

		//数据格式处理
		IPage<JSONObject> pageData = (IPage<JSONObject>) reportData.get("data");
		table.put("count", pageData.getTotal());
		table.put("dbType", "0");
		table.put("isList", "1");
		table.put("isPage", "1");
		table.put("linkList", null);
		table.put("expData", new JSONObject());
		table.put("list", pageData.getRecords());
		table.put("total", pageData.getPages());

		resp.put(tableEntity.getCode(), table);

		return CommonResponse.success(resp);
	}

}
