package com.ejianc.foundation.report.controller;

import java.io.Serializable;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.controller.param.GridHeader;
import com.ejianc.foundation.report.es.EsSqlQueryTemplate;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.framework.core.response.CommonResponse;

@RestController
@RequestMapping("ai")
public class AiSqlController implements Serializable {
	private static final Logger logger = LoggerFactory.getLogger(AiSqlController.class);
	private static final long serialVersionUID = 169770636614492806L;
    private SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
    private SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private DecimalFormat decimalFormat = new DecimalFormat("###,##0.00");

	@Autowired
	private EsSqlQueryTemplate sqlQueryTemplate;
	@Autowired
	private ITableService tableService;
	@Autowired
	private IColumnService columnService;
	
	@PostMapping( "getDataBySql")
	@ResponseBody
	public CommonResponse<JSONArray> getDataBySql(@RequestBody Map<String,String> params,HttpServletRequest req) {
		String sql = params.get("sql");
		String index = params.get("tableId");
		Long tenantId = Long.parseLong(params.get("tenantId"));
		logger.info("执行ES查询sql:{}",sql);
		JSONArray res = sqlQueryTemplate.execute(sql);
		JSONArray result = new JSONArray();
		if(!res.isEmpty()){
			TableEntity table = tableService.selectById(index);
			if(table==null){
				logger.error("根据索引{}未找到对应的表信息",index);
				return CommonResponse.success(res);
			}
			List<GridHeader> gridHeaderList = columnService.queryGridHeadList(table.getId(), tenantId);
			if(gridHeaderList.isEmpty()){
				logger.error("根据表id{}未找到对应的列信息",table.getId());
				return CommonResponse.success(res);
			}
			for (int i = 0; i < res.size(); i++) {
				JSONObject obj = res.getJSONObject(i);
				JSONObject copy = new JSONObject();
				for(String key:obj.keySet()){
					boolean isExist = false;
					for (GridHeader gridHeader : gridHeaderList) {
						if (gridHeader.getCode().equals(key)) {
							String value = formatField(obj.getString(key), gridHeader.getExportFormat());
							copy.put(gridHeader.getName(), value);
							isExist = true;
							break;
						}
					}
					if(!isExist){
						copy.put(key, obj.getString(key));
					}
				}
				result.add(copy);
			}
		}
		return CommonResponse.success(result);
	}
	
	private String formatField(String value, String format) {
        String result = null;

        if(StringUtils.isBlank(value) || StringUtils.isBlank(format)) {
            return value;
        }
        try {
            switch (format) {
                case "twoDecimalNumbers":
                    result = new BigDecimal(value).setScale(2, BigDecimal.ROUND_HALF_UP).toString();
                    break;
                case "integer":
                    result = new BigDecimal(value).setScale(0, BigDecimal.ROUND_HALF_UP).toString();
                    break;
                case "mnyFormat":
                    result = decimalFormat.format( new BigDecimal(value).setScale(6, BigDecimal.ROUND_HALF_UP));
                    break;
                case "yyyy-MM-dd":
                    result = dateFormat.format(dateFormat.parse(value));
                    break;
                case "yyyy-MM-dd HH:mm:ss":
                    result = dateTimeFormat.format(dateTimeFormat.parse(value));
                    break;
                default:
            }
        } catch (Exception e) {
            logger.error("依据模式[{}],格式化格式化[{}]异常：", format, value, e);
            return value;
        }
        return result;
    }

	/**
	 * 根据表ID获取提示模板ID。
	 * <p>
	 * 通过表ID查询数据库中的表信息，并返回该表对应的提示模板ID。如果表不存在，则返回错误信息。
	 * 此接口用于前端获取特定表的提示模板ID，以便进行进一步的操作或展示。
	 *
	 * @param tableId 表的唯一标识ID，用于查询表信息。
	 * @return 返回包含提示模板ID的响应对象。如果表不存在，则响应对象中包含错误信息。
	 */
	@GetMapping("getPromptTmpIdByTableId")
	@ResponseBody
	public CommonResponse<String> getPromptTmpIdByTableId(Long tableId) {
	    // 根据表ID查询表信息
	    TableEntity table = tableService.selectById(tableId);
	    // 检查查询结果，如果表不存在，则记录错误日志并返回错误响应
	    if (table == null) {
	        logger.error("根据TableId---{}未找到对应的表信息", tableId);
	        return CommonResponse.error("未找到相关报表");
	    }
	    // 表存在时，返回成功响应，并包含表的提示模板ID
	    return CommonResponse.success("查询成功！",table.getPromptTmpId());
	}
}
